# --*-- coding:utf-8 --*--
# Copyright (c) Huawei Technologies Co., Ltd. 2023-2023. All rights reserved.
import logging
import re

FSTAB_COL_MIN_VALUE = 6
MOUNT_COL_MIN_VALUE = 6
DF_COL_MIN_VALUE = 7


def parse_discard_fs_from_fstab(fstab_lines):
    discard_fs_set = set()
    for line in fstab_lines:
        if not line.startswith("/dev"):
            continue
        split_items = re.split("\\s+", line)
        if len(split_items) < FSTAB_COL_MIN_VALUE:
            logging.warning("fstab line cols [%d] can't less than [%d]", len(split_items), FSTAB_COL_MIN_VALUE)
            continue
        if "discard" not in split_items[-3]:
            continue
        discard_fs_set.add(split_items[0])
    return discard_fs_set


def parse_discard_fs_from_mount(mount_line):
    discard_fs_set = set()
    for line in mount_line:
        if not line.startswith("/dev"):
            continue
        split_items = re.split("\\s+", line)
        if len(split_items) < MOUNT_COL_MIN_VALUE:
            logging.warning("mount line cols [%d] can't less than [%d]", len(split_items), MOUNT_COL_MIN_VALUE)
            continue
        if "discard" not in split_items[-1]:
            continue
        discard_fs_set.add(split_items[0])
    return discard_fs_set


def parse_discard_fs_from_rc_local(rc_local_lines):
    discard_fs_set = set()
    for line in rc_local_lines:
        if not line.startswith("mount"):
            continue
        if "/dev" not in line:
            continue
        if "discard" not in line.split("/dev")[0]:
            continue
        discard_fs_set.add([item for item in re.split("\\s+", line) if item.startswith("/dev")][0])
    return discard_fs_set


def parse_filesystem_from_df_command(file_system_info):
    filesystems = []
    for line in file_system_info[1:]:
        if len(line.split()) >= DF_COL_MIN_VALUE:
            filesystems.append(parse_filesystem_from_df_command_line(line))
        else:
            logging.warning("mount line cols [%d] can't less than [%d]", len(line.split()), DF_COL_MIN_VALUE)
    return filesystems


def parse_filesystem_from_df_command_line(line):
    fs_mount_point = parse_filesystem_point_from_df_command_line(line)
    split_line = line.split()
    filesystem = {
        "filesystem": split_line[0],
        "fs_type": split_line[1],
        "fs_total_size": split_line[2],
        "fs_used_size": split_line[3],
        "fs_mount_point": fs_mount_point
    }
    return filesystem


def parse_filesystem_point_from_df_command_line(line):
    sing_index = line.find("%")
    if sing_index == -1 or sing_index >= len(line) - 1:
        logging.warning("df command return is abnormal")
        return line.split()[6]
    else:
        return line[sing_index + 1:].lstrip()


def parse_fs_filesystem_from_fstab(fstab_filesystem_info):
    filesystems = []
    for line in fstab_filesystem_info[1:]:
        if "#" in line:
            continue
        split_items = re.split("\\s+", line)
        if len(split_items) < FSTAB_COL_MIN_VALUE:
            logging.warning("fstab line cols [%d] can't less than [%d]", len(split_items), FSTAB_COL_MIN_VALUE)
            continue
        filesystems.append(parse_fs_filesystem_from_fstab_line(line))
    return filesystems


def parse_fs_filesystem_from_fstab_line(line):
    split = line.split()
    file_system = split[0]
    fs_type = split[-4]

    mount_point_end_index = line.find(" " + fs_type + " ")
    mount_point = line[len(file_system):mount_point_end_index].strip()

    filesystem = {
        "fstab_filesystem": file_system,
        "fstab_mount_point": mount_point,
        "fstab_fs_type": fs_type}
    return filesystem
