# --*-- coding:utf-8 --*--
import json
import traceback

from common.migration_constants import HOST_TYPE_LINUX
from common.contentParse import get_return_from_txt
from hosts.common.migra_summary_util import get_udev_result, get_multi_type_version
from hosts.common.migra_summary_util import get_host_name
from hosts.common.migra_summary_util import get_cluster_group
from hosts.common.migra_summary_util import get_empty_overview_result
from hosts.linux.migration.linux_os import get_os_type_version
from hosts.common.migration_parse_utils import get_host_ip


def execute(context):
    result = get_result(context)
    ret_map = context.get("map")
    ret_map.put("data", json.dumps(result))
    return context


def get_result(context):
    """
    获取数据
    :param context: 上下文
    :return: 数据
    """
    logger = context.get("Logger")
    try:
        result = {}
        multi_type, multi_version, multi_status = get_multi_type_version_linux(context)
        cluster_name, cluster_version = get_cluster_name_version(context)
        db_name, db_version = get_db_name_version(context)
        result["host_ip"] = context.get("ip")
        result["host_name"] = get_host_name(context)
        result["os_version"] = get_os_info(context)
        result["hba_type"] = get_hba_type(context)
        result["hba_wwn_list"] = get_hba_wwn_list(context)
        result["multi_type"] = multi_type
        result["multi_version"] = multi_version
        result["cluster_name"] = cluster_name
        # 当前只有OracleRac集群，版本就是oracle版本
        if cluster_name:
            result["cluster_version"] = db_version
        else:
            result["cluster_version"] = ""
        result["cluster_ips"] = get_cluster_group(context)
        result["lvm_type"] = get_lvm_type(context)
        result["db_name"] = db_name
        result["db_version"] = db_version
        result["disk_use"] = get_disk_use(context)
        result["real_ip"] = get_host_ip(context)
        result["multi_path_status"] = multi_status
        result["hba_detail"] = get_hba_detail(context)
        return result
    except Exception:
        logger.error("parse error: %s" % str(traceback.format_exc()))
        return get_empty_overview_result(context)


def get_hba_detail(context):
    """
    获取hba_detail
    :param context: 上下文
    :return: 获取hba_detail
    """
    context["command"] = "ls /sys/class/fc_host"
    hba_host_ids = get_return_from_txt(context)
    host_ids = get_host_ids(hba_host_ids)
    hba_detail = [get_single_detail(context, host_id) for host_id in host_ids]
    return "\n".join(hba_detail).strip()


def get_host_ids(hba_host_ids):
    host_ids = []
    for host_id_line in hba_host_ids:
        if host_id_line:
            host_ids.extend(host_id_line.split())
    return host_ids


def get_single_detail(context, host_id):
    single_detail = []
    context["command"] = "cat /sys/class/fc_host/%s/port_name" % host_id
    wwn = get_return_from_txt(context)
    context["command"] = "cat /sys/class/fc_host/%s/port_type" % host_id
    port_type = get_return_from_txt(context)
    context["command"] = "cat /sys/class/scsi_host/%s/model_desc" % host_id
    desc_info = get_desc_info(get_return_from_txt(context))
    single_detail.append("ID=" + (wwn[0] if wwn else ""))
    single_detail.append("Vendor=" + (desc_info[1] if len(desc_info) > 1 else ""))
    single_detail.append("Type=" + (port_type[0] if port_type else ""))
    single_detail.append("Speed=" + get_speed(desc_info))
    single_detail.append("Location=")
    return ";".join(single_detail).strip()


def get_desc_info(model_desc):
    desc_info = []
    for desc_info_line in model_desc:
        if desc_info_line:
            desc_info.extend(desc_info_line.split())
    return desc_info


def get_speed(desc_info):
    speed = [item for item in desc_info if item.endswith("Gb")]
    return speed[0] if speed else ""


def get_disk_use(context):
    """
    获取 磁盘绑定形式
    :param context: 上下文
    :return:磁盘绑定形式
    """
    asm_flag = get_asm_result(context)
    if asm_flag:
        return "ASMlib"

    udev_flag = get_udev_result(context)
    if udev_flag:
        return "udev"
    return ""


def get_db_name_version(context):
    """
    获取db的name、version
    :param context: 上下文
    :return: db的name、version
    """
    context["command"] = "sqlplus / as sysdba"
    version_info = get_return_from_txt(context)
    for line in version_info:
        if "Release" in line:
            version = line.split("Release")[1].split()[0]
            return "oracle", version
    return "", ""


def get_lvm_type(context):
    """
    获取 lvm 的类型
    :param context: 上下文
    :return: lvm 的类型
    """
    lvm_type = []
    context["command"] = "lvm version"
    lvm_info = get_return_from_txt(context)
    for line in lvm_info:
        if "LVM version" in line:
            lvm_type.append("LVM")

    context["command"] = "hastatus -sum"
    lvm_info = get_return_from_txt(context)
    for line in lvm_info:
        if "-- SYSTEM STATE" in line:
            lvm_type.append("VxVM")
    if not lvm_type:
        return ""
    return ", ".join(lvm_type)


def get_os_info(context):
    """
    获取类型+版本
    :param context:上下文
    :return: 类型+版本
    """
    os_type, os_version = get_os_type_version(context)
    return os_type + " " + os_version


def get_cluster_name_version(context):
    """
    获取集群的名字、版本
    :param context: 上下文
    :return: 名字、版本
    """
    context["command"] = "select value from v$parameter where name='cluster_database';"
    cluster_info = get_return_from_txt(context)
    flag = False

    for line in cluster_info:
        if "----------" in line:
            flag = True
            continue
        if flag:
            if "TRUE" in line:
                return "Oracle RAC", ""
    return "", ""


def get_hba_wwn_list(context):
    """
    获取hba_wwn
    :param context: 上下文
    :return: hba_wwn
    """
    context["command"] = "cat  /sys/class/fc_host/host*/port*name"
    hba_list = get_return_from_txt(context)
    for line in hba_list:
        if "No such file or directory" in line:
            return ""
    return "\n".join(hba_list).strip()


def get_hba_type(context):
    """
    获取hba 型号
    :param context: 上下文
    :return: hba_wwn
    """
    context["command"] = "cat  /sys/class/scsi_host/host*/model*name"
    hba_type_info = get_return_from_txt(context)
    hba_type = []
    for line in hba_type_info:
        line = line.strip()
        if "No such file or directory" in line:
            break
        if line:
            hba_type.append(line)
    hba_type.extend(get_huawei_hba(context))
    return "\n".join(list(set(hba_type)))


def get_huawei_hba(context):
    context["command"] = "lspci -vvnn | grep -iA2 'Fibre Channel'"
    hba_info = get_return_from_txt(context)
    hba_type = []
    for line in hba_info:
        if '[19e5:d301]' in line.lower():
            hba_type.append('SP520')
    return hba_type


def get_multi_type_version_linux(context):
    """
    获取多路径的类型、版本
    :param context: 上下文
    :return: 类型、版本
    """
    multi_type, multi_version = get_multi_type_version(context, HOST_TYPE_LINUX)
    if multi_type:
        return multi_type, multi_version, ""
    multi_status = get_multi_path_status(context)
    context["command"] = "multipath -ll|cat"
    dm_multi = get_return_from_txt(context)
    for line in dm_multi:
        if "dm-" in line:
            return "DM-Multipath", "NA", multi_status
    return "", "", ""


def get_multi_path_status(context):
    context["command"] = "service multipathd status | cat"
    multi_status = get_return_from_txt(context)
    return get_multi_status(multi_status)


def get_multi_status(multi_status_lines):
    for line in multi_status_lines:
        items = line.strip().split()
        if "multipathd" in line:
            if "is running" in line or u"运行" in line:
                return "running"
            if "is stopped" in line or u"停" in line:
                return "stopped"
        if "multipathd is" in line:
            return items[2] if len(items) > 2 else "NA"
        if "Checking for multipathd:" in line:
            return items[3] if len(items) > 3 else "NA"
        if "Active:" in line:
            return items[1] if len(items) > 1 else "NA"
    return "NA"


def get_asm_result(context):
    """
    获取 oracleasm 数据结果
    :param context: 上下文
    :return: oracleasm 数据结果
    """
    flag = False
    command = "oracleasm listdisks |xargs oracleasm querydisk -p"
    context["command"] = command

    asm_info = get_return_from_txt(context)
    for line in asm_info:
        if line.strip():
            flag = True
        if "No such file or directory" in line:
            flag = False
            break
    return flag
