# -*- coding: UTF-8 -*-
#  Copyright (c) Huawei Technologies Co., Ltd. 2019-2021. All rights reserved.

from com.huawei.ism.tool.deploy.entity import DeployHbaInfo
from common.cliUtil import cmd_execute
from common.base_deploy_collect import BaseDeployCollect
import re


def execute(context):
    """
    开局兼容性信息收集入口
    :param context: 传入上下文
    """
    linux_host_info = HostInfo(context)
    linux_host_info.get_info()


class HostInfo(BaseDeployCollect):

    def set_server_model(self):
        """
        设置服务器型号
        :return:
        """
        server_model_list = list()
        cmd = "uname -i|cat"
        _, cli_ret_dmi = cmd_execute(
            self.cli_ret_list, self.cli, cmd, self.logger
        )
        server_model_list.append(self.get_cli_ret(cli_ret_dmi, cmd))

        cmd = "prtdiag -v | grep 'System Configuration'|cat"
        _, cli_ret = cmd_execute(
            self.cli_ret_list, self.cli, cmd, self.logger
        )
        server_model_list.append(self.get_cli_ret(cli_ret, cmd))
        self.host_info.setServerModel("\n".join(server_model_list))

    def set_operating_system(self):
        """
        设置操作系统信息
        :return:
        """
        cmd = "cat /etc/release|cat"
        flag, cli_ret = cmd_execute(
            self.cli_ret_list, self.cli, cmd, self.logger
        )
        # 和兼容性对齐，取第二行数据
        if flag and len(cli_ret.splitlines()) > 1:
            self.host_info.setOperatingSystem(cli_ret.splitlines()[1].strip())
            return
        self.host_info.setOperatingSystem(cli_ret)

    def set_kernel(self):
        """
        设置内核信息
        Solaris内核命令及回显Solaris 10和Solaris 11 命令不一样
        solaris10:uname -ra
        solaris11:pkg info entire
        先执行命令：
        pkg info entire | grep Version，
        如果关键字 Version在回显里面，返回对应回显，如果无，
        则使用uname -ra 命令获取回显。
        :return:
        """
        cmd = "pkg info entire | grep Version|cat"
        flag, cli_ret = cmd_execute(
            self.cli_ret_list, self.cli, cmd, self.logger
        )
        if flag and 'Version' in self.get_last_line(cli_ret):
            self.host_info.setKernel(self.get_last_line(cli_ret))
            return
        cmd = "uname -ra|cat"
        flag, cli_ret = cmd_execute(
            self.cli_ret_list, self.cli, cmd, self.logger
        )
        self.host_info.setKernel(self.get_last_line(cli_ret))


    def set_bus_adapter(self):
        """
        设置HBA卡信息
        :return:
        """
        hba_info_list = list()
        cmd = "fcinfo hba-port|cat"
        flag, cli_ret = cmd_execute(
            self.cli_ret_list, self.cli, cmd, self.logger
        )
        if flag:
            hba_list = cli_ret.split("HBA Port WWN:")
            for hba in hba_list:
                if "Node WWN" in hba:
                    hba_info_list.append(self.build_hba_info(hba.strip()))
        self.host_info.setHbaInfoList(hba_info_list)

    def set_multi_path_info(self):
        """
        设置多路径信息
        注：检查以下3种多路径：UltraPath、VxVM (DMP) 7.4.0、STMS
        命令：
        华为多路径：upadm show version
        --有版本(关键字Software Version)回显就是UltraPath
        Veritas VxVM多路径：pkginfo -l VRTSaslapm
        ---有VRTSaslapm软件包(关键字VRTSaslapm)回显及其版本就是DMP多路径，
        需要返回版本信息VxVM (DMP) 7.4.0
        系统自带STMS多路径：mpathadm list lu
        ---如果有回显表示启用STMS系统自带多路径，为空则不是，返回字符串STMS
        :return:
        """
        multi_path_list = list()
        cmd = "upadmin show version|cat"
        flag, cli_ret = cmd_execute(
            self.cli_ret_list, self.cli, cmd, self.logger
        )
        if flag and "Software Version" in cli_ret:
            multi_path_list.append(self.ULTRA_PATH)

        cmd = "pkginfo -l VRTSaslapm|cat"
        flag, cli_ret = cmd_execute(
            self.cli_ret_list, self.cli, cmd, self.logger
        )
        if (
                flag
                and "PKGINST:VRTSaslapm" in re.sub('\s', '', cli_ret)
                and cmd not in self.get_last_line(cli_ret)
        ):
            multi_path_list.append(self.DMP_PATH)

        cmd = "mpathadm list lu|cat"
        flag, cli_ret = cmd_execute(
            self.cli_ret_list, self.cli, cmd, self.logger
        )

        if flag and 'TotalPathCount' in re.sub('\s', '', cli_ret):
            multi_path_list.append("STMS")
        self.host_info.setMultipathingSoftware(";".join(multi_path_list))

    def build_hba_info(self, hba_card):
        """
        设置每个HBA的详细信息
        hostbusadapter 为主机总线适配器
        hbadriverversion 未HBA卡驱动版本
        hbafirmwareversion 为HBA卡固件版本
        :param hba_card: hba卡名称
        :return:
        """
        hba_info = DeployHbaInfo()
        hba_info.setHostBusAdapter(hba_card)
        hba_info.setHbaDriverVersion(hba_card)
        hba_info.setHbaFirmwareVersion(hba_card)

        return hba_info
