# --*-- coding:utf-8 --*--
# Copyright (c) Huawei Technologies Co., Ltd. 2019-2022. All rights reserved.
import json
import traceback

from common.contentParse import get_return_from_txt
from common.util import chang_unit2_gb


def execute(context):
    result = get_result(context)
    ret_map = context.get("map")
    ret_map.put("data", json.dumps(result))
    return context


def get_result(context):
    """
    获取vg相关数据
    :param context:上下文
    :return: vg数据
    """
    result = {}
    result["host_ip"] = context.get("ip")
    try:
        map_dic, svm_list = get_solaris_svm_list(context)
        zfs_list = get_solaris_zfs_list(context)
        file_system_list = get_solaris_file_system_list(context)
        lvm_list = get_solaris_lvm_list(map_dic, svm_list, zfs_list, file_system_list)
        lvm_list.extend(get_metadb_list(context))
        if not lvm_list:
            lvm_list = get_empty_result()
        result["lvm_list"] = lvm_list
    except Exception:
        context.get("Logger").error("parse error: %s" % str(traceback.format_exc()))
        result["lvm_list"] = get_empty_result()
    return result


def get_solaris_lvm_list(map_dic, svm_list, zfs_list, file_system_list):
    """
    合成lvm_list
    :param map_dic: -m参数的映射关系
    :param svm_list: svm类型列表
    :param zfs_list: zfs类型列表
    :param file_system_list: 文件系统列表
    :return: lvm_list
    """
    lvm_list = []
    if svm_list:
        lvm_list.extend(get_svm_list(map_dic, svm_list, file_system_list))
    if zfs_list:
        lvm_list.extend(get_zfs_list(zfs_list, file_system_list))
    return lvm_list


def get_metadb_list(context):
    """
    获取metadb命令中磁盘信息
    :param context: 上下文
    :return: metadb disk list
    """
    context["command"] = "metadb"
    metadb = get_return_from_txt(context)
    disk_list = []
    disk_names = []
    for line in metadb:
        if "/dev/" in line:
            disk_name = line[line.index("/dev/"):].strip()
            if disk_name in disk_names:
                continue
            disk = {
                "lvm_type": "METADB",
                "disk_name": disk_name
            }
            disk_list.append(disk)
            disk_names.append(disk_name)
    return disk_list


def get_zfs_list(zfs_list, file_system_list):
    """
    组装zfs
    :param zfs_list: zfs类型列表
    :param file_system_list: 文件系统列表
    :return: lvm列表
    """
    lvm_list = []
    for zfs in zfs_list:
        lvm_type = "ZFS"
        pool_name = zfs.get("pool_name", "")
        volume_name = ""
        disk_names = zfs.get("disk_names", [])
        filesystem = ""
        total_size = "0"
        used_size = "0"
        fs_mount_point = ""
        for file in file_system_list:
            if pool_name == file["filesystem"].strip():
                total_size = file.get("fs_total_size", "0")
                used_size = file.get("fs_used_size", "0")
                fs_mount_point = file.get("fs_mount_point", "")
                filesystem = file.get("filesystem", "")
                break
        for disk_name in disk_names:
            one_disk = {}
            one_disk["lvm_type"] = lvm_type
            one_disk["pool_name"] = pool_name
            one_disk["volume_name"] = volume_name
            one_disk["disk_name"] = disk_name
            one_disk["filesystem"] = filesystem
            one_disk["total_size"] = total_size
            one_disk["used_size"] = used_size
            one_disk["fs_mount_point"] = fs_mount_point
            lvm_list.append(one_disk)
    return lvm_list


def get_svm_list(map_dic, svm_list, file_system_list):
    """
    组装svm
    :param map_dic: -m参数的映射关系
    :param svm_list: svm类型列表
    :param file_system_list: 文件系统列表
    :return: lvm列表
    """
    lvm_list = []
    for svm in svm_list:
        lvm_type = "SVM"
        pool_name = ""
        volume_name = svm.get("volume_name", "")
        disk_names = svm.get("disk_names", [])
        filesystem = ""
        total_size = "0"
        used_size = "0"
        fs_mount_point = ""
        file_name = \
            map_dic.get(volume_name.split("/")[-1], volume_name.split("/")[-1])
        for file_system in file_system_list:
            if file_name.split("/")[-1] == \
                    file_system["filesystem"].split("/")[-1].strip():
                filesystem = file_system.get("filesystem", "")
                total_size = file_system.get("fs_total_size", "0")
                used_size = file_system.get("fs_used_size", "0")
                fs_mount_point = file_system.get("fs_mount_point", "")
                break
        for disk_name in disk_names:
            one_disk = {}
            one_disk["lvm_type"] = lvm_type
            one_disk["pool_name"] = pool_name
            one_disk["volume_name"] = volume_name
            one_disk["disk_name"] = disk_name
            one_disk["filesystem"] = filesystem
            one_disk["total_size"] = total_size
            one_disk["used_size"] = used_size
            one_disk["fs_mount_point"] = fs_mount_point
            lvm_list.append(one_disk)
    return lvm_list


def get_solaris_file_system_list(context):
    """
    获取solaris文件系统信息
    :param context: 上下文
    :return: solaris文件系统信息
    """
    context["command"] = "df -k"
    file_info = get_return_from_txt(context)
    file_list = []
    all_info = get_file_system_all_info(file_info)
    line_num = len(all_info) / 6
    for i in range(0, line_num):
        one_file = {}
        one_file["filesystem"] = all_info[6 * i]
        total_size = all_info[6 * i + 1].strip() + " Kib"
        free_size = all_info[6 * i + 2].strip() + " Kib"
        one_file["fs_total_size"] = str(round(chang_unit2_gb(total_size), 2))
        one_file["fs_used_size"] = str(round(chang_unit2_gb(free_size), 2))
        one_file["fs_mount_point"] = all_info[6 * i + 5]
        file_list.append(one_file)
    return file_list


def get_file_system_all_info(file_info):
    """
    提取回显信息
    :param file_info: 回显行信息
    :return: all items
    """
    all_info = []
    start = False
    for line in file_info:
        if not start:
            lower_line = line.lower().strip()
            start = lower_line.startswith("filesystem") and "used" in lower_line and "mounted on" in lower_line
            continue
        all_info.extend(line.split())
    return all_info


def get_solaris_svm_list(context):
    """
    获取solaris主机的svm_list
    :param context: 上下文
    :return: svm类型列表
    """
    map_dic, lower_svm_list = get_solaris_svm_list_lower(context)
    if lower_svm_list:
        return map_dic, lower_svm_list
    return get_solaris_svm_list_higher(context)


def get_set_names(context):
    """
    获取setNames
    :param context: 上下文
    :return: setNames
    """
    context["command"] = "metaset | grep 'Set name'"
    set_name_info = get_return_from_txt(context)
    set_names = []
    for line in set_name_info:
        if line.startswith("Set name"):
            set_names.append(line.split()[3][:-1])
    return set_names


def get_solaris_svm_list_higher(context):
    """
    获取高版本solaris主机的svm_list
    :param context: 上下文
    :return: svm类型列表
    """
    set_names = get_set_names(context)
    map_dic = {}
    svm_list = []
    for set_name in set_names:
        context["command"] = "metastat -p -s " + set_name
        set_name_info = get_return_from_txt(context)
        one_svm = {}
        disk_names = []
        more_flag = False
        for line in set_name_info:
            if not line.strip():
                continue
            item_arr = line.split()
            if len(item_arr) > 4 and item_arr[1] == "-m":
                for item_tmp in item_arr[2:-1]:
                    map_dic[item_tmp.split("/")[-1]] = \
                        item_arr[0].split("/")[-1]
            if line.split()[0].split("/")[-1].startswith("d") and \
                    not line.strip().endswith("\\") and len(line.split()) == 4:
                one_svm = {}
                disk_names = []
                one_svm["volume_name"] = line.strip().split()[0].strip()
                disk_names.append(line.strip().split()[-1].strip())
                one_svm["disk_names"] = disk_names
                svm_list.append(one_svm)
                continue
            if line.split()[0].split("/")[-1].startswith("d") and \
                    line.strip().endswith("\\"):
                more_flag = True
                one_svm = {}
                disk_names = []
                one_svm["volume_name"] = line.strip().split()[0].strip()
                num = int(line.split()[2])
                disk_names.extend(line.split()[3:3 + num])
                continue
            if more_flag:
                end_line = line.replace("\\", "").strip()
                num = int(line.split()[0])
                disk_names.extend(end_line.split()[1:1 + num])
                if not line.strip().endswith("\\"):
                    one_svm["disk_names"] = disk_names
                    svm_list.append(one_svm)
    return map_dic, svm_list


def get_map_dic(m_info_list):
    """
    解析-m参数的映射
    :param m_info_list:-m回显
    :return: -m参数的映射
    """
    map_dic = {}
    for line in m_info_list:
        item_arr = line.split()
        if len(item_arr) > 4 and item_arr[1] == "-m":
            for item_tmp in item_arr[2:-1]:
                map_dic[item_tmp] = item_arr[0]
    return map_dic


def get_solaris_svm_list_lower(context):
    """
    获取低版本solaris主机的svm_list
    :param context: 上下文
    :return: svm类型列表
    """
    m_info_list, svm_info_list = get_svm_info_list_lower(context)
    map_dic = get_map_dic(m_info_list)
    svm_list = []
    for svm_info in svm_info_list:
        one_svm = {}
        disk_names = []
        more_flag = False
        for line in svm_info:
            if line.strip().startswith("d") and \
                    not line.strip().endswith("\\") and len(line.split()) == 4:
                one_svm = {}
                disk_names = []
                one_svm["volume_name"] = line.strip().split()[0].strip()
                disk_names.append(line.strip().split()[-1].strip())
                one_svm["disk_names"] = disk_names
                svm_list.append(one_svm)
                continue
            if line.strip().startswith("d") and line.strip().endswith("\\"):
                more_flag = True
                one_svm = {}
                disk_names = []
                one_svm["volume_name"] = line.strip().split()[0].strip()
                for disk_item in line.strip().split():
                    if disk_item.strip().startswith("c"):
                        disk_names.append(disk_item.strip())
                continue
            if more_flag:
                end_line = line.replace("\\", "").strip()
                for disk_item in end_line.strip().split():
                    if disk_item.strip().startswith("c"):
                        disk_names.append(disk_item.strip())
                if not line.strip().endswith("\\"):
                    one_svm["disk_names"] = disk_names
                    svm_list.append(one_svm)
    return map_dic, svm_list


def get_svm_info_list_lower(context):
    """
    获取低版本的svm_info_list
    :param context: 上下文
    :return: svm回文列表
    """
    context["command"] = "metastat -p"
    svm_info = get_return_from_txt(context)
    svm_info_list = []
    m_info_list = []
    one_svm_info = []
    flag = False
    for index, line in enumerate(svm_info):
        if line.strip().startswith("d") and one_svm_info:
            svm_info_list.append(one_svm_info)
            one_svm_info = []
            flag = False
        if line.strip().startswith("d") and len(line.strip().split()) > 2 \
                and "-m" not in line.strip().split()[1]:
            flag = True
        if line.strip().startswith("d") and len(line.strip().split()) > 2 \
                and "-m" in line.strip().split()[1]:
            m_info_list.append(line)
        if flag:
            one_svm_info.append(line)
        if flag and index == len(svm_info) - 1:
            svm_info_list.append(one_svm_info)
    return m_info_list, svm_info_list


def get_zfs_info_list(context):
    """
    获取zfs_info_list
    :param context: 上下文
    :return: zfs回文列表
    """
    context["command"] = "zpool status -v"
    zfs_info = get_return_from_txt(context)
    zfs_info_list = []
    one_zfs_info = []
    flag = False
    for index, line in enumerate(zfs_info):
        if line.strip().startswith("pool:"):
            flag = True
        if line.strip().startswith("pool:") and one_zfs_info:
            zfs_info_list.append(one_zfs_info)
            one_zfs_info = []
        if flag:
            one_zfs_info.append(line)
        if flag and index == len(zfs_info) - 1:
            zfs_info_list.append(one_zfs_info)
    return zfs_info_list


def get_solaris_zfs_list(context):
    """
    获取solaris主机的zfs_list
    :param context: 上下文
    :return: zfs类型列表
    """
    zfs_info_list = get_zfs_info_list(context)
    zfs_list = []
    for zfs_info in zfs_info_list:
        one_zfs = {}
        disk_names = []
        more_flag = False
        for line in zfs_info:
            if line.strip().startswith("pool:"):
                one_zfs["pool_name"] = line.strip().split(":")[1].strip()
                continue
            if line.strip().startswith("NAME") \
                    and line.strip().endswith("CKSUM"):
                more_flag = True
                continue
            if more_flag and not line.strip():
                break
            if more_flag:
                disk_names.append(line.strip().split()[0])
        one_zfs["disk_names"] = disk_names
        zfs_list.append(one_zfs)
    return zfs_list


def get_empty_result():
    """
    空数据格式
    :return: 空数据
    """
    lvm_list = []
    one_disk = {}
    one_disk["lvm_type"] = ""
    one_disk["pool_name"] = ""
    one_disk["volume_name"] = ""
    one_disk["disk_name"] = ""
    one_disk["filesystem"] = ""
    one_disk["total_size"] = ""
    one_disk["used_size"] = ""
    one_disk["fs_mount_point"] = ""
    lvm_list.append(one_disk)
    return lvm_list
