#  coding=UTF-8
#  Copyright (c) Huawei Technologies Co., Ltd. 2022-2023. All rights reserved.
import json
import traceback
from common.contentParse import get_return_from_txt


def execute(context):
    try:
        result = get_oracle_rac_info(context)
    except Exception:
        context.get("Logger").error("parse Solaris OracleRAC info error: %s" % str(traceback.format_exc()))
        result = get_empty_oracle_rac_info(context.get("ip"))
    ret_map = context.get("map")
    ret_map.put("data", json.dumps(result))
    return context


def get_empty_oracle_rac_info(ip):
    return {
        "host_ip": ip,
        "oracle_version": "",
        "asm_diskstring_value": "",
        "disk_list": [get_empty_oracle_disk()]
    }


def get_empty_oracle_disk():
    return {
        "asmdisk_name": "",
        "asmdisk_path": "",
        "asmdisk_state": "",
        "asmdisk_total_size": "",
        "asmdisk_free_size": "",
        "lun_wwn": "",
        "user": "",
        "group": "",
    }


def get_oracle_rac_info(context):
    """
    解析Solaris主机OracleRAC集群信息
    :param context: 上下文
    :return: 结果集合
    """
    oracle_version = get_oracle_version(context)
    asm_diskstring_value = get_asm_disk_string_value(context)
    oracle_disk_list = get_oracle_data(context)
    return {
        "host_ip": context.get("ip"),
        "oracle_version": oracle_version,
        "asm_diskstring_value": asm_diskstring_value,
        "disk_list": oracle_disk_list
    }


def get_oracle_data(context):
    asm_disk_data = get_asm_disk(context)
    asm_disk_path_detail = get_asm_disk_path_detail(context)
    asm_afd_path = get_asm_afd_dsk(context, asm_disk_path_detail)
    disk_user_group = get_disk_user_and_group(context)
    result_list = []
    for data_dict in asm_disk_data:
        name = data_dict.get("NAME", "")
        disk_path = data_dict.get("PATH", "")
        if not name or not disk_path:
            continue
        wwn = get_lun_wwn_from_path(get_asm_disk_path(disk_path, asm_afd_path, asm_disk_path_detail))
        user_group = disk_user_group.get(wwn, "").split("/")
        user = user_group[0]
        group = user_group[-1]
        result_dict = {
            "asmdisk_name": name,
            "asmdisk_path": disk_path,
            "asmdisk_state": data_dict.get("STATE", ""),
            "asmdisk_total_size": data_dict.get("TOTAL_MB", ""),
            "asmdisk_free_size": data_dict.get("FREE_MB", ""),
            "lun_wwn": wwn,
            "user": user,
            "group": group,
        }
        result_list.append(result_dict)
    if not result_list:
        result_list.append(get_empty_oracle_disk())
    return result_list


def get_asm_disk_path(path, asm_afd_path, asm_disk_path_detail):
    if "/dev/rdsk/" in path:
        return path
    if "AFD:" in path:
        return asm_afd_path.get(path[4:], "")
    return asm_disk_path_detail.get(path, "")


def get_lun_wwn_from_path(path):
    """
    示例： /dev/rdsk/c0t60001440000000103029208322B82089d0s6
    :param path: 磁盘路径
    :return: LUN WWN
    """
    if "/dev/rdsk/" in path and len(path) > 17:
        return path.split("/")[-1][3:-4]
    return ""


def get_asm_disk(context):
    command = "select group_number,disk_number,name,mount_status,header_status,path,state,total_mb,free_mb," \
              "sector_size from v$asm_disk;"
    asm_disk_lines = get_command_return(context, command)
    start = False
    headers = []
    result_dicts = []
    for line in asm_disk_lines:
        if not start and "GROUP_NUMBER" in line and "DISK_NUMBER" in line:
            headers = line.split()
            start = True
        if start and "----" not in line and "GROUP_NUMBER" not in line:
            items = line.split()
            # NAME没有值时，只展示8列数据，不解析此数据
            if len(items) != 9:
                continue
            result_dict = {}
            for index, header in enumerate(headers):
                result_dict[header] = items[index]
            result_dicts.append(result_dict)
    return result_dicts


def get_asm_afd_dsk(context, asm_disk_path_detail):
    """
    示例：
    --------------------------------------------------------------------------------
    Label                     Filtering   Path
    ================================================================================
    DATA1                       ENABLED   /sharedisk/asm3
    DATA2                       ENABLED   /sharedisk/asm2
    :param context: 上下文
    :param asm_disk_path_detail: asm disk路径链接信息
    :return: Lebel和磁盘路径对应关系
    """
    asm_afd_dsk_lines = get_command_return(context, "asmcmd afd_lsdsk")
    start = False
    label_path_dict = {}
    for line in asm_afd_dsk_lines:
        if not start:
            if "Label" in line and "Filtering" in line:
                start = True
            continue
        if "====" in line:
            continue
        items = line.strip().split()
        if len(items) < 3:
            continue
        path = items[2]
        if path in asm_disk_path_detail:
            path = asm_disk_path_detail[path]
        label_path_dict[items[0]] = path
    return label_path_dict


def get_asm_disk_path_detail(context):
    detail_dict = {}
    result_paths = get_asm_disk_string_paths(context)
    for result_path in result_paths:
        detail = get_command_return(context, "ls -l " + result_path)
        for line in detail:
            items = line.split()
            if len(items) < 3 or "/dev/rdsk/" not in line:
                continue
            detail_dict[result_path + items[-3]] = items[-1]
    return detail_dict


def get_asm_disk_string_paths(context):
    """
    示例： profile:AFD:*,/dev/rdsk/*,/sharedisk/*
    :param context: 上下文
    :return: 除 AFD:*,/dev/rdsk/* 外路径集合
    """
    asm_disk_string = get_command_return(context, "asmcmd dsget")
    for line in asm_disk_string:
        if "profile:" in line:
            return get_asm_disk_path_list(line)
    return []


def get_asm_disk_path_list(line):
    paths = line[line.index(":") + 1:].strip().split(",")
    return list(map(lambda path: path[0:-1], filter(lambda path: path != "AFD:*" and path != "/dev/rdsk/*", paths)))


def get_oracle_version(context):
    version_info = get_command_return(context, "select * from v$version;")
    for line in version_info:
        if "Release" in line:
            items = line[line.index("Release"):].split()
            if len(items) > 1:
                return items[1]
    return ""


def get_asm_disk_string_value(context):
    asm_disk_string = get_command_return(context, "show parameter asm_diskstring")
    start = False
    result_line = ""
    for line in asm_disk_string:
        if "asm_diskstring" in line:
            start = True
            result_line = line.strip()
            continue
        if start:
            result_line += line.strip()
    return " ".join(result_line.split()[2:])


def get_disk_user_and_group(context):
    """
    示例： crw-r-----   1 root     sys      243, 128 Dec 29 14:00 ssd@g600014400000001020292022ad145dce:a,raw
    :param context: 上下文
    :return:
    """
    root_sys = "root/sys"
    disk_user_group_dict = {}
    disk_detail = get_command_return(context, "ls -l /devices/scsi_vhci/")
    for line in disk_detail:
        items = line.split()
        if len(items) > 5 and "@" in items[-1]:
            user_and_group = items[2] + "/" + items[3]
            disk = parse_disk_wwn(items[-1])
            user_group = disk_user_group_dict.get(disk, "")
            # 优先取非 root/sys 的第一项
            if not user_group or user_group == root_sys:
                disk_user_group_dict[disk] = user_and_group
    return disk_user_group_dict


def parse_disk_wwn(disk):
    start_index = disk.index("@") + 2
    if ":" in disk:
        return disk[start_index:disk.index(":")].upper()
    return disk[start_index:].upper()


def get_command_return(context, command):
    context["command"] = command
    return get_return_from_txt(context)
