# --*-- coding:utf-8 --*--
# Copyright (c) Huawei Technologies Co., Ltd. 2019-2020. All rights reserved.

import json
import traceback

from common.migration_constants import HOST_TYPE_SOLARIS
from common.contentParse import get_return_from_txt
from hosts.common.migra_summary_util import get_host_name, get_multi_type_version, get_cluster_group, get_stms_disks, \
    get_dmp_version, get_empty_overview_result
from hosts.solaris.migration.solaris_os import get_type_version
from hosts.solaris.migration.solaris_lvm import get_solaris_svm_list, get_solaris_zfs_list
from hosts.common.migration_parse_utils import get_host_ip
from hosts.parse_util import get_key_values_from_content_lines


def execute(context):
    global logger
    logger = context.get("Logger")
    result = get_result(context)
    ret_map = context.get("map")
    ret_map.put("data", json.dumps(result))
    return context


def get_result(context):
    """
    获取数据
    :param context: 上下文
    :return: 数据
    """
    result = {}
    try:
        multi_type, multi_version = get_multi_type_version_solaris(context)
        cluster_name = get_cluster_name(context)
        db_name, db_version = get_db_name_version(context)
        hba_list, hba_type = get_hba_wwn_list_type(context)
        result["host_ip"] = context.get("ip")
        result["host_name"] = get_host_name(context)
        result["os_version"] = get_os_version(context)
        result["hba_type"] = hba_type
        result["hba_wwn_list"] = hba_list
        result["multi_type"] = multi_type
        result["multi_version"] = multi_version
        result["cluster_name"] = cluster_name
        result["hba_detail"] = get_hba_detail(context)
        if cluster_name:
            result["cluster_version"] = db_version
        else:
            result["cluster_version"] = ""
        result["cluster_ips"] = get_cluster_group(context)
        result["lvm_type"] = get_lvm_type(context)
        result["db_name"] = db_name
        result["db_version"] = db_version
        result["disk_use"] = ""
        result["real_ip"] = get_host_ip(context)
        result["multi_path_status"] = ""
    except Exception:
        logger.error("parse error: %s" % str(traceback.format_exc()))
        result = get_empty_overview_result(context)
    return result


def get_hba_detail(context):
    context["command"] = "fcinfo hba-port"
    hba_port = get_return_from_txt(context)
    start_bid = "HBA Port WWN:"
    keys_tuple = ('HBA Port WWN', 'Port Name', 'Manufacturer', 'Type', 'Supported Speeds', 'Current Speed')
    delimiter = ":"
    hba_detail_info = get_key_values_from_content_lines(hba_port, start_bid, keys_tuple, delimiter)
    hba_detail = []
    for info in hba_detail_info:
        hba_detail.append(build_single_hba_info(info))
    return "\n".join(hba_detail).strip()


def build_single_hba_info(info):
    speed = info.get('Current Speed', '')
    supported_speeds = info.get('Supported Speeds', '')
    if not speed and supported_speeds:
        speed = supported_speeds[-1]
    single_hba_detail = [
        "ID=" + info.get('HBA Port WWN', ''),
        "Vendor=" + info.get('Manufacturer', ''),
        "Type=" + info.get('Type', ''),
        "Speed=" + speed,
        "Location="
    ]
    return ";".join(single_hba_detail).strip()


def get_db_name_version(context):
    """
    获取db的name、version
    :param context: 上下文
    :return: db的name、version
    """
    context["command"] = "sqlplus / as sysdba"
    version_info = get_return_from_txt(context)
    for line in version_info:
        if "Release" in line:
            if len(line.split("Release")) > 1:
                version = line.split("Release")[1].split()[0]
                return "oracle", version
    return "", ""


def get_lvm_type(context):
    """
    获取 lvm 的类型
    :param context: 上下文
    :return: lvm 的类型
    """
    lvm_type = []
    map_dic, svm_list = get_solaris_svm_list(context)
    zfs_list = get_solaris_zfs_list(context)
    if svm_list:
        lvm_type.append("SVM")
    if zfs_list:
        lvm_type.append("ZFS")
    if not lvm_type:
        return ""
    return ", ".join(lvm_type)


def get_os_version(context):
    """
    获取类型+版本
    :param context:上下文
    :return: 类型+版本
    """
    os_type, os_version = get_type_version(context)
    return os_type + " " + os_version


def get_cluster_name(context):
    """
    获取集群的名字、版本
    :param context: 上下文
    :return: 名字、版本
    """
    context["command"] = "select value from v$parameter " \
                         "where name='cluster_database';"
    cluster_info = get_return_from_txt(context)
    flag = False

    for line in cluster_info:
        if "----------" in line:
            flag = True
            continue
        if flag:
            if "TRUE" in line:
                return "Oracle RAC"
    return ""


def get_hba_wwn_list_type(context):
    """
    获取hba_wwn、type
    :param context: 上下文
    :return: hba_wwn
    """
    context["command"] = "fcinfo hba-port"
    hba_info = get_return_from_txt(context)
    hba_list = []
    hba_type = ""
    for line in hba_info:
        if "HBA Port WWN" in line:
            hba_list.append(line.split(":")[-1].strip())
        if line.strip().startswith("Model") and not hba_type:
            hba_type = line.split(":")[-1].strip()
    return "\n".join(hba_list), hba_type


def get_multi_type_version_solaris(context):
    """
    获取多路径的类型、版本
    :param context: 上下文
    :return: 类型、版本
    """
    dmp_version = get_dmp_version(context)
    if dmp_version:
        return "DMP", dmp_version
    multi_type, multi_version = get_multi_type_version(context, HOST_TYPE_SOLARIS)
    if multi_type:
        return multi_type, multi_version
    stms_multi = get_stms_disks(context)
    if stms_multi:
        return "STMS", "NA"
    return "", ""
