# coding=utf-8
'''
    DocStr: collect HyperMetro feature configuration on Solaris server host
    Since: V3R1C00RC2
'''
import os
import sys
CURRENT_PATH = os.path.dirname(os.path.abspath(__file__))
sys.path.append(CURRENT_PATH)
CURRENT_PATH = os.path.join(CURRENT_PATH, "..\\..")
sys.path.append(CURRENT_PATH)

from common import util
from common import contentParse
from common import constants
OS_VERSION_CMD_MAP = {'id' : 'solaris_os_info',
                      'cmd' : 'cat /etc/release'}
FORMAT_CMD_MAP = {'id':'solaris_disk_format' ,
                  'cmd' : r'format'}
MPIOCHECK_CMD_MAP_OLD = {'id':'solaris_disk_logic_unit_info' ,
                         'cmd' : r' mpathadm show lu /dev/dsk/'}
MIN_SUPPORT_OS_VER = 9

def execute(context):
    '''
    Function describe  : collect windows host HyperMetro information
    Return             : cmd display
    '''
    hyperMetroChecker = HyperMetroChecker(context)
    return hyperMetroChecker.doGrab(context)


class HyperMetroChecker(object):
    '''
    @summary: collect HyperMetro feature configuration on solaris server host
    '''

    def __init__(self, context):
        '''
        @summary: constructor
        '''
        self.context = context
        self.sshCon = context.get("SSH")
        self.lang = context.get("lang")
        self.cmd_display = context.get("ret_map")

    def doGrab(self, context):
        '''
        @summary: evaluation entrance
        '''
        util.updateItemProgress(context, constants.PROG20)
        resultStatus = ""
        util.log.info(self.context, "[Solaris] checking os version...")
        cmdEcho = self.sshCon.execCmdHasLogTimout(OS_VERSION_CMD_MAP['cmd'], constants.HOST_CMD_TIMEOUT)
        self.cmd_display.put("cmd_display_" + OS_VERSION_CMD_MAP['id'], cmdEcho)
        status, result = self.getResultMsg(OS_VERSION_CMD_MAP['cmd'], cmdEcho)
        resultStatus += result
        util.updateItemProgress(context, constants.PROG40)
        if not status or not self.isOsVerSupported(cmdEcho):
            self.cmd_display.put("err_msg", resultStatus)
            return self.cmd_display

        util.log.info(self.context, "[Solaris] start to fetch hyperMetro configuration.")
        cmdEcho = ""
        cmdArgs = {}
        cmdArgs['timeout'] = constants.LINUX_UPSCAN_CMD_TIMEOUT
        cmdArgs['recordLog'] = True
        cmdArgs['_SPEFIC_END_STRS'] = ['(enter its number):', '$']
        cmdArgs['needCheckResult'] = False
        cmdEcho = self.sshCon.execCmdWithConditions(FORMAT_CMD_MAP['cmd'], cmdArgs)
        self.cmd_display.put("cmd_display_" + FORMAT_CMD_MAP['id'], cmdEcho)
        status, result = self.getResultMsg(FORMAT_CMD_MAP['cmd'], cmdEcho)
        resultStatus += result
        util.updateItemProgress(context, constants.PROG60)
        if not status:
            self.cmd_display.put("err_msg", resultStatus)
            util.updateItemProgress(context, constants.PROG80)
            return self.cmd_display

        diskNoList = self.parse4HuaweiDiskNo(cmdEcho)
        if len(diskNoList) < 1:
            util.log.info(self.context, "[Solaris] current host has no huawei disk detected..")
            self.cmd_display.put("err_msg", resultStatus)
            util.updateItemProgress(context, constants.PROG80)
            self.sshCon.execCmdWithTimout("0", 1)
            return self.cmd_display
        #send Ctrl+C if timeout
        util.updateItemProgress(context, constants.PROG80)
        self.sshCon.execCmdWithTimout("0", 1)

        for diskNo in diskNoList:
            cmd = MPIOCHECK_CMD_MAP_OLD["cmd"] + diskNo + "s2"
            util.log.info(self.context, "[Solaris] cmd:" + cmd)
            diskInfo = self.sshCon.execCmdWithTimout(cmd, constants.HOST_CMD_TIMEOUT)
            self.cmd_display.put("cmd_display_" + MPIOCHECK_CMD_MAP_OLD['id'] + diskNo, diskInfo)
            status, result = self.getResultMsg(cmd, diskInfo)
            resultStatus += result

        self.cmd_display.put("err_msg", resultStatus)
        return self.cmd_display

    def parse4HuaweiDiskNo(self, cmdEcho):
        '''
        @summary: parse 4 huawei's disk logic number
        @return: diskNo. list
        '''
        huaweiDisks = []
        isParseStart = False
        for  line in cmdEcho.splitlines():
            if "available disk selections:" in line.lower():
                isParseStart = True
                continue
            if isParseStart and "<" in line and ">" in line:
                if "huawei" in line.lower():
                    lineParams = line.split()
                    if len(lineParams) > 2:
                        huaweiDisks.append(lineParams[1])
            if "(enter its number):" in line:
                break
        return huaweiDisks

    def isOsVerSupported(self, cmdEcho):
        '''
        @summary: parse /etc/release file to fetch current OS version
        @note: if cur ver. is lower than 9,return unsupported
        '''
        for line in cmdEcho.lower().splitlines()[1:]:
            if 'solaris' in line:
                line = line[line.index('solaris') + len('solaris'):].strip()
                if not line:
                    #failed to parse 
                    util.log.info(self.context, "[Solaris] failed to parse os version.")
                    return False
                curVer = line.split()[0].split(".")[0]
                if not curVer.isdigit():
                    util.log.info(self.context, "[Solaris] failed to get os version Num. .")
                    return False
                if MIN_SUPPORT_OS_VER > int(curVer) :
                    util.log.info(self.context, "[Solaris] current os version %s unsupported." % str(curVer))
                    return False
                else:
                    util.log.info(self.context, "[Solaris] current os version %s." % str(curVer))
                    return True

    def getResultMsg(self, step_name, cmd_display_temp):
        '''
        @summary: get result status's message
        '''
        result_msg = ""
        result = True
        if None == cmd_display_temp or '' == cmd_display_temp or\
            'TOOLKIT_SEND_CMD_TIME_OUT' in  cmd_display_temp or\
                'TOOLKIT_EXE_CMD_FAILED' in cmd_display_temp:
            result = False

        if "en" == self.lang:
            if result:
                result_msg += step_name + ":\texecute success\r\n"
            else:
                result_msg += step_name + ":\texecute failed\r\n"
        else:
            if result:
                result_msg += step_name + u":\t执行成功\r\n"
            else:
                result_msg += step_name + u":\t执行失败\r\n"


        return result, result_msg
