#  coding=UTF-8
#  Copyright (c) Huawei Technologies Co., Ltd. 2019-2019. All rights reserved.

"""
@version: SmartKit V200R006C00
@time: 209/10/26
@file: solaris_lun_wwn_info.py
@function:
@modify:
"""

import sys
import os
import re
import traceback

current_absolute_path = os.path.dirname(os.path.abspath(__file__))
path = os.path.join(current_absolute_path, "..\\..")
sys.path.append(path)
from common.contentParse import create_result_info
from common import constants
from common.util import log, HostResultInfo, HostLunInfo, \
    create_lun_info, get_host_wwn_info, qryUltrapathVersion, \
    getUltrapathIntVer

LAUNCHER = "fcinfo hba-port or iscsiadm list initiator-node"


def get_result_info(context):
    """
    :param context:
    :return:
    """
    host_result_info = HostResultInfo()
    host_lun_info = HostLunInfo()
    cli = context.get("SSH")
    host_lun_info.set_host_id(cli.getHost())
    cli_rets = ''
    try:
        log.info(context, "Start query solaris lun wwn info.")
        # from launcher get initiator info
        _, cli_rets, initiator_wwn = get_host_wwn_info(context)
        if initiator_wwn:
            host_lun_info.set_initiator(initiator_wwn)
            host_result_info.set_hosts(host_lun_info)
        else:
            log.info(context, "Cannot find host initiator info, set pass.")
            create_result_info(context, "True", cli_rets, LAUNCHER,
                               host_result_info)
            return

        is_ultrapath_qry_succ, ultrapath_version, cli_ret = \
            qryUltrapathVersion(context)
        cli_rets += cli_ret
        log.info(context,
                 "The ultrapath version is(%s)." % ultrapath_version)

        if not is_ultrapath_qry_succ:
            log.error(context, 'Query ultrapath version failed.')
            fail_cmd = constants.QRY_ULTRAPATH_CMD
            create_result_info(context, "False", cli_rets, fail_cmd,
                               host_result_info)
            return
        # ultrapath check
        if ultrapath_version:
            is_qry_succ, fail_cmd, cli_ret, lun_wwn = \
                query_ultrapath_lun_wwn_info(context, ultrapath_version)
            cli_rets += cli_ret
            if is_qry_succ is None:
                create_result_info(context, "True", cli_rets, "",
                                   host_result_info)
                log.info(context, 'ultrapath pass, return.')
                return
            elif is_qry_succ:
                create_lun_info(host_lun_info, lun_wwn)
                host_result_info.set_hosts(host_lun_info)
                log.info(context, 'ultrapath need to check.')
            else:
                create_result_info(context, "False", cli_rets, fail_cmd,
                                   host_result_info)
                log.info(context, 'ultrapath not pass, return.')
                return
        # multipath check
        is_qry_succ, _, cli_ret, disk_wwn_dict = \
            query_multipath_lun_wwn_info(context)
        cli_rets += cli_ret
        if is_qry_succ:
            create_lun_info(host_lun_info, disk_wwn_dict)
            host_result_info.set_hosts(host_lun_info)
            create_result_info(context, "nocheck", cli_rets, "",
                               host_result_info)
        else:
            fail_cmd = constants.QRY_DISK_PTH_MANAGED_BY_MULTIPATHD_CMD
            create_result_info(context, "False", cli_rets, fail_cmd,
                               host_result_info)
    except Exception as e:
        log.info(context, "get host result info exception:%s:%s" % (
            str(e), str(traceback.format_exc())))
        create_result_info(context, "False", cli_rets, '',
                           host_result_info)


def query_ultrapath_lun_wwn_info(context, ultrapath_version):
    """
    :param context:
    :param ultrapath_version:
    :return:
    """
    tup_ver = getUltrapathIntVer(context, ultrapath_version)
    middle_tup_ver = getUltrapathIntVer(context,
                                        constants.SOLARIS_MIDDLE_ULTRAPATH_VER)
    consistency_tup_ver = getUltrapathIntVer(
        context, constants.SOLARIS_CONSISTENCY_ULTRAPATH_VER)

    if tup_ver <= middle_tup_ver:
        return query_high_version_info(context, is_high_version=False)
    elif tup_ver < consistency_tup_ver:
        return query_high_version_info(context)
    else:
        log.info(context, 'the ultrapath version greater or equal to '
                          '8.06.011, set the result pass.')
        return None, "", "", {}


def query_high_version_info(context, is_high_version=True):
    """
    :param context:
    :param is_high_version:
    :return:
    """
    cli = context.get("SSH")
    lun_wwn = {}
    cli_ret = ''
    vlu_id_cmd = constants.QRY_VLUN_CMD_HIGHT_VER_SOLARIS if is_high_version \
        else constants.QRY_VLUN_CMD_LOWER_VER_SOLARIS
    try:
        cli_ret = cli.execCmdWithNoCheckResult(vlu_id_cmd,
                                               constants.HOST_CMD_TIMEOUT)

        is_data_start = False
        for line in cli_ret.splitlines():
            if not is_data_start and re.search('Vlun ID', line,
                                               re.I) and re.search('WWN', line,
                                                                   re.I):
                is_data_start = True
                continue

            fields = line.split()
            if is_data_start and len(fields) >= 3 and fields[0].isdigit():
                lun_wwn[fields[3].strip()] = fields[0].strip()
        return True, '', cli_ret, lun_wwn
    except Exception as e:
        log.error(context, 'Query ultrapath VLUN IDs exception:' + str(e))
        return False, vlu_id_cmd, cli_ret, {}


def query_multipath_lun_wwn_info(context):
    disk_wwn_pattern = re.compile(r".*c\dt([\d|a-f]{32})d0.*",
                                  flags=re.IGNORECASE)

    cli = context.get("SSH")
    lun_wwn = {}
    cli_ret = ''
    vlu_id_cmd = constants.QRY_VLUN_CMD_MULTIPATH_SOLARIS
    try:
        cli_ret = cli.execCmdWithNoCheckResult(
            vlu_id_cmd, constants.HOST_CMD_TIMEOUT)

        for line in cli_ret.splitlines():
            match = disk_wwn_pattern.search(line)
            if not match:
                continue
            lun_wwn[match.group(1)] = ""
        return True, '', cli_ret, lun_wwn
    except Exception as e:
        log.error(context, 'Query multipath VLUN IDs exception:' + str(e))
        return False, vlu_id_cmd, cli_ret, {}
