# -*- coding: UTF-8 -*-
#  Copyright (c) Huawei Technologies Co., Ltd. 2019-2021. All rights reserved.
from com.huawei.ism.tool.deploy.entity import DeployHbaInfo
from common.cliUtil import cmd_execute
from common.base_deploy_collect import BaseDeployCollect
import re


def execute(context):
    """
    开局兼容性信息收集入口
    :param context: 传入上下文
    :return:
    """
    host_info = HostInfo(context)
    host_info.get_info()


class HostInfo(BaseDeployCollect):
    def set_server_model(self):
        """
        设置服务器型号
        :return:
        """
        # VMware ESXi仅支持x86服务器，因此不需要收集服务器信息。
        self.host_info.setServerModel("")

    def set_operating_system(self):
        """
        设置操作系统信息
        :return:
        """
        cmd = "vmware -lv|cat"
        _, cli_ret = cmd_execute(self.cli_ret_list, self.cli, cmd, self.logger)
        self.host_info.setOperatingSystem(self.get_last_line(cli_ret))

    def set_bus_adapter(self):
        """
        设置HBA卡信息
        :return:
        """
        hba_info_list = list()
        cmd = (
            "esxcli storage core adapter list |grep -i 'Fibre Channel' | "
            "awk '{print $1}'|cat"
        )
        _, cli_ret = cmd_execute(self.cli_ret_list, self.cli, cmd, self.logger)
        hba_list = cli_ret.splitlines()
        for hba in hba_list:
            if self.check_vmware_valid_echo(cmd, hba):
                continue
            hba_info_list.append(self.build_hba_info(hba))
        self.build_roce_info(hba_info_list)
        self.host_info.setHbaInfoList(hba_info_list)

    def build_roce_info(self, hba_info_list):
        """
        构建roce协议卡数据
        :param hba_info_list: hba接口卡集合
        :return:
        """
        cmd = "esxcfg-nics -l|cat"
        # 这条命令的回文需要放在首行
        _, cli_ret = cmd_execute(
            self.cli_ret_list, self.cli, cmd, self.logger
        )
        roce_list = cli_ret.splitlines()
        roce_hba_list = list()
        self.get_hba_card(roce_hba_list, "esxcfg-nics -l| awk '{print $1}'")
        for roce_hba_card in roce_hba_list:
            hba_info = DeployHbaInfo()
            cmd = "esxcli network nic get -n {}|cat".format(roce_hba_card)
            # 这条命令的回文放在第二行
            _, cli_ret = cmd_execute(self.cli_ret_list, self.cli, cmd,
                                     self.logger)
            roce_hba_info = self.get_cli_ret(cli_ret, cmd)
            for roce_card in roce_list:
                if roce_hba_card in roce_card:
                    roce_hba_info = "\n".join([roce_card, roce_hba_info])
                    break
            hba_info.setHostBusAdapter(roce_hba_info)
            hba_info.setHbaDriverVersion(roce_hba_info)
            hba_info.setHbaFirmwareVersion(roce_hba_info)
            hba_info_list.append(hba_info)

    def set_multi_path_info(self):
        """
        设置多路径信息
        注：检查以下2种多路径：UltraPath、VMware NMP
        upadm show version：（--有版本(关键字Software Version)
        回显就是UltraPath）--返回字符串UltraPath
        # esxcli upadm show version
        Software Version   : XX.XX.XXX
        Driver   Version   : XX.XX.XXX

        esxcli storage core device list---
        NMP多路径判定----如果是，返回字符串VMware NMP
        回显示例：
        naa.xxxxxxxx
        Display Name: HUAWEI Fibre Channel Disk
        (naa.6xxxxx)
        Has Settable Display Name: true
        Size: 4194304
        Device Type: Direct-Access
        Multipath Plugin: NMP
        每个LUN naa.**字符串，匹配到HUAWEI Fibre Channel Disk
        和Multipath Plugin: NMP则为NMP多路径
        :return:
        """
        multi_path_list = list()
        cmd = "esxcli upadm show version|cat"
        flag, cli_ret = cmd_execute(
            self.cli_ret_list, self.cli, cmd, self.logger
        )
        if flag and "Software Version" in cli_ret:
            multi_path_list.append(self.ULTRA_PATH)

        cmd = "esxcli storage core device list|cat"
        flag, cli_ret = cmd_execute(
            self.cli_ret_list, self.cli, cmd, self.logger
        )
        if (
            flag
            and re.compile(r"HUAWEI[\s\S]*Disk").findall(cli_ret)
            and "Multipath Plugin:" in cli_ret
        ):
            multi_path_list.append("VMware NMP")

        self.host_info.setMultipathingSoftware(";".join(multi_path_list))

    def build_hba_info(self, hba_card):
        """
        设置每个HBA的详细信息
        hostbusadapter 为主机总线适配器
        hbadriverversion 未HBA卡驱动版本
        hbafirmwareversion 为HBA卡固件版本
        :param hba_card: hba卡名称
        :return:
        """
        hba_info = DeployHbaInfo()
        self.set_interface_bus_adapter(hba_card, hba_info)

        self.set_interface_driver(hba_card, hba_info)

        self.set_interface_firmware(hba_card, hba_info)

        return hba_info

    def set_interface_firmware(self, hba_card, hba_info):
        cmd = (
            "esxcli storage san fc list --adapter %s | grep 'Firmware "
            "Version: ' | awk -F 'Firmware Version: ' '{print $2}'" % hba_card
        )
        _, cli_ret = cmd_execute(self.cli_ret_list, self.cli, cmd, self.logger)
        hba_info.setHbaFirmwareVersion(self.vmware_get_last_line(cli_ret, cmd))

    def set_interface_driver(self, hba_card, hba_info):
        cmd = (
            "esxcli storage san fc list --adapter %s | grep 'DriverVersion:"
            " ' | awk -F 'DriverVersion: ' '{print $2}'" % hba_card
        )
        _, cli_ret = cmd_execute(self.cli_ret_list, self.cli, cmd, self.logger)
        hba_info.setHbaDriverVersion(self.vmware_get_last_line(cli_ret, cmd))

    def set_interface_bus_adapter(self, hba_card, hba_info):
        cmd = (
            "esxcli storage core adapter list |grep -i 'Fibre Channel' | "
            "grep -w %s | awk -F ') ' '{print $2}'" % hba_card
        )
        _, cli_ret = cmd_execute(self.cli_ret_list, self.cli, cmd, self.logger)
        hba_info.setHostBusAdapter(self.vmware_get_last_line(cli_ret, cmd))

    def check_vmware_valid_echo(self, cli_ret, cmd):
        """
        vmware发现当命令超过76个字符时，会把命令截取
        :param cli_ret: 回文
        :param cmd: 命令
        :return: 是否有效命令
        """
        intercepted_char = 76
        if not cli_ret:
            return True
        if cmd in cli_ret:
            return True
        return len(cmd) >= intercepted_char and (
            cmd[intercepted_char:] in cli_ret
            or cmd[:intercepted_char] in cli_ret
        )

    def vmware_get_last_line(self, cli_ret, cmd):
        """
        由于存在命令被截取的问题，所以针对vmware做了特殊处理
        :param cli_ret: 回文
        :param cmd: 命令
        :return: 最后一行有效回文
        """
        last_line = self.get_last_line(cli_ret)
        return (
            "" if self.check_vmware_valid_echo(last_line, cmd) else last_line
        )
