# -*- coding: UTF-8 -*-
import sys
import os
import json

path = os.path.dirname(os.path.abspath(__file__))
path = os.path.join(path, "..\\..")
sys.path.append(path)
from common import util
from common import contentParse
from common import constants
from common.exception import CmdExecuteException
from java.lang import OutOfMemoryError

ITEM_KEY = 'LunCapacityConsistencyCheck'
SHOW_DISK_CMD = 'esxcfg-info -s | grep "=+LUN \|\-External Id\.\|\-Vendor\.' \
                '\|\-Size\.\|\-Number of Outstanding IO Requests"'
LUN_KEY = '=+LUN'
LUN_END_KEY = '-Number of Outstanding IO Requests'
LUN_WWN_KEY = '-External Id.'
LUN_CAPACITY = '-Size.'
VENDOR = '-Vendor.'
ERROR_KEY = 'lun.capacity.consistency.check.lun.execute.error'
CMD_KEY_WORD = "capacity_consitency_"


def execute(context):
    return LunCapacityConsistencyVmWare(context).execute()


class LunCapacityConsistencyVmWare:

    def __init__(self, context):
        self.context = context
        self.cli = context.get("SSH")
        self.logger = context.get("Logger")
        self.language = context.get("lang")
        self.all_cli_ret = []
        self.lun_wwn_capacity_dict = {}
        self.execute_error_keys = []
        self.fail_cmds = []

    def execute(self):
        util.updateItemProgress(self.context, constants.PROG5)
        try:
            self.get_lun_wwn_capacity()
        except CmdExecuteException as cmdException:
            self.logger.error(
                "exsi lun capacacity consistency cmd error" +
                str(cmdException))
        except (Exception, OutOfMemoryError):
            self.logger.error(
                "execute exsi lun capacacity consistency error.")
            self.execute_error_keys.append(ERROR_KEY)
        self.build_result()
        self.build_result_for_infograb()
        util.updateItemProgress(self.context, constants.PROG90)

    def get_lun_wwn_capacity(self):
        """
        获取磁盘信息
        example:
        esxcfg-info
        |===+LUN:
        |----Name..................................naa.12344556678
        |----External ID...........................naa.12344556678
        |----Size..................................naa.12344556678

        :return:
        """
        cli_ret = self.cli.execCmdNoLogTimout(SHOW_DISK_CMD,
                                              constants.HOST_CMD_TIMEOUT)
        cli_ret_lines = cli_ret.splitlines()
        # 回显超大（数十万行），减轻解析压力
        if len(cli_ret_lines) <= 2:
            self.all_cli_ret.append(cli_ret)
            self.fail_cmds.append(SHOW_DISK_CMD)
            raise CmdExecuteException(SHOW_DISK_CMD, '')
        wwn = ''
        capacity = ''
        # 现在解析的片段是否属于LUN
        find_lun = False
        # 当前解析的LUN是否可以被记录
        save_lun = False
        # 装载命令行
        self.all_cli_ret.append(SHOW_DISK_CMD)
        for line in cli_ret_lines:
            if LUN_KEY in line:
                # 检查到lun开头时初始两个参数值
                wwn = ''
                capacity = ''
                find_lun = True
                save_lun = True
            # 只记录LUN指定位置间的回显，防止回显过大
            if not find_lun:
                continue
            if LUN_END_KEY in line:
                self.all_cli_ret.append(line)
                find_lun = False
            if find_lun:
                self.all_cli_ret.append(line)
            if VENDOR in line:
                vendor = self.analysis_value(line)
                if vendor != 'HUAWEI':
                    # 非HW Lun不装载
                    save_lun = False
                    continue
            if LUN_WWN_KEY in line:
                wwn = self.analysis_value(line)
                if wwn.startswith('naa.'):
                    wwn = wwn[4:]
            if LUN_CAPACITY in line:
                capacity = self.analysis_value(line)
            # 发现lun且装载完wwn和容量后，加载数据
            if save_lun is True and bool(wwn) and bool(capacity):
                save_lun = False
                self.lun_wwn_capacity_dict[wwn] = capacity
        return True

    def analysis_value(self, line):
        """
        解析 XXX...............xxxxxx,取出...之后的值
        :param line:
        :return:
        """
        if '.' not in line:
            return ''
        start = line.rindex('.') + 1
        return line[start:].strip()

    def build_result(self):
        """
        组装查询结果
        :return:
        """
        ret_map = self.context.get("ret_map")
        luns_in_precise_mode = {"cli_rets": '\n'.join(self.all_cli_ret),
                                "execute_error": self.execute_error_keys,
                                "fail_cmds": ",".join(self.fail_cmds),
                                "lun_wwn_capacitys":
                                    self.lun_wwn_capacity_dict}
        self.logger.info(
            'vmware_luns_in_precise_mode:' + str(luns_in_precise_mode))
        contentParse.setCmdRet4Eval(ret_map,
                                    json.dumps(luns_in_precise_mode),
                                    "HostLunWWNCapacsityInfo")

    def build_result_for_infograb(self):
        """
        为infograb收集项构造结果
        :return:
        """
        ret_dict = self.context.get("ret_map")
        cmd_key = "cmd_display_" + CMD_KEY_WORD + "lun_info"
        ret_dict.put(cmd_key, '\n'.join(self.all_cli_ret))
        message = SHOW_DISK_CMD + util.get_collect_cli_result(True,
                                                              self.language)
        ret_dict.put("err_msg", message)
