# coding=utf-8
# 获取vmvare主机的华为lun的aa_fix模式的lun
import os
import sys
import json

path = os.path.dirname(os.path.abspath(__file__))
path = os.path.join(path, "..\\..")
sys.path.append(path)
from common import util, constants, contentParse, cliUtil
from common import contextUtil
from common.constants import EvaluStatus

CHECK_RESULT_KEY_TO_EVAL = "VMwareAAFixedModelCheck"
CHECK_RESULT_FLAG = "checkResultFlag"
CHECK_ORIGINAL_INFO = "checkOriginalInfo"
VMWARE_AA_FIXED_DEVICE_NAME = "vmwareAaFixedDeviceName"


def execute(context):
    VMwareAAFixedModelCheck(context).get_huawei_aa_fixed_lun_device_name()


class VMwareAAFixedModelCheck(object):
    """
    检查方法：
    步骤1：以ssh方式使用root账户登录ESXi主机
    步骤2：执行命令esxcli storage core device list，显示当前阵列所有映射的LUN信息
    步骤3：匹配回显中带有Vendor: HUAWEI和Model: XSG1和Multipath Plugin: NMP字段的华为LUN
    步骤4：匹配出带有上述字段所有LUN的device name(例如：naa.618cf24100f8f4571c25813600000ad3)
    步骤5：对所有步骤4中的华为LUN遍历执行命令esxcli storage nmp device list -d [device name],获取华为LUN的信息
    （例如：esxcli storage nmp device list -d naa.618cf24100f8f4571c25813600000ad3）
    步骤6：匹配回显中带有Storage Array Type: VMW_SATP_DEFAULT_AA和Path Selection Policy: VMW_PSP_FIXED两个字段的LUN
    步骤7：打印出步骤6中匹配的LUN 的device name 和ESXi 主机的IP

	检查标准：
    1 步骤3中能匹配到字段Vendor值为HUAWEI且Model为XSG1且Multipath Plugin为NMP，则继续执行步骤4，否则检查结果为通过
    2 步骤4匹配出所有多路径为NMP的华为LUN，继续执行步骤5
    3 步骤6获取多路径为NMP的华为LUN的信息，
    命令返回结果同时包含Storage Array Type: VMW_SATP_DEFAULT_AA和Path Selection Policy: VMW_PSP_FIXED，
    则继续执行步骤7;否则检查结果为通过

    """

    def __init__(self, context):
        self.context = context
        self.cli = context.get("SSH")
        self.ret_map = contextUtil.get_ret_map(context)
        self.logger = contextUtil.get_logger(context)
        self.lang = contextUtil.get_lang(context)
        self.cli_rets = list()
        self.device_name_list = []
        self._err_msg = ""

    def get_vmware_lun_info_list(self):
        """
            执行命令esxcli storage core device list，显示当前阵列所有映射的LUN信息
            :param cli_ret: 需要解析的回文
            :return: True命令执行结果,  vlu_id_cmd执行命令, cli_ret命令回显,
            lun_info_list解析的回文信息{device_name:{luninfo}}
        """

        self.logger.info("get_vmware_lun_info_list start.")
        vlu_id_cmd = constants.NMP_EXIST_COMMAND
        lun_info_device_name = ""
        lun_info = {}
        lun_info_list = {}
        i = 0

        flag, cli_ret, infoGrab_err_msg = cliUtil.exec_cmd_has_log_vmware(self.cli, vlu_id_cmd, self.lang)
        self.cli_rets.append(cli_ret)
        self.ret_map.put("cmd_display_all_lun_info", cli_ret)
        self._err_msg += infoGrab_err_msg
        self.logger.info("get_vmware_lun_info_list cliret:{}".format(cli_ret))
        if flag is not True:
            return False, vlu_id_cmd, cli_ret, {}

        for line in cli_ret.splitlines():
            i += 1
            strip_line = line.strip()
            if "esxcli storage core device list" in strip_line:
                continue
            if strip_line.startswith("naa"):
                lun_info_device_name = strip_line
                continue
            if len(strip_line) == 0 or i == len(cli_ret.splitlines()):
                self.logger.info(
                    "get_vmware_lun_info_list len(strip_line):{},len(cli_ret):{} i={}".format(len(strip_line),
                                                                                              len(cli_ret), i))
                if len(lun_info_device_name) == 0:
                    continue
                lun_info_list[lun_info_device_name] = lun_info.copy()
                lun_info_device_name = ""
                lun_info.clear()
                continue
            if ":" in strip_line:
                split_line = strip_line.split(":")
                lun_info[split_line[0].strip()] = split_line[1].strip()
        return True, vlu_id_cmd, cli_ret, lun_info_list

    def get_vmware_lun_device_name(self, lun_info_list):
        """
            匹配回显中带有Vendor: HUAWEI和Model: XSG1和Multipath Plugin: NMP字段的华为LUN
            :param lun_info_list: 所用lun的信息的字典。
            :return: lun_device_name_list,匹配出所有多路径为NMP的华为LUN的devicename
        """

        lun_device_name_list = []
        for device_name, lun_info in lun_info_list.items():
            if lun_info["Vendor"] == "HUAWEI" and lun_info["Model"] == "XSG1" and \
                    lun_info["Multipath Plugin"] == "NMP":
                lun_device_name_list.append(device_name)
        return lun_device_name_list

    def get_huawei_lun_nmp_info(self, device_name):

        """
            对所有步骤4中的华为LUN遍历执行命令esxcli storage nmp device list -d [device name],获取华为LUN的信息
            esxcli storage nmp device list -d naa.618cf24100f8f4571c25813600000ad3
            :param device_name: 带有Vendor: HUAWEI和Model: XSG1和Multipath Plugin: NMP字段的华为LUN的devicename。
            :return: True命令执行结果,  cmd执行命令, cli_ret命令回显,
            huawei_lun_info解析的回文信息{ Storage Array Type: VMW_SATP_DEFAULT_AA,.....}
        """

        self.logger.info("get_huawei_lun_nmp_info start:")
        huawei_lun_info = {}
        i = 0
        cmd = "esxcli storage nmp device list -d {}".format(device_name)
        flag, cli_ret, err_msg = cliUtil.exec_cmd_has_log_vmware(self.cli, cmd, self.lang)
        self.cli_rets.append(cli_ret)
        self.ret_map.put("cmd_display_huawei_lun_info_{}".format(device_name), cli_ret)
        self._err_msg += err_msg
        self.logger.info("get_huawei_lun_nmp_info cliret:{} ,flag:{}".format(cli_ret, flag))
        if flag is not True:
            return False, cmd, cli_ret, {}

        for line in cli_ret.splitlines():
            i += 1
            strip_line = line.strip().lower()
            if "esxcli storage nmp device list -d" in strip_line:
                continue
            if strip_line.startswith("naa"):
                continue
            if len(strip_line) == 0 or i == len(cli_ret) or ":" not in strip_line:
                continue
            split_line = strip_line.split(":")
            huawei_lun_info[split_line[0].strip()] = split_line[1].strip()
        return True, cmd, cli_ret, huawei_lun_info

    def check_huawei_lun_aa_fixed(self, huawei_lun_info):
        """
            匹配回显中带有Storage Array Type: VMW_SATP_DEFAULT_AA和Path Selection Policy: VMW_PSP_FIXED两个字段的LUN
            :param huawei_lun_info: 多路径为NMP的华为LUN的信息的字典。
            :return: True,是否匹配。
        """

        if huawei_lun_info["storage array type"] == "vmw_satp_default_aa" and \
                huawei_lun_info["path selection policy"] == "vmw_psp_fixed":
            return True
        return False

    def get_huawei_aa_fixed_lun_device_name(self):

        try:
            util.updateItemProgress(self.context, constants.PROG30)
            self.logger.info("get_huawei_aa_fixed_lun_device_name start.")
            flag, vlu_id_cmd, cli_ret, lun_info_list = self.get_vmware_lun_info_list()
            if flag is not True:
                self._set_check_result(EvaluStatus.NOTPASS)
                self.logger.error("get vmware all lun info failed.")
                return
            self.logger.info("lun_info_list:{}".format(lun_info_list))
            lun_device_name_list = self.get_vmware_lun_device_name(lun_info_list)
            if not lun_device_name_list:
                self._set_check_result(EvaluStatus.PASS)
                self.logger.info("get vmware huawei lun info is empty.")
                return
            self.logger.info("lun_device_name_list:{}".format(lun_device_name_list))
            for device_name in lun_device_name_list:
                util.updateItemProgress(self.context, constants.PROG50)
                result, cmd, cli_ret, huawei_lun_info = self.get_huawei_lun_nmp_info(device_name)
                self.logger.info("resulut:{} ".format(result))

                if result is not True:
                    self._set_check_result(EvaluStatus.NOTPASS)
                    self.logger.error("get vmware huawei lun info failed.device_name:{}".format(device_name))
                    return

                flag = self.check_huawei_lun_aa_fixed(huawei_lun_info)
                if flag is True:
                    self.device_name_list.append(device_name)
            self.logger.info("device_name_list:{}".format(self.device_name_list))
            if len(self.device_name_list) == 0:
                self._set_check_result(EvaluStatus.PASS)
                return
            # 继续到兼容性评估中做检查
            self._set_check_result(EvaluStatus.OPTIM_RECOMD)
        except Exception as exception:
            self.logger.error("[VMwareAAFixedModelCheck] " + str(exception))
            self._set_check_result(EvaluStatus.NO_CHECK)
        finally:
            util.updateItemProgress(self.context, constants.PROG95)
            self.ret_map.put("err_msg", self._err_msg)

    def get_cli_ret(self):
        if len(self.cli_rets) == 0:
            return "no information."
        return '\n'.join(self.cli_rets)

    def get_device_name_list(self):
        if len(self.device_name_list) == 0:
            return "no information."
        return '\n'.join(self.device_name_list)

    def _set_check_result(self, check_flag):
        """
        设置检查结果给兼容性评估
        :param check_flag: 检查结果
        """
        check_result = {CHECK_RESULT_FLAG: check_flag,
                        CHECK_ORIGINAL_INFO: self.get_cli_ret(),
                        VMWARE_AA_FIXED_DEVICE_NAME: self.get_device_name_list()}

        contentParse.setCmdRet4Eval(self.ret_map, json.dumps(check_result),
                                    CHECK_RESULT_KEY_TO_EVAL)
