# coding=utf-8
import os
import re
import sys
import traceback

path = os.path.dirname(os.path.abspath(__file__))
path = os.path.join(path, "..\\..")
sys.path.append(path)
from common import util
from common import contentParse

cmdInfoIdList = ["cmd_info_hba_later55_Emulex_driver_version_%s",
                 "cmd_info_hba_later55_Emulex_fw_version_%s",
                 "cmd_info_hba_later55_Qlogic_driver",
                 "cmd_info_hba_later55_Qlogic_driver_info_%s",
                 "cmd_info_hba_berfore55_Emulex_info_%s",
                 "cmd_info_hba_berfore55_Qlogic_info_%s",
                 "cmd_info_hba_later55_Qlogic_info",
                 ]
commandList = ["esxcfg-module -i %s | grep -i version",
               "/usr/lib/vmware/vmkmgmt_keyval/vmkmgmt_keyval -l -i %s/Emulex",
               "/usr/lib/vmware/vmkmgmt_keyval/vmkmgmt_keyval -d",
               "/usr/lib/vmware/vmkmgmt_keyval/vmkmgmt_keyval -l -i %s",
               "head -5 %s",
               "head -5 %s",
               "/usr/lib/vmware/vmkmgmt_keyval/vmkmgmt_keyval -a",
               ]

def qryVMwareVersion(context, CLI):
    '''
    @summary: Query the VMware version.
    @param context: Python execution context.
    @return (VMwareVersion)
    '''
    try:
        cliRet = CLI.execCmdWithNoCheckResult("vmware -l", 60)
        verPattern = re.compile(r'(\d+\.\d+\.\d+)')
        version = verPattern.search(cliRet)
        if version:
            return tuple(map(int, version.group().split('.')))
        else:
            return (0, 0, 0)
    except:
        util.log.error(context, 'Query VMware version exception:' + traceback.format_exc())
        return (0, 0, 0)

def qryHBAInfo(context, CLI):
    '''
    @summary: Query the HBA infomation.
    @param context: Python execution context.
    @return (HbaVersion)
    '''
    
    qlogicInfoList = [] #里面是元组（name,driver,Model）
    emulexInfoList = [] #里面是元组（name,driver,Model）
    try:
        qlogicPattern = re.compile(r'(link-up).*(QLogic)', re.I)
        emulexPattern = re.compile(r'(link-up).*(Emulex).*(LP[0-9a-z\-]*|OCe[0-9a-z\-]*)', re.I)
        cliRet = CLI.execCmdWithNoCheckResult("esxcli storage core adapter list", 60)
        for line in cliRet.splitlines():
            qlogicInfo = qlogicPattern.search(line)
            emulexInfo = emulexPattern.search(line)
            if qlogicInfo:
                qlogicInfoList.append((line.split()[0], line.split()[1], qlogicInfo.group(2)))
            elif emulexInfo:
                emulexInfoList.append((line.split()[0], line.split()[1], emulexInfo.group(3)))
            else:
                continue
        util.log.info(context, 'Query qlogic hba infomation:' + str(qlogicInfoList))
        util.log.info(context, 'Query emulex hba infomation:' + str(emulexInfoList))
        return qlogicInfoList, emulexInfoList
    except:
        util.log.error(context, 'Query hba infomation exception:' + traceback.format_exc())
        return [], []
def qryEmulexInfoLater55(context, CLI, emulexInfoList):
    '''
    @summary: Query the Emulex HBA later 5.5 infomation.
    @param context: Python execution context.
    @return (Emulexinfo)
    '''
    emulexInfoTupList = [] #Model,driVer,FwVer
    emulexDriverVer = ""
    emulexFWVer = ""
    try:
        emulexDriverVerPattern = re.compile(r'(\d+\.\d+\.\d+\.\d+)[\-]', re.I)
        for (hbaName, hbaDriver, hbaModel) in emulexInfoList:
            driverCliRet = contentParse.getSingleCommandRetPureRetWithoutResCheck(context, \
                     cmdInfoIdList[0] % hbaDriver, commandList[0] % hbaDriver)
            for line in driverCliRet.splitlines():
                emulexDriverRe = emulexDriverVerPattern.search(line)
                if emulexDriverRe:
                    emulexDriverVer = emulexDriverRe.group(1)
                    break
            fwVercliRet = contentParse.getSingleCommandRetPureRetWithoutResCheck(context, \
                     cmdInfoIdList[1] % hbaName, commandList[1] % hbaName)
            for line in fwVercliRet.splitlines():
                if "fw version:" in line.lower():
                    emulexFWVer = line.split(":")[-1].strip()
                    break
            if hbaModel and emulexDriverVer and emulexFWVer:
                emulexInfoTupList.append((hbaModel, emulexDriverVer, emulexFWVer))

        util.log.info(context, 'Query emulex hba infomation later 5.5:' + str(emulexInfoTupList))
        return list(set(emulexInfoTupList))
    except:
        util.log.error(context, 'Query 5.5 later version emulex hba infomation exception:' + traceback.format_exc())
        return []

def qryQlogicInfoLater55(context, CLI, qlogicInfoList):
    '''
    @summary: Query the qlogic HBA later 5.5 infomation.
    @param context: Python execution context.
    @return (qlogicinfo)
    '''

    qlogicInfoTupList = [] ##Model,driVer,FwVer
    qlogicDriver = []
    qlogicInfo = ""
    qlogicModel = ""
    qlogicVersionInfo = ""
    qlogicFwVer = ""
    qlogicDriVer = ""

    try:
        driverCliRet = contentParse.getSingleCommandRetPureRetWithoutResCheck(context, cmdInfoIdList[2], commandList[2])
        for line in driverCliRet.splitlines():
            if "key value instance:" in line.lower():
                qlogicDriverTmp = line.split(":")[-1].strip()
                util.log.info(context, 'Query qlogic hba driver later 5.5 is:' + str(qlogicDriverTmp))
                for (hbaName, _, _) in qlogicInfoList:
                    if hbaName in qlogicDriverTmp:
                        qlogicDriver.append(qlogicDriverTmp)
                        break
        util.log.info(context, 'Query qlogic hba driver later 5.5:' + str(qlogicDriver))
        qlogicPattern = re.compile(r'(QLogic).*(QLA[0-9]*|BR-[0-9a-z\-]*|QLE[0-9a-z\(\)]*|QME[0-9]*|QMH[0-9]*|QMI[0-9]*)', re.I)
        qlogicVersionPattern = re.compile(r'Firmware version:* ([0-9a-z\.]*).*Driver Version:* ([0-9a-z\.]*)', re.I)
        for driver in qlogicDriver:
            driverInfoCliRet = contentParse.getSingleCommandRetPureRetWithoutResCheck(context, \
                           cmdInfoIdList[3] % driver, commandList[3] % driver)
            for line in driverInfoCliRet.splitlines():
                qlogicInfo = qlogicPattern.search(line)
                qlogicVersionInfo = qlogicVersionPattern.search(line)
                if qlogicInfo:
                    qlogicModel = qlogicInfo.group(2)
                elif qlogicVersionInfo:
                    qlogicFwVer = qlogicVersionInfo.group(1)
                    qlogicDriVer = qlogicVersionInfo.group(2)
                    break
            if qlogicModel and qlogicDriVer and qlogicFwVer:
                qlogicInfoTupList.append((qlogicModel, qlogicDriVer, qlogicFwVer))

        if not qlogicInfoTupList:
            util.log.info(context, 'Query qlogic hba infomation failed')
            qlogicPattern = re.compile(r'(QLogic).*(QLA[0-9]*|BR-[0-9a-z\-]*|QLE[0-9a-z\(\)]*|QME[0-9]*|QMH[0-9]*|QMI[0-9]*)', re.I)
            qlogicVersionPattern = re.compile(r'Firmware version:* ([0-9a-z\.]*).*Driver Version:* ([0-9a-z\.]*)', re.I)
            driverInfoCliRet = contentParse.getSingleCommandRetPureRetWithoutResCheck(context, \
                           cmdInfoIdList[6], commandList[6])
            for line in driverInfoCliRet.splitlines():
                qlogicInfo = qlogicPattern.search(line)
                qlogicVersionInfo = qlogicVersionPattern.search(line)
                if qlogicInfo:
                    qlogicModel = qlogicInfo.group(2)
                    continue
                elif qlogicVersionInfo:
                    qlogicFwVer = qlogicVersionInfo.group(1)
                    qlogicDriVer = qlogicVersionInfo.group(2)

                if qlogicModel and qlogicDriVer and qlogicFwVer:
                    qlogicInfoTupList.append((qlogicModel, qlogicDriVer, qlogicFwVer))
                    qlogicModel = ""
                    qlogicDriVer = ""
                    qlogicFwVer = ""

        util.log.info(context, 'Query qlogic hba infomation later 5.5:' + str(qlogicInfoTupList))
        return list(set(qlogicInfoTupList))
    except:
        util.log.error(context, 'Query 5.5 later version qlogic hba infomation exception:' + traceback.format_exc())
        return []

def qryQlogicInfoBefore55(context, CLI, qlogicInfoList):
    '''
    @summary: Query the qlogic HBA before 5.5 infomation.
    @param context: Python execution context.
    @return (qlogicinfo)
    '''
    qlogicInfoTupList = [] #Model,driVer,FwVer
    portNumberList = []
    qlogicDrVerInfo = ''
    qlogicFwVerInfo = ''
    qlogicModeInfo = ''
    qlogicModelPattern = re.compile(r'(QLogic).*(QLA[0-9]*|BR-[0-9a-z\-]*|QLE[0-9a-z\(\)]*|QME[0-9]*|QMH[0-9]*|QMI[0-9]*)', re.I)
    qlogicVersionPattern = re.compile(r'Firmware version:* ([0-9a-z\.]*).*Driver Version:* ([0-9a-z\.]*)', re.I)
    try:
        for (hbaName, hbaDriver, _) in qlogicInfoList:
            _ = CLI.execCmdWithNoCheckResult("cd /proc/scsi/%s" % hbaDriver, 60)
            cliRet = CLI.execCmdWithNoCheckResult("ls |cat", 60)
            for line in cliRet.splitlines():
                if line.strip().isdigit():
                    portNumberList.append(line.strip())

            for protNum in portNumberList:
                qlogicInfoCliRet = contentParse.getSingleCommandRetPureRetWithoutResCheck(context, \
                                   cmdInfoIdList[5] % protNum, commandList[5] % protNum)

                if hbaName not in qlogicInfoCliRet:
                    continue

                for line in qlogicInfoCliRet.splitlines():
                    qlogicModeliInfo = qlogicModelPattern.search(line)
                    qlogicVersionInfo = qlogicVersionPattern.search(line)
                    if qlogicModeliInfo:
                        qlogicModeInfo = qlogicModeliInfo.group(2)
                    elif qlogicVersionInfo:
                        qlogicFwVerInfo = qlogicVersionInfo.group(1)
                        qlogicDrVerInfo = qlogicVersionInfo.group(2)

                if qlogicModeInfo and qlogicDrVerInfo and qlogicFwVerInfo:
                    qlogicInfoTupList.append((qlogicModeInfo, qlogicDrVerInfo, qlogicFwVerInfo))

        util.log.info(context, 'Query qlogic hba infomation before 5.5:' + str(qlogicInfoTupList))
        return list(set(qlogicInfoTupList))
    except:
        util.log.error(context, 'Query 5.5 before version qlogic hba infomation exception:' + traceback.format_exc())
        return []

def qryEmulexInfoBefore55(context, CLI, emulexInfoList):
    '''
    @summary: Query the Emulex HBA before 5.5 infomation.
    @param context: Python execution context.
    @return (Emulexinfo)
    '''

    emulexInfoTupList = [] #Model,driVer,FwVer
    portNumberList = []
    emulexDrVerInfo = ''
    emulexFwVerInfo = ''
    emulexDriVersionPattern = re.compile(r'(Emulex).* (\d+\.\d+\.\d+\.\d+)', re.I)
    emulexFwVersionPattern = re.compile(r'Firmware Version:* ([0-9a-z\.]*)', re.I)
    try:
        for (hbaName, hbaDriver, hbaModel) in emulexInfoList:
            _ = CLI.execCmdWithNoCheckResult("cd /proc/scsi/%s" % hbaDriver, 60)
            cliRet = CLI.execCmdWithNoCheckResult("ls|cat", 60)
            for line in cliRet.splitlines():
                if line.strip().isdigit():
                    portNumberList.append(line.strip())
            for protNum in portNumberList:
                emulexInfoCliRet = contentParse.getSingleCommandRetPureRetWithoutResCheck(context, \
                                   cmdInfoIdList[4] % protNum, commandList[4] % protNum)

                if hbaName not in emulexInfoCliRet:
                    continue

                for line in emulexInfoCliRet.splitlines():
                    emulexDrVersionInfo = emulexDriVersionPattern.search(line)
                    emulexFwVersionInfo = emulexFwVersionPattern.search(line)
                    if emulexDrVersionInfo:
                        emulexDrVerInfo = emulexDrVersionInfo.group(2)
                    elif emulexFwVersionInfo:
                        emulexFwVerInfo = emulexFwVersionInfo.group(1)

                if hbaModel and emulexDrVerInfo and emulexFwVerInfo:
                    emulexInfoTupList.append((hbaModel, emulexDrVerInfo, emulexFwVerInfo))

        util.log.info(context, 'Query emulex hba infomation before 5.5:' + str(emulexInfoTupList))
        return list(set(emulexInfoTupList))
    except:
        util.log.error(context, 'Query 5.5 before version emulex hba infomation exception:' + traceback.format_exc())
        return []

def combinationHBAInfo(hbaList):
    '''
    @summary: Query the HBA infomation.
    @param context: Python execution context.
    @return (HBAinfo)
    '''
    hbaModelInfo = ''
    hbaDriverVerInfo = ''
    hbaFwVerInfo = ''
    splitStr = ';'
    for (hbaModel, hbaDriverVer, hbaFwVer) in hbaList:
        hbaModelInfo += hbaModel + splitStr
        hbaDriverVerInfo += hbaDriverVer + splitStr
        hbaFwVerInfo += hbaFwVer + splitStr
    return hbaModelInfo, hbaDriverVerInfo, hbaFwVerInfo

def writeResultToHtml(context, cmd_info_ret):
    '''
    @summary: write Result HBA infomation to Html.
    @param context: Python execution context.
    @return (display)
    '''
    display = context.get("ret_map")
    display.put("HBA_information", "HBA_information \r\n" + cmd_info_ret)
    return display

def execute(context):
    '''
    @summary: Query the HBA infomation.
    @param context: Python execution context.
    @return (HBAinfo)
    '''
    CLI = context.get("SSH")
    HbaCardInfoDict = {"Model":"", "DriverVersion":"", "FWVersion":""}

    qhbaModelInfo = ''
    qhbaDriverVerInfo = ''
    qhbaFwVerInfo = ''
    ehbaModelInfo = ''
    ehbaDriverVerInfo = ''
    ehbaFwVerInfo = ''
    qlogicInfo = []
    emulexInfo = []
    VmwareVersion = qryVMwareVersion(context, CLI)
    qlogicInfoList, emulexInfoList = qryHBAInfo(context, CLI)
    if not qlogicInfoList and not emulexInfoList:
        HbaCardInfoDict["Model"] = 'LINK-NA;'
        HbaCardInfoDict["DriverVersion"] = 'LINK-NA;'
        HbaCardInfoDict["FWVersion"] = 'LINK-NA;'
        util.log.info(context, 'Query hba infomation find all hba status are link-down.')
        writeResultToHtml(context, str(HbaCardInfoDict))
        return HbaCardInfoDict

    if VmwareVersion < (5, 5, 0):
        if qlogicInfoList:
            qlogicInfo = qryQlogicInfoBefore55(context, CLI, qlogicInfoList)
            qhbaModelInfo, qhbaDriverVerInfo, qhbaFwVerInfo = combinationHBAInfo(qlogicInfo)

        if emulexInfoList:
            emulexInfo = qryEmulexInfoBefore55(context, CLI, emulexInfoList)
            ehbaModelInfo, ehbaDriverVerInfo, ehbaFwVerInfo = combinationHBAInfo(emulexInfo)

        HbaCardInfoDict["Model"] = qhbaModelInfo + ehbaModelInfo
        HbaCardInfoDict["DriverVersion"] = qhbaDriverVerInfo + ehbaDriverVerInfo
        HbaCardInfoDict["FWVersion"] = qhbaFwVerInfo + ehbaFwVerInfo

        util.log.info(context, 'Query hba infomation before 5.5:' + str(HbaCardInfoDict))
        writeResultToHtml(context, str(HbaCardInfoDict))
        return HbaCardInfoDict
    else:
        if qlogicInfoList:
            qlogicInfo = qryQlogicInfoLater55(context, CLI, qlogicInfoList)
            qhbaModelInfo, qhbaDriverVerInfo, qhbaFwVerInfo = combinationHBAInfo(qlogicInfo)

        if emulexInfoList:
            emulexInfo = qryEmulexInfoLater55(context, CLI, emulexInfoList)
            ehbaModelInfo, ehbaDriverVerInfo, ehbaFwVerInfo = combinationHBAInfo(emulexInfo)

        HbaCardInfoDict["Model"] = qhbaModelInfo + ehbaModelInfo
        HbaCardInfoDict["DriverVersion"] = qhbaDriverVerInfo + ehbaDriverVerInfo
        HbaCardInfoDict["FWVersion"] = qhbaFwVerInfo + ehbaFwVerInfo

        util.log.info(context, 'Query hba infomation later 5.5:' + str(HbaCardInfoDict))
        writeResultToHtml(context, str(HbaCardInfoDict))
        return HbaCardInfoDict
