# --*-- coding:utf-8 --*--
# Copyright (c) Huawei Technologies Co., Ltd. 2021-2021. All rights reserved.
import json
import traceback

from common.migration_constants import NA, WINDOWS_CLUSTER_DISK_CSV, WINDOWS_CLUSTER_DISK_QUORUM, \
    WINDOWS_CLUSTER_DISK_GENERAL
from hosts.common.windows_parse_utils import get_horizontal_cli_ret, get_windows_physical_disks, get_vertical_result, \
    get_id_letter_data
from common.contentParse import get_return_from_txt

global LOGGER
DISK = "Disk "


def execute(context):
    """
    执行入口
    :param context: 上下文
    :return: 解析后数据
    """
    global LOGGER
    LOGGER = context.get("Logger")
    result = get_result(context)
    ret_map = context.get("map")
    ret_map.put("data", json.dumps(result))
    return context


def get_result(context):
    """
    获取数据
    :param context: 上下文
    :return: 数据
    """
    result = {"host_ip": context.get("ip")}
    try:
        result["cluster_name"] = get_cluster_name(context)
        result["disks"] = build_result_data(get_cluster_disk_objects(context))
    except Exception:
        LOGGER.error("migration error: " + str(traceback.format_exc()))
        result = get_empty_data(result)
    return result


def get_cluster_name(context):
    """
    获取集群名字
    :param context: 上下文
    :return: 数据
    """
    context["command"] = "powershell get-cluster"
    cluster_info = get_return_from_txt(context)
    flag = False
    for line in cluster_info:
        if line.strip().startswith("----"):
            flag = True
            continue
        if flag and line.strip():
            return line.strip()
    return ""


def build_result_data(cluster_disks):
    """
    标准化最终数据
    :param cluster_disks: 盘数据
    :return: 数据
    """
    result = []
    if not cluster_disks:
        return get_empty_disks()
    for disk in cluster_disks:
        result.append(
            {"cluster_disk": disk.cluster_disk, "cluster_disk_type": disk.cluster_disk_type,
             "system_disk_name": disk.system_disk_name, "disk_state": disk.disk_state, "disk_info": disk.disk_info})
    return result


def get_cluster_disk_objects(context):
    """
    获取集群磁盘数据--对象
    :param context: 上下文
    :return: 数据
    """
    disks = []
    pd_device_id = get_pd_2_device_id(context)
    id_letter_data = get_id_letter_data(context)
    disks.extend(get_disks(context, pd_device_id, disks, id_letter_data, WINDOWS_CLUSTER_DISK_QUORUM))
    disks.extend(get_disks(context, pd_device_id, disks, id_letter_data, WINDOWS_CLUSTER_DISK_CSV))
    disks.extend(get_disks(context, pd_device_id, disks, id_letter_data, WINDOWS_CLUSTER_DISK_GENERAL))
    return disks


def get_quorum_names(disks):
    """
    获取仲裁盘 名字
    :param disks: 已解析出的仲裁和共享盘
    :return: 数据
    """
    quorum_names = []
    for disk in disks:
        if WINDOWS_CLUSTER_DISK_QUORUM == disk.cluster_disk_type:
            quorum_names.append(disk.cluster_disk)
    return quorum_names


def get_disk_command_by_disk_type(disk_type):
    """
    根据盘类型获取 查询盘的命令
    :param disk_type: 盘类型
    :return: 数据
    """
    if WINDOWS_CLUSTER_DISK_GENERAL == disk_type:
        return "powershell Get-ClusterResource"
    if WINDOWS_CLUSTER_DISK_CSV == disk_type:
        return "powershell Get-ClusterSharedVolume"
    if WINDOWS_CLUSTER_DISK_QUORUM == disk_type:
        return "powershell Get-ClusterQuorum"


def check_disk_dict(disk_type, one_dict, quorum_names):
    """
    检查盘数据：
        非普通盘时，都解析
        普通盘时，ResourceType需要是Physical Disk，Name不是仲裁盘
    :param disk_type: 盘类型
    :param one_dict: 当前盘数据
    :param quorum_names: 仲裁盘
    :return: True-解析，False-不解析
    """
    if WINDOWS_CLUSTER_DISK_GENERAL != disk_type:
        return True
    resource_type = one_dict.get("ResourceType", "")
    disk_name = one_dict.get("Name", "")
    return "Physical Disk" == resource_type and disk_name not in quorum_names


def get_disks(context, pd_device_id, parsed_disks, id_letter_data, disk_type):
    """
    获取 普通盘、仲裁盘、共享盘
    :param context: 上下文
    :param pd_device_id: pd与设备id的映射
    :param parsed_disks: 已解析出的仲裁和共享盘
    :param id_letter_data: id与盘符的映射
    :param disk_type: 盘类型
    :return: 数据
    """
    context["command"] = get_disk_command_by_disk_type(disk_type)
    data_info = get_return_from_txt(context, True)
    dicts = get_horizontal_cli_ret("".join(data_info))
    quorum_names = get_quorum_names(parsed_disks) if WINDOWS_CLUSTER_DISK_GENERAL == disk_type else []
    disks = []
    for one_dict in dicts:
        if not check_disk_dict(disk_type, one_dict, quorum_names):
            continue
        disk = WindowsCluster()
        disk_name = one_dict.get("QuorumResource", "") \
            if WINDOWS_CLUSTER_DISK_QUORUM == disk_type else one_dict.get("Name", "")
        disk.cluster_disk = disk_name
        disk.cluster_disk_type = one_dict.get("OwnerGroup", "") \
            if WINDOWS_CLUSTER_DISK_GENERAL == disk_type else disk_type
        disk.disk_state = get_disk_state(context, disk.cluster_disk) \
            if WINDOWS_CLUSTER_DISK_QUORUM == disk_type else one_dict.get("State", "")
        device_id = get_device_id(context, disk_name, pd_device_id, disk_type)
        disk.system_disk_name = DISK + device_id if device_id else ""
        disk.disk_info = get_disk_info_csv(context, disk.cluster_disk) \
            if WINDOWS_CLUSTER_DISK_CSV == disk_type else id_letter_data.get(device_id, "")
        disks.append(disk)
    return disks


def get_device_id(context, cluster_disk, pd_device_id, disk_type):
    """
    获取系统盘
    :param context: 上下文
    :param cluster_disk: 磁盘名
    :param pd_device_id: pd与设备id的映射
    :param disk_type: 磁盘类型
    :return: 系统盘
    """
    if WINDOWS_CLUSTER_DISK_CSV == disk_type:
        context["command"] = "powershell \"Get-ClusterSharedVolume -Name '%s' | Get-ClusterParameter\"" % cluster_disk
    else:
        context["command"] = "powershell \"Get-ClusterResource -Name '%s' | Get-ClusterParameter\"" % cluster_disk
    data_info = get_return_from_txt(context, True)
    parameter_dicts = get_horizontal_cli_ret("".join(data_info))
    for parameter in parameter_dicts:
        if parameter.get("Name") == "DiskGuid":
            disk_guid = parameter.get("Value", "").strip("{}")
            return pd_device_id.get(disk_guid, "")
    return ""


def get_disk_info_csv(context, cluster_disk):
    """
    获取 CSV 盘信息
    :param context: 上下文
    :param cluster_disk: 磁盘名
    :return: 盘信息
    """
    context["command"] = "powershell \"Get-ClusterSharedVolume '%s' | select *\"" % cluster_disk
    data_info = get_return_from_txt(context, True)
    parameter_dict = get_vertical_result(data_info, ":")
    return parameter_dict.get("SharedVolumeInfo", "").strip("{}")


def get_disk_state(context, cluster_disk):
    """
    获取盘的状态
    :param context: 上下文
    :param cluster_disk: 磁盘名
    :return: 系统盘
    """
    context["command"] = "powershell \"Get-ClusterResource -Name '%s'\"" % cluster_disk
    data_info = get_return_from_txt(context, True)
    data_dicts = get_horizontal_cli_ret("".join(data_info))
    for data in data_dicts:
        return data.get("State", "")
    return ""


def get_pd_2_device_id(context):
    """
    获取PD与DeviceId的映射
    :param context: 上下文
    :return: 数据
    """
    disks = get_windows_physical_disks(context)
    pd_2_device_id = {}
    for disk in disks:
        disk_dict = get_vertical_result(disk, ":")
        device_id = disk_dict.get("DeviceId", "")
        object_id = disk_dict.get("ObjectId", "")
        items = object_id.split("PD:")
        pd = items[1].strip("\"{}") if len(items) > 1 else ""
        if pd and device_id:
            pd_2_device_id[pd] = device_id
    return pd_2_device_id


def get_empty_data(result):
    """
    空数据
    :param result: 结果
    :return: 数据
    """
    result["cluster_name"] = ""
    result["disks"] = get_empty_disks()
    return result


def get_empty_disks():
    """
    空的盘数据
    :return: 数据
    """
    return [{"cluster_disk": "", "cluster_disk_type": "", "system_disk_name": "", "disk_state": "", "disk_info": ""}]


class WindowsCluster(object):
    def __init__(self):
        self.cluster_disk = NA
        self.cluster_disk_type = NA
        self.system_disk_name = NA
        self.disk_state = NA
        self.disk_info = NA

    def str(self):
        return "cluster_disk:" + self.cluster_disk + ",cluster_disk_type:" + self.cluster_disk_type + \
               ",system_disk_name:" + self.system_disk_name + ",disk_state:" + self.disk_state + ",disk_info:" + \
               self.disk_info
