# --*-- coding:utf-8 --*--
# Copyright (c) Huawei Technologies Co., Ltd. 2021-2021. All rights reserved.
import json
import traceback

from common.migration_constants import NA
from hosts.common.windows_parse_utils import get_horizontal_cli_ret
from common.contentParse import get_return_from_txt

global LOGGER


def execute(context):
    """
    执行入口
    :param context: 上下文
    :return: 解析后数据
    """
    global LOGGER
    LOGGER = context.get("Logger")
    result = get_result(context)
    ret_map = context.get("map")
    ret_map.put("data", json.dumps(result))
    return context


def get_result(context):
    """
    获取数据
    :param context: 上下文
    :return: 数据
    """
    result = {"host_ip": context.get("ip")}
    try:
        result["vms"] = get_vms_data(context)
    except Exception:
        LOGGER.error("migration error: " + str(traceback.format_exc()))
        result["vms"] = get_empty_data()
    return result


def get_vms_data(context):
    """
    获取集群数据
    :param context: 上下文
    :return: 数据
    """
    vm_dicts = get_vm_dicts(context)
    cluster_names = get_vm_cluster_names(context)
    vm_data = []
    for vm_dic in vm_dicts:
        hyper = WindowsHyperV()
        vm_name = vm_dic.get("Name", "")
        hyper.virtual_machine = vm_name
        hyper.state = vm_dic.get("State", "")
        hyper.is_in_cluster = is_vm_in_cluster(vm_name, cluster_names)
        hyper.vm_disk_path = get_vm_disk_path(context, vm_name)
        hyper.virtual_hba = get_virtual_hba(context, vm_name)
        vm_data.append(hyper)
    return build_result_data(vm_data)


def get_virtual_hba(context, vm_name):
    """
    获取虚拟hba
    :param context: 上下文
    :param vm_name: 虚拟机名字
    :return: 数据
    """
    context["command"] = "powershell get-vmfibrechannelhba '%s'" % vm_name
    hba_info = get_return_from_txt(context)
    virtual_hba = []
    separator = ":"
    for line in hba_info:
        if line.startswith("WorldWidePortNameSet") and separator in line:
            virtual_hba.append(line.split(separator)[1].strip())
    virtual_hba_str = "\n".join(virtual_hba).strip()
    return virtual_hba_str if virtual_hba_str else NA


def get_vm_disk_path(context, vm_name):
    """
    获取虚拟硬盘
    :param context: 上下文
    :param vm_name: 虚拟机名字
    :return: 数据
    """
    context["command"] = "powershell get-vmharddiskdrive '%s'" % vm_name
    path_info = get_return_from_txt(context, True)
    path_dicts = get_horizontal_cli_ret("".join(path_info))
    node_path = []
    for path_dict in path_dicts:
        vm_path = path_dict.get("Path", "")
        if vm_path:
            node_path.append(vm_path)
    return "\n".join(node_path)


def is_vm_in_cluster(vm_name, cluster_names):
    """
    获取vm是否处于集群中
    :param vm_name: 虚拟机名
    :param cluster_names: 集群数据
    :return: 数据
    """
    return "TRUE" if vm_name in cluster_names else "FALSE"


def get_vm_cluster_names(context):
    """
    获取Virtual Machine类型的集群数据
    :param context: 上下文
    :return: 数据
    """
    context["command"] = "powershell Get-ClusterResource"
    data_dicts = get_horizontal_cli_ret("".join(get_return_from_txt(context, True)))
    cluster_names = []
    for data_dict in data_dicts:
        if "Virtual Machine" == data_dict.get("ResourceType", ""):
            cluster_names.append(data_dict.get("Name", ""))
    return cluster_names


def get_vm_dicts(context):
    """
    获取集群数据字典
    :param context: 上下文
    :return: 数据
    """
    context["command"] = "powershell get-vm"
    return get_horizontal_cli_ret("".join(get_return_from_txt(context, True)))


def build_result_data(vm_data):
    """
    标准化最终数据
    :param vm_data: 虚拟机数据
    :return: 数据
    """
    result = []
    if not vm_data:
        return get_empty_data()
    for data in vm_data:
        result.append(
            {"virtual_machines": data.virtual_machine, "state": data.state, "is_vm_in_cluster": data.is_in_cluster,
             "vm_disk_path": data.vm_disk_path, "virtual_hba": data.virtual_hba})
    return result


def get_empty_data():
    """
    空数据
    :return: 数据
    """
    return [{"virtual_machines": "", "state": "", "is_vm_in_cluster": "", "vm_disk_path": "", "virtual_hba": ""}]


class WindowsHyperV(object):
    def __init__(self):
        self.virtual_machine = NA
        self.state = NA
        self.is_in_cluster = NA
        self.vm_disk_path = NA
        self.virtual_hba = NA
