# --*-- coding:utf-8 --*--
# Copyright (c) Huawei Technologies Co., Ltd. 2021-2021. All rights reserved.
import json
import re
import traceback

from common.migration_constants import HOST_TYPE_WINDOWS, MULTI_TYPE_ULTRA, NA, MULTI_TYPE_POWER_PATH, \
    MULTI_TYPE_HDLM, MULTI_TYPE_VXDMP, MULTI_TYPE_MPIO
from common.util import chang_unit2_gb
from hosts.common.migra_summary_util import get_ultra_version, get_powerpath_version, get_hdlm_version
from common.contentParse import get_return_from_txt
from hosts.common.migration_parse_utils import safe_get_split
from hosts.common.windows_parse_utils import get_vertical_result, get_windows_physical_disks, get_id_letter_data, \
    get_horizontal_cli_ret, get_vxdmp_multi_type_version
from hosts.windows.migration.windows_cluster import DISK


def execute(context):
    """
    执行入口
    :param context: 上下文
    :return: 解析后数据
    """
    result = get_result(context)
    ret_map = context.get("map")
    ret_map.put("data", json.dumps(result))
    return context


def get_result(context):
    """
    获取数据
    :param context: 上下文
    :return: 数据
    """
    result = {}
    host_ip = context.get("ip")
    result["host_ip"] = host_ip
    multipaths = get_window_multipaths(context)
    if not multipaths:
        multipaths = get_window_multipaths_old(context)
    result["multipaths"] = multipaths
    return result


def get_window_multipaths(context):
    """
    获取多路径
    :param context: 上下文
    :return: 数据
    """
    try:
        disks = get_windows_multipath_basic(context)
        build_ultra_info(context, disks)
        build_power_info(context, disks)
        build_hdlm_info(context, disks)
        build_vxdmp_info(context, disks)
        build_mpio_info(context, disks)
        build_drive_letter(context, disks)
        return build_result_data(disks)
    except Exception as e:
        context.get("Logger").error("migration error: " + str(traceback.format_exc()))
        return get_empty_multi_new()


def build_result_data(disks):
    """
    标准化最终数据
    :param disks: 盘数据
    :return: 数据
    """
    result = {}
    for disk in disks:
        result[disk.multi_type] = get_multi_data(disk, result)
    return result.values()


def get_multi_data(disk, result):
    """
    获取结果数据中的指定类型的多路径数据（结果集中没有时，需要新增该类型）
    :param disk: 盘数据
    :param result: 结果数据
    :return: 多路径数据
    """
    if disk.multi_type not in result:
        result[disk.multi_type] = {"multipath_type": disk.multi_type, "version": disk.multi_version, "disk_list": []}
    multi_data = result.get(disk.multi_type)
    disk_list = multi_data.get("disk_list")
    disk_list.append(get_one_disk_data(disk))
    multi_data["disk_list"] = disk_list
    return multi_data


def get_one_disk_data(disk):
    """
    生成一个盘的标准化数据
    :param disk: 盘数据
    :return: 数据
    """
    system_disk = DISK + disk.system_disk_name if disk.system_disk_name else ""
    return {"system_disk": system_disk, "lun_wwn": disk.lun_wwn, "lun_size": disk.lun_size, "vendor": disk.vendor,
            "product": disk.product, "alua": disk.alua_strategy, "drive_letter": disk.drive_letter}


def build_ultra_info(context, disks):
    """
    填充ultra多路径
    :param context: 上下文
    :param disks: 盘数据
    """
    version = get_ultra_version(context, HOST_TYPE_WINDOWS)
    if not version:
        return
    context["command"] = "upadm show vlun"
    lun_list = get_ultra_lun_list(context)
    if not lun_list:
        context["command"] = "upadm show vlun type=all"
        lun_list = get_ultra_lun_list(context)
    build_disks(MULTI_TYPE_ULTRA, version, lun_list, disks)


def build_power_info(context, disks):
    """
    填充PowerPath多路径
    :param context: 上下文
    :param disks: 盘数据
    """
    version = get_powerpath_version(context)
    if not version:
        return
    lun_list = get_power_lun_list(context)
    build_disks(MULTI_TYPE_POWER_PATH, version, lun_list, disks)


def build_hdlm_info(context, disks):
    """
    填充HDLM多路径
    :param context: 上下文
    :param disks: 盘数据
    """
    version = get_hdlm_version(context, HOST_TYPE_WINDOWS)
    if not version:
        return
    lun_list = get_hdlm_lun_list(context)
    build_disks(MULTI_TYPE_HDLM, version, lun_list, disks)


def build_vxdmp_info(context, disks):
    """
    填充VXDMP多路径
    :param context: 上下文
    :param disks: 盘数据
    """
    _, multi_version = get_vxdmp_multi_type_version(context)
    if not multi_version:
        return
    lun_list = get_vxdmp_lun_list(context)
    build_disks(MULTI_TYPE_VXDMP, multi_version, lun_list, disks)


def build_mpio_info(context, disks):
    """
    填充MPIO多路径
    :param context: 上下文
    :param disks: 盘数据
    """
    system_disk_id_2_mpio_disk_id = get_system_disk_id_2_mpio_disk_id(context)
    if not system_disk_id_2_mpio_disk_id:
        return
    for disk in disks:
        if NA == disk.multi_type and disk.system_disk_name in system_disk_id_2_mpio_disk_id:
            disk.multi_version = NA
            disk.multi_type = MULTI_TYPE_MPIO
            disk.alua_strategy = get_alua_strategy(context, system_disk_id_2_mpio_disk_id.get(disk.system_disk_name))


def get_alua_strategy(context, mpio_disk_id):
    """
    获取MPIO的alua策略
    :param context: 上下文
    :param mpio_disk_id: mpio_disk_id
    :return: 数据
    """
    context["command"] = "mpclaim -s -d %s" % mpio_disk_id
    disk_info = get_return_from_txt(context)
    for line in disk_info:
        if "TPG_Id:" in line:
            return "ALUA"
    return "AA"


def get_system_disk_id_2_mpio_disk_id(context):
    """
    获取system_disk的数值与mpio_disk的数值的对应关系
    :param context: 上下文
    :return: 数据
    """
    system_disk_id_2_mpio_disk_id = {}
    context["command"] = "mpclaim -s -d"
    ids_info = get_return_from_txt(context)
    start_flag = False
    for line in ids_info:
        if line.strip().startswith("-----------------"):
            start_flag = True
            continue
        if not start_flag:
            continue
        items = re.split(r"\s+", line)
        if len(items) > 4:
            mpio_disk_id = items[1].replace("Disk", "").strip()
            system_disk_id = items[3].strip()
            system_disk_id_2_mpio_disk_id[system_disk_id] = mpio_disk_id
    return system_disk_id_2_mpio_disk_id


def build_disks(multi_type, version, lun_list, disks):
    """
    填充多路径
    :param multi_type: 多路径类型
    :param version: 多路径版本
    :param lun_list: 该多路径的lun wwn
    :param disks: 盘数据
    """
    if not lun_list:
        return
    for disk in disks:
        if NA == disk.multi_type and disk.lun_wwn in lun_list:
            disk.multi_version = version
            disk.multi_type = multi_type


def get_vxdmp_lun_list(context):
    """
    解析VXDMP多路径的lun
    :param context: 上下文
    :return: 数据
    """
    context["command"] = "vxdisk list"
    multi_info = get_return_from_txt(context)
    lun_list = []
    for line in multi_info:
        line = handle_vxdmp_line(line)
        items = line.split()
        # vxdmp多路径，以空白分割，获取第11个字段作为lun wwn
        if len(items) > 10:
            lun_list.append(items[10])
    return lun_list


def handle_vxdmp_line(line):
    """
    处理vxdmp多路径的行数据，便于获取lun wwn：以空白进行分割，获取第11个字段（字段间存在单空格、多空格）
    1.MediaName字段正常是不含空格的，异常数据(No Signature)需要替换为没有空格的，如Signature
    后期识别到的进行补充
    :param line: 上下文
    :return: 数据
    """
    return line.replace("(No Signature)", "Signature")


def get_hdlm_lun_list(context):
    """
    解析HDLM多路径的lun
    :param context: 上下文
    :return: 数据
    """
    context["command"] = "dlnkmgr view -lu"
    multi_info = get_return_from_txt(context)
    lun_list = []
    for line in multi_info:
        items = line.split()
        # 日立多路径，以空白分割，获取四个字段中的第一个字段，作为lun wwn
        if len(items) == 4:
            lun_list.append(items[0].lower())
    return lun_list


def get_power_lun_list(context):
    """
    解析power path多路径的lun
    :param context: 上下文
    :return: 数据
    """
    context["command"] = "powermt display dev=all"
    multi_info = get_return_from_txt(context)
    lun_list = []
    for line in multi_info:
        if "Logical device ID" in line and "=" in line:
            lun_list.append(line.split("=")[-1].lower())
    return lun_list


def get_ultra_lun_list(context):
    """
    解析ultra多路径的lun
    :param context: 上下文
    :return: 数据
    """
    multi_info = get_return_from_txt(context)
    header_line = False
    lun_list = []
    for line in multi_info:
        line = line.lower()
        if "Vlun ID".lower() in line and "Lun WWN".lower() in line:
            header_line = True
            continue
        items = re.split("\\s{2,}", line)
        if header_line and len(items) > 4:
            lun_list.append(items[3].lower())
    return lun_list


def get_windows_multipath_basic(context):
    """
    获取windows多路径基础数据
    :param context: 上下文
    :return: 数据
    """
    basic_multi = []
    disks = get_windows_physical_disks(context)
    for disk in disks:
        multi = WindowsMultiPath()
        disk_dict = get_vertical_result(disk, ":")
        multi.system_disk_name = disk_dict.get("DeviceId", "")
        multi.lun_wwn = disk_dict.get("UniqueId", "").lower()
        disk_size = disk_dict.get("Size", "") + "Bytes"
        multi.lun_size = str(round(chang_unit2_gb(disk_size), 2))
        multi.vendor = disk_dict.get("Manufacturer", "")
        multi.product = disk_dict.get("Model", "")
        basic_multi.append(multi)
    return basic_multi


def build_drive_letter(context, disks):
    """
    填充盘符数据
    :param context: 上下文
    :param disks: 盘数据
    """
    drive_letter_data = get_id_letter_data(context)
    for disk in disks:
        disk.drive_letter = drive_letter_data.get(disk.system_disk_name, NA)


def get_empty_multi_new():
    """
    构造空数据
    """
    return {"multipath_type": "", "version": "", "disk_list": [get_empty_disk_new()]}


def get_empty_disk_new():
    """
    构造空数据
    """
    return {"system_disk": "", "lun_wwn": "", "lun_size": "", "vendor": "", "product": "", "alua": "",
            "drive_letter": ""}


class WindowsMultiPath(object):
    def __init__(self):
        self.multi_type = NA
        self.multi_version = NA
        self.system_disk_name = NA
        self.lun_size = NA
        self.lun_wwn = NA
        self.vendor = NA
        self.product = NA
        self.alua_strategy = NA
        self.drive_letter = NA

    def str(self):
        return "multi_type:" + self.multi_type + "\tmulti_version:" + self.multi_version + "\tsystem_disk_name:" + \
               self.system_disk_name + "\tlun_size:" + self.lun_size + "\tlun_wwn:" + self.lun_wwn + "\tvendor:" + \
               self.vendor + "\tproduct:" + self.product + "\talua_strategy:" + self.alua_strategy + \
               "\tdrive_letter:" + self.drive_letter


def get_window_multipaths_old(context):
    """
    获取多路径，旧方案，只支持MPIO，但是不需要powershell命令
    :param context: 上下文
    :return: 数据
    """
    try:
        disks = get_windows_multipath_disks(context)
        multipath_dic = get_multipath_dic(disks)
        return get_multi_paths(multipath_dic)
    except Exception:
        context.get("Logger").error("migration error: " + str(traceback.format_exc()))
        return get_empty_multi()


def get_multi_paths(multipath_dic):
    """
    获取多路径
    :param multipath_dic: 多路径映射
    :return: 数据
    """
    result = []
    for multipath_type in multipath_dic.keys():
        one_multipath = {}
        if "Microsoft DSM" == multipath_type:
            one_multipath["multipath_type"] = "MPIO"
        else:
            one_multipath["multipath_type"] = multipath_type
        one_multipath["version"] = "NA"
        one_multipath["disk_list"] = multipath_dic.get(multipath_type, get_empty_disk_list())
        result.append(one_multipath)
    if not result:
        result.append(get_empty_multi())
    return result


def get_multipath_dic(disks):
    """
    获取disks
    :param disks: 所有盘
    :return: 数据
    """
    multipath_dic = {}
    for disk in disks:
        multipath_type = disk.get("multipath_type", "")
        if multipath_type not in multipath_dic:
            multipath_dic[multipath_type] = []
        multipath_dic[multipath_type].append(disk)
    return multipath_dic


def get_windows_multipath_disks(context):
    """
    获取disks
    :param context: 上下文
    :return: 数据
    """
    mpio_disk_id_2_system_disk = get_disk_id_2_system_disk(context)
    mpio_disk_id_2_size_vendor = get_disk_id_2_size_vendor(context)
    disks = []
    for disk_id in mpio_disk_id_2_system_disk.keys():
        one_disk = get_empty_disk()
        context["command"] = "mpclaim -s -d %s" % disk_id
        disk_info = get_return_from_txt(context)
        for line in disk_info:
            if line.strip().startswith("Controlling DSM:"):
                one_disk["multipath_type"] = line.split(":")[-1].strip()
            if line.strip().startswith("SN:"):
                one_disk["lun_wwn"] = line.split(":")[-1].strip()
            if line.strip().startswith(mpio_disk_id_2_system_disk.get(disk_id, {}).get("mpio_disk", "") + ":"):
                one_disk["alua"] = line.split(":")[-1].strip()
        sydtem_disk = mpio_disk_id_2_system_disk.get(disk_id, {}).get("system_disk", "")
        one_disk["system_disk"] = sydtem_disk
        one_disk["mpio_disk"] = mpio_disk_id_2_system_disk.get(disk_id, {}).get("mpio_disk", "")
        sydtem_disk_id = safe_get_split(sydtem_disk.split(), 1).strip()
        one_disk["lun_size"] = mpio_disk_id_2_size_vendor.get(sydtem_disk_id, {}).get("size", "")
        one_disk["vendor"] = mpio_disk_id_2_size_vendor.get(sydtem_disk_id, {}).get("vendor", "")
        disks.append(one_disk)
    return disks


def get_disk_id_2_system_disk(context):
    """
    获取disk的id与system_disk的对应关系
    :param context: 上下文
    :return: 数据
    """
    id_2_system_disk = {}
    context["command"] = "mpclaim -s -d"
    ids_info = get_return_from_txt(context)
    start_flag = False
    for line in ids_info:
        if line.strip().startswith("-----------------"):
            start_flag = True
            continue
        if start_flag and line:
            params = re.split(r"\s+", line)
            disk_id = params[1].replace("Disk", "")
            if disk_id.isdigit():
                disk_id_obj = {"mpio_disk": " ".join(params[0:2]), "system_disk": " ".join(params[2:4])}
                id_2_system_disk[disk_id] = disk_id_obj
    return id_2_system_disk


def get_disk_id_2_size_vendor(context):
    """
    获取disk的id与size、vendor的对应关系
    :param context: 上下文
    :return: 数据
    """
    system_id_2_size_vendor = {}
    context["command"] = "powershell Get-Disk"
    result_info = get_return_from_txt(context, True)
    dic_list = get_horizontal_cli_ret("".join(result_info))
    for obj_tmp in dic_list:
        disk_id_obj = {"size": chang_unit2_gb(obj_tmp.get("Total Size", "")),
                       "vendor": obj_tmp.get("Friendly Name", "")}
        system_id_2_size_vendor[obj_tmp.get("Number", "")] = disk_id_obj
    return system_id_2_size_vendor


def get_empty_multi():
    """
    构造空数据
    """
    return {"multipath_type": "", "version": "", "disk_list": get_empty_disk_list()}


def get_empty_disk_list():
    """
    构造空数据
    """
    return [get_empty_disk()]


def get_empty_disk():
    """
    构造空数据
    """
    return {"mpio_disk": "", "system_disk": "", "lun_wwn": "", "alua": "", "lun_size": "", "vendor": ""}
