# --*-- coding:utf-8 --*--
# Copyright (c) Huawei Technologies Co., Ltd. 2019-2021. All rights reserved.
import json
import traceback
from common.contentParse import get_return_from_txt
from common.migration_constants import HOST_TYPE_WINDOWS
from hosts.common.migra_summary_util import get_multi_type_version
from hosts.common.windows_parse_utils import get_os_type_version, check_not_found_command, get_vxdmp_multi_type_version
from hosts.common.windows_parse_utils import get_horizontal_cli_ret
from hosts.parse_util import get_key_values_from_content_lines


def execute(context):
    """
    执行入口
    :param context: 上下文
    :return: 解析后数据
    """
    result = get_result(context)
    ret_map = context.get("map")
    ret_map.put("data", json.dumps(result))
    return context


def get_result(context):
    """
    获取数据
    :param context: 上下文
    :return: 数据
    """
    result = {"host_ip": context.get("ip"), "host_name": get_windows_host_name(context)}
    os_type, os_version = get_os_type_version(context)
    result["os_version"] = os_version
    result["hba_type"] = get_windows_hba_type(context)
    result["hba_wwn_list"] = get_windows_hba_str(context)
    multi_type, multi_version = get_multi_type_version_windows(context)
    result["multi_type"] = multi_type
    result["multi_version"] = multi_version
    result["cluster_name"] = get_cluster_data(context)
    result["hba_detail"] = get_hba_detail(context)
    # 当前只有OracleRac集群，版本就是oracle版本
    result["cluster_version"] = ""
    result["cluster_ips"] = []
    result["lvm_type"] = ""
    result["db_name"] = ""
    result["db_version"] = ""
    result["disk_use"] = ""
    result["real_ip"] = ""
    return result


def get_hba_detail(context):
    # 这里需要解析多个，二条命令的结果通过WWN关联
    context["command"] = "winrm e wmi/root/wmi/MSFC_FibrePortHBAAttributes"
    port_hba_attribute = get_return_from_txt(context)
    start_bid = "MSFC_FibrePortHBAAttributes"
    key_words = ('NodeWWN', 'PortSpeed', 'PortWWN')
    delimiter = ' ='
    fibre_attribute = get_key_values_from_content_lines(port_hba_attribute, start_bid, key_words, delimiter)
    context["command"] = "winrm e wmi/root/wmi/MSFC_FCAdapterHBAAttributes"
    adapter_hba_attribute = get_return_from_txt(context)
    start_bid = "MSFC_FCAdapterHBAAttributes"
    key_words = ('NodeWWN', 'Manufacturer', 'Model')
    adapter_attribute = get_key_values_from_content_lines(adapter_hba_attribute, start_bid, key_words, delimiter)
    result = []
    for fibre in fibre_attribute:
        for adapter in adapter_attribute:
            if fibre['NodeWWN'] == adapter["NodeWWN"]:
                speed = fibre['PortSpeed'] if fibre['PortSpeed'] else fibre['PortSupportedSpeed']
                temp = ["ID=" + dec_to_hex_transform(fibre['NodeWWN'], context.get("Logger")),
                        "Vendor=" + adapter['Manufacturer'], "Type=" + adapter['Model'],
                        "Speed=" + speed + "Gb", "Location="]
                result.append(";".join(temp).strip())
    return "\n".join(result).strip()


def dec_to_hex_transform(string_num, logger):
    """
    十进制转16进制
    :param string_num: 十进制
    :return: 16进制
    """
    one_hba = []
    for item in string_num.split(","):
        one_hba.append(dec_to_hex(item, logger))
    return "".join(one_hba).strip()


def get_anonymous_host_name(context):
    """
    获取windows主机名(匿名时)
    :param context: 上下文
    :return: 数据
    """
    context["command"] = "hostname"
    host_name_info = get_return_from_txt(context)
    for line in host_name_info:
        line = line.strip()
        if line:
            return line
    context.get("Logger").error("can not get windows host name.")
    return ""


def get_windows_host_name(context):
    """
    获取windows主机名
    :param context: 上下文
    :return: 数据
    """
    context["command"] = "systeminfo"
    host_name_info = get_return_from_txt(context)
    for line in host_name_info:
        if line.strip().startswith("Host Name:") or line.strip().startswith("主机名:"):
            return line.split(":")[-1].strip()
    return get_anonymous_host_name(context)


def get_windows_hba_str(context):
    """
    获取windows启动器
    :param context: 上下文
    :return: 数据
    """
    context["command"] = "powershell Get-InitiatorPort"
    hba_info = get_return_from_txt(context, True)
    hba_dics = get_horizontal_cli_ret("".join(hba_info))
    hba_list = []
    for hba_dic in hba_dics:
        port_address = hba_dic.get("PortAddress", "")
        if port_address == "ISCSI ANY PORT":
            hba_list.append(hba_dic.get("NodeAddress", ""))
        else:
            hba_list.append(port_address)
    hba_initiators = get_initiator_old(context)
    hba_list.extend(filter(lambda initiator: initiator not in hba_list, hba_initiators))
    npiv_initiators = get_npiv_initiators(context)
    hba_list.extend(filter(lambda initiator: initiator not in hba_list, npiv_initiators))
    return "\n".join(hba_list)


def get_windows_hba_type(context):
    """
    获取windows启动器的类型
    :param context: 上下文
    :return: 数据
    """
    models = []
    instances = get_instances(context)
    instance_model_dicts = get_instance_model_dict(context)
    for instance in instances:
        model = instance_model_dicts.get(instance, "")
        if model and model not in models:
            models.append(model)
    return "\n".join(models)


def get_instance_model_dict(context):
    """
    获取instance与model的映射关系
    :param context: 上下文
    :return: 数据
    """
    context["command"] = "powershell Get-WmiObject -Class MSFC_FCAdapterHBAAttributes -Namespace root\\WMI"
    instance_model_dict = get_instance_model_dict_common(context, "__RELPATH", ":")
    if not instance_model_dict:
        context["command"] = "winrm e wmi/root/wmi/MSFC_FCAdapterHBAAttributes"
        return get_instance_model_dict_common(context, "InstanceName", "=")
    return instance_model_dict


def get_instance_model_dict_common(context, instance_key, model_split):
    """
    获取instance与model的映射关系(方法提取)
    :param context: 上下文
    :param instance_key: instance标志
    :param model_split: model的分隔符
    :return: 数据
    """
    instance_model_dict = {}
    data_info = get_return_from_txt(context)
    instance_name = ""
    model = ""
    for line in data_info:
        if line.startswith(instance_key) and "=" in line:
            if instance_name and model:
                instance_model_dict[instance_name] = model
            instance_name = line.split("=")[1].strip("\"").strip().replace("\\", "")
            model = ""
        if line.startswith("Model ") and model_split in line:
            model = line.split(model_split)[1]
    if instance_name and model:
        instance_model_dict[instance_name] = model
    return instance_model_dict


def get_instances(context):
    """
    获取windows启动器的instance名字
    :param context: 上下文
    :return: 数据
    """
    context["command"] = "powershell Get-InitiatorPort"
    hba_info = get_return_from_txt(context, True)
    dict_list = get_horizontal_cli_ret("".join(hba_info))
    instance_list = []
    for hba_dic in dict_list:
        instance_list.append(hba_dic.get("InstanceName", "").strip().replace("\\", ""))
    if not instance_list:
        instance_list = get_instances_old(context)
    return instance_list


def get_instances_old(context):
    """
    获取windows启动器的instance名字(2012之前的版本)
    :param context: 上下文
    :return: 数据
    """
    context["command"] = "winrm e wmi/root/wmi/MSFC_FibrePortHBAAttributes"
    hba_info = get_return_from_txt(context)
    instance_list = []
    for line in hba_info:
        if line.strip().startswith("InstanceName") and "=" in line:
            instance_list.append(line.split("=")[1].strip().replace("\\", ""))
    return instance_list


def get_mpio_multi_type_version(context):
    """
    获取windows主机的vxdmp多路径类型、版本
    :param context: 上下文
    :return: 数据
    """
    context["command"] = "mpclaim -s -d"
    data_info = get_return_from_txt(context)
    for line in data_info:
        if "MPIO Disk" in line:
            return "MPIO", ""
    return "", ""


def get_multi_type_version_windows(context):
    """
    获取windows主机的多路径类型、版本
    :param context: 上下文
    :return: 数据
    """
    multi_type, multi_version = get_multi_type_version(context, HOST_TYPE_WINDOWS)
    if multi_type:
        return multi_type, multi_version

    multi_type, multi_version = get_vxdmp_multi_type_version(context)
    if multi_type:
        return multi_type, multi_version

    multi_type, multi_version = get_mpio_multi_type_version(context)
    if multi_type:
        return multi_type, multi_version
    return "", ""


def get_cluster_data(context):
    """
    获取windows的集群信息（当前只判断WSFC集群）
    :param context: 上下文
    :return: 数据
    """
    context["command"] = "powershell get-cluster"
    data_info = get_return_from_txt(context)
    for line in data_info:
        if check_not_found_command(line):
            return ""
    data_str = "".join(data_info)
    if "Name" in data_str and "----" in data_str:
        return "WSFC"
    return ""


def get_initiator_old(context):
    """
    获取windows启动器(2012之前的版本)
    :param context: 上下文
    :return: 数据
    """
    context["command"] = "winrm e wmi/root/wmi/MSFC_FibrePortHBAAttributes"
    return parse_port_wwn_list(context)


def get_npiv_initiators(context):
    """
    获取NPIV相关启动器
    :param context: 上下文
    :return: 启动器数据
    """
    context["command"] = "winrm e wmi/root/wmi/MSFC_FibrePortNPIVAttributes"
    return parse_port_wwn_list(context)


def parse_port_wwn_list(context):
    logger = context.get("Logger")
    hba_info = get_return_from_txt(context)
    hba_list = []
    for line in hba_info:
        if line.strip().startswith("PortWWN") and "=" in line:
            hba_wwn_dec = line.split("=")[1]
            one_hba = []
            for item in hba_wwn_dec.split(","):
                one_hba.append(dec_to_hex(item, logger))
            hba_list.append("".join(one_hba))
    if not hba_list:
        hba_list.append("")
    return hba_list


def dec_to_hex(string_num, logger):
    """
    十进制转16进制
    :param string_num: 十进制
    :param logger: 日志
    :return: 16进制
    """
    try:
        hex_temp = str(hex(int(string_num.strip())))[2:]
        return hex_temp if len(hex_temp) == 2 else "0" + hex_temp
    except Exception:
        logger.error("parse error: %s %s" % (string_num, str(traceback.format_exc())))
        return string_num
