#  coding=UTF-8
#  Copyright (c) Huawei Technologies Co., Ltd. 2019-2019. All rights reserved.

"""
@version: SmartKit V200R006C00
@time: 2019/10/28
@file: windows_lun_wwn_info.py
@function:
@modify:
"""

import sys
import os
import re
import traceback

current_absolute_path = os.path.dirname(os.path.abspath(__file__))
path = os.path.join(current_absolute_path, "..\\..")
sys.path.append(path)
from common.contentParse import create_result_info
from common import constants
from common.util import log, HostResultInfo, HostLunInfo, getUltrapathIntVer, \
    create_lun_info, get_host_wwn_info

LAUNCHER = "'powershell Get-InitiatorPort' or " \
           "'winrm e wmi/root/wmi/MSFC_FibrePortHBAAttributes' or " \
           "'powershell Get-WmiObject -Namespace root\\wmi -Class " \
           "MSiSCSIInitiator_MethodClass'"
STANDARD_WWN_LEN = 32


def get_result_info(context):
    """
    :param context:
    :return:
    """
    host_result_info = HostResultInfo()
    host_lun_info = HostLunInfo()

    cli = context.get("SSH")

    host_lun_info.set_host_id(cli.getIp())
    cli_rets = ''

    try:
        # from launcher get initiator info
        _, cli_rets, initiator_wwn = get_host_wwn_info(context)
        if initiator_wwn:
            host_lun_info.set_initiator(initiator_wwn)
            host_result_info.set_hosts(host_lun_info)
        else:
            log.info(context, "Cannot find host initiator info, set pass.")
            create_result_info(context, "True", cli_rets, "",
                               host_result_info)
            return

        is_ultrapath_qry_succ, ultrapath_version, cli_ret = \
            qry_ultrapath_version(context)
        cli_rets += ("\r\n" + cli_ret)

        if not is_ultrapath_qry_succ:
            log.error(context, 'Query ultrapath version failed.')
            fail_cmd = constants.QRY_ULTRAPATH_CMD
            create_result_info(context, "False", cli_rets, fail_cmd,
                               host_result_info)
            return

        if ultrapath_version:
            is_qry_succ, fail_cmd, cli_ret, lun_wwn = \
                query_ultrapath_lun_wwn_info(context, ultrapath_version)
            cli_rets += ("\r\n" + cli_ret)
            if is_qry_succ is None:
                create_result_info(context, "True", cli_rets, "",
                                   host_result_info)
                log.info(context, 'ultrapath pass, return.')
                return
            elif is_qry_succ:
                create_lun_info(host_lun_info, lun_wwn)
                host_result_info.set_hosts(host_lun_info)
                log.info(context, 'ultrapath need to check.')
            else:
                create_result_info(context, "False", cli_rets, fail_cmd,
                                   host_result_info)
                log.info(context, 'ultrapath not pass, return.')
                return

        is_qry_succ, fail_cmd, cli_ret, disk_wwn_dict = \
            qry_mpio_lun_wwn_info(context)
        cli_rets += ("\r\n" + cli_ret)
        if is_qry_succ:
            create_lun_info(host_lun_info, disk_wwn_dict)
            host_result_info.set_hosts(host_lun_info)
            create_result_info(context, "nocheck", cli_rets, "",
                               host_result_info)
        elif not is_qry_succ and disk_wwn_dict:
            host_result_info.set_err_msg(
                "eval.host.evaluation.item.lunMapChangeCheck.no.patch",
                ";".join(disk_wwn_dict.keys()))
            create_result_info(context, "False", cli_rets, "",
                               host_result_info)
        else:
            create_result_info(context, "False", cli_rets, fail_cmd,
                               host_result_info)
    except Exception as e:
        log.info(context, "get host result info exception:%s:%s" % (
            str(e), str(traceback.format_exc())))
        create_result_info(context, "False", cli_rets, '',
                           host_result_info)


def query_ultrapath_lun_wwn_info(context, ultrapath_version):
    """
    :param context:
    :param ultrapath_version:
    :return:
    """
    tup_ver = getUltrapathIntVer(context, ultrapath_version)
    middle_tup_ver = getUltrapathIntVer(context,
                                        constants.LINUX_MIDDLE_ULTRAPATH_VER)
    consistency_tup_ver = \
        getUltrapathIntVer(context, constants.LINUX_CONSISTENCY_ULTRAPATH_VER)

    if tup_ver <= middle_tup_ver:
        return query_higher_version_info(
            context, False)
    elif tup_ver < consistency_tup_ver:
        return query_higher_version_info(
            context, True)
    else:
        log.info(context, 'the ultrapath version greater or equal to '
                          '8.06.034, set the result pass.')
        return None, "", "", {}


def query_higher_version_info(context, is_high_version):
    """
    :param context:
    :param is_high_version:
    :return:
    """
    cli = context.get("SSH")
    lun_wwn = {}
    cli_ret = ''
    vlu_id_cmd = "upadm show vlun type=all" if is_high_version \
        else "upadm show vlun"
    try:
        cli_ret = cli.executeCmdTimeoutLcalWmic(vlu_id_cmd,
                                                constants.HOST_CMD_TIMEOUT)

        is_data_start = False
        for line in cli_ret.splitlines():
            if not is_data_start and re.search('Vlun ID', line,
                                               re.I) and re.search('WWN', line,
                                                                   re.I):
                is_data_start = True
                continue

            fields = line.split()
            if is_data_start and len(fields) >= 3 and fields[0].isdigit():
                lun_wwn[fields[3].strip()] = fields[0].strip()
        log.error(context, 'Query VLUN IDs are:' + str(lun_wwn))
        return True, '', cli_ret, lun_wwn
    except Exception as e:
        log.error(context, 'Query VLUN IDs exception:' + str(e))
        log.error(context,
                  'Query VLUN IDs exception:' + str(traceback.format_exc()))
        return False, vlu_id_cmd, cli_ret, {}


def get_all_disk_numbers(context):
    """
    :param context:
    :return:
    """
    cmd = "mpclaim -s -d"
    cli = context.get("SSH")
    disk_number_list = []
    cli_ret = ''
    try:
        cli_ret = cli.executeCmdTimeoutLcalWmic(cmd,
                                                constants.HOST_CMD_TIMEOUT)
        for line in cli_ret.splitlines():
            if re.search("^MPIO Disk[0-9]+", line.strip(), re.I):
                re_result = re.search("^MPIO Disk[0-9]+", line.strip(),
                                      re.I).group()
                disk_number_list.append(re_result[9:])
        log.info(context, "current windows's disk number lists: %s" % ",".join(
            disk_number_list))
        return True, '', cli_ret, disk_number_list
    except Exception as e:
        log.error(context, 'Query disk IDs exception:' + str(e))
        return False, cmd, cli_ret, []


def qry_mpio_lun_wwn_info(context):
    """
    :param context:
    :return:
    """
    is_qry_succ, fail_cmd, cli_rets, disk_letters = get_all_disk_numbers(
        context)
    if not is_qry_succ:
        log.error(context, 'Query array managed by ultrapath failed.')
        return False, fail_cmd, cli_rets, {}

    failed_disk_cmds = []  # 查询失败的硬盘结果
    wrong_sn_disks = {}  # 磁盘SN不正确的磁盘
    disk_results = {}  # 查询成功的磁盘
    try:
        for disk_number in disk_letters:
            is_qry_succ, fail_cmd, cli_ret, disk_wwn = get_specify_disk_infos(
                context, disk_number)
            cli_rets += ("\r\n" + cli_ret)

            if not is_qry_succ or not disk_wwn:
                log.error(context, "current disk query disk info failed ")
                failed_disk_cmds.append(fail_cmd)
            elif len(disk_wwn) < STANDARD_WWN_LEN:
                log.error(context, "current disk's SN is not standard length")
                wrong_sn_disks[disk_wwn] = disk_number
            else:
                disk_results[disk_wwn] = disk_number
        if failed_disk_cmds:
            log.error(context, 'failed disk cmds are:' + str(failed_disk_cmds))
            return False, ','.join(failed_disk_cmds), cli_rets, {}
        elif wrong_sn_disks:
            log.error(context, 'wrong lun wwns are:' + str(wrong_sn_disks))
            return False, "", cli_rets, wrong_sn_disks
        else:
            log.error(context, 'lun wwns are:' + str(disk_results))
            return True, "", cli_rets, disk_results

    except Exception as e:
        log.error(context, 'Query array IDs exception:' + str(e))
        log.error(context,
                  'Query array IDs exception:' + str(traceback.format_exc()))
        return False, fail_cmd, cli_rets, {}


def get_specify_disk_infos(context, disk_number):
    """
    :param context:
    :param disk_number:
    :return:
    """
    cli = context.get("SSH")
    cmd = "mpclaim -s -d %s " % disk_number
    cli_ret = ""
    try:
        cli_ret = cli.executeCmdTimeoutLcalWmic(cmd,
                                                constants.HOST_CMD_TIMEOUT)
        for line in cli_ret.splitlines():
            if re.search(r"^sn(\s)*:(\s)*[\S]+", line.strip(), re.I):
                sre_match = re.search(r"^sn(\s)*:(\s)*[\S]+", line.strip(),
                                      re.I).group()
                sn_num = sre_match.split(":")[1].strip()
                log.info(context, "current windows disk %s 's sn: %s" % (
                    disk_number, sn_num))

                return True, '', cli_ret, sn_num
        return False, cmd, cli_ret, ""
    except Exception as e:
        log.error(context, 'Query disk IDs exception:' + str(e))
        log.error(context,
                  'Query disk IDs exception:' + str(traceback.format_exc()))
        return False, cmd, cli_ret, ""


def qry_ultrapath_version(context):
    '''
    @summary: Query the HUAWEI ultrapath version.
    @param context: Python execution context.
    @return (isQrySucc, ultrapathVersion, cliRet)
    '''
    cliRet = ''
    cli = context.get("SSH")
    if not cli:
        return False, '', ''

    try:
        cliRet = cli.executeCmdTimeoutLcalWmic(constants.QRY_ULTRAPATH_CMD,
                                               constants.HOST_CMD_TIMEOUT)
        verPattern = 'Software Version|UltraPath for Linux|Driver   Version'
        if re.search(verPattern, cliRet, re.I):
            for line in cliRet.splitlines():
                if re.search(verPattern, line, re.I):
                    return True, line.split(':')[1].strip(), cliRet

        return True, '', cliRet
    except Exception as e:
        log.error(context, 'Query ultrapath version exception:' + str(e))
        log.error(context,
                  'Query ultrapath version exception:' +
                  traceback.format_exc())
        return False, '', cliRet
