#  coding=UTF-8
#  Copyright (c) Huawei Technologies Co., Ltd. 2019-2019. All rights reserved.

"""
@time: 2020/06/17
@file: linux_NFS_share_hangs_kernel_version_check.py
@function:
"""
import json
import sys
import os

path = os.path.dirname(os.path.abspath(__file__))
path = os.path.join(path, os.path.pardir)
sys.path.append(path)
from common import util, contextUtil, constants, contentParse, cliUtil
from common.constants import EvaluStatus

RISK_HOST_VERSION_TUPLE = ("Red Hat 7.6", "CentOS 7.6")
RISK_KERNEL_VERSION_DICT = {
    "Min": "3.10.0-957.el7", "Max": "3.10.0-957.5.1.el7"
}
CHECK_RESULT_KEY_TO_EVAL = "NFSMatchKernelVersionCheck"
CHECK_RESULT_FLAG = "checkResultFlag"
CHECK_ORIGINAL_INFO = "checkOriginalInfo"
HOST_MOUNT_INFO = "hostMountInfo"


def execute(context):
    NFSShareHangsKernelVersionCheck(context).exec_check()


class NFSShareHangsKernelVersionCheck(object):

    def __init__(self, context):
        self._context = context
        self._host_version = contextUtil.get_dev_type(context)
        self._cli = context.get("SSH")
        self._ret_map = contextUtil.get_ret_map(context)
        self._logger = contextUtil.get_logger(context)
        self._lang = contextUtil.get_lang(context)
        self._cli_rets = list()
        self._infoGrab_err_msg = ""
        self._mount_info = ""

    def exec_check(self):
        try:
            util.updateItemProgress(self._context, constants.PROG30)
            # Step 1 版本判断
            if not self._is_risk_host_version():
                self._set_check_result(EvaluStatus.PASS)
                return
            # Step 2 内核版本判断
            if not self._is_risk_kernel_version():
                self._set_check_result(EvaluStatus.PASS)
                return
            util.updateItemProgress(self._context, constants.PROG50)
            # 获取主机挂载信息
            self._query_host_mount_info()
            # 继续到兼容性评估中做检查
            self._set_check_result(EvaluStatus.CONTINUE_CHECK)
        except Exception as exception:
            self._logger.error("[NFS share hangs] " + str(exception))
            self._set_check_result(EvaluStatus.NO_CHECK)
        finally:
            util.updateItemProgress(self._context, constants.PROG95)
            self._ret_map.put("err_msg", self._infoGrab_err_msg)

    def _is_risk_host_version(self):
        """
        是否是风险的设备版本
        :return:
        """
        self._query_host_version_to_cli_ret()
        self._logger.info("NFS host ver: {}".format(self._host_version))

        for risk_version in RISK_HOST_VERSION_TUPLE:
            if risk_version.lower() in self._host_version.lower():
                return True
        return False

    def _is_risk_kernel_version(self):
        host_version = self._get_host_kernel_version()

        host_version_section_list = self._kernel_version_to_section_list(
            host_version)
        min_version_section_list = self._kernel_version_to_section_list(
            RISK_KERNEL_VERSION_DICT.get("Min"))
        max_version_section_list = self._kernel_version_to_section_list(
            RISK_KERNEL_VERSION_DICT.get("Max"))

        max_length = max(len(min_version_section_list),
                         len(max_version_section_list),
                         len(host_version_section_list))

        for index in range(max_length):
            # 缺少位补零
            if index == len(host_version_section_list):
                host_version_section_list.append(0)
            if index == len(min_version_section_list):
                min_version_section_list.append(0)
            if index == len(max_version_section_list):
                max_version_section_list.append(0)
            # 版本范围外无风险
            if min_version_section_list[index] > host_version_section_list[
                        index] or host_version_section_list[index] > \
                    max_version_section_list[index]:
                return False
            # 版本范围内有风险
            if min_version_section_list[index] <= host_version_section_list[
                        index] and host_version_section_list[index] < \
                    max_version_section_list[index]:
                return True
        return False

    def _query_host_version_to_cli_ret(self):
        cmd = "cat /etc/*release"
        flag, cli_ret, infoGrab_err_msg = \
            cliUtil.exec_cmd_has_log(self._cli, cmd, self._lang)

        self._ret_map.put("cmd_display_host_version", cli_ret)
        self._cli_rets.append(cli_ret)
        self._infoGrab_err_msg += infoGrab_err_msg

    def _get_host_kernel_version(self):
        cmd = "uname -r"
        flag, cli_ret, infoGrab_err_msg = \
            cliUtil.exec_cmd_has_log(self._cli, cmd, self._lang)
        self._ret_map.put("cmd_display_kernel_version", cli_ret)
        self._cli_rets.append(cli_ret)
        self._infoGrab_err_msg += infoGrab_err_msg
        if not flag:
            raise Exception("get host kernel version failed.")
        cli_ret_version_line = 1
        kernel_version = cli_ret.splitlines()[cli_ret_version_line].strip()
        self._logger.info("NFS host kernel ver: {}".format(kernel_version))
        return kernel_version

    def _kernel_version_to_section_list(self, version):
        removed_end_el7_version = version[:version.index(".el7")]
        replaced_before_patch_cross_version = removed_end_el7_version.replace(
            "-", ".")
        version_section_list = list()
        for section in replaced_before_patch_cross_version.split("."):
            version_section_list.append(int(section))
        return version_section_list

    def _query_host_mount_info(self):
        cmd = "mount"
        flag, cli_ret, infoGrab_err_msg = \
            cliUtil.exec_cmd_has_log(self._cli, cmd, self._lang)
        self._ret_map.put("cmd_display_mount_info", cli_ret)
        self._infoGrab_err_msg += infoGrab_err_msg
        if not flag:
            raise Exception("get host mount info failed.")
        self._mount_info = cli_ret

    def get_cli_ret(self):
        if len(self._cli_rets) == 0:
            return "no information."
        return '\n'.join(self._cli_rets)

    def _set_check_result(self, check_flag):
        """
        设置检查结果给兼容性评估
        :param check_flag: 检查结果
        """
        check_result = {CHECK_RESULT_FLAG: check_flag,
                        CHECK_ORIGINAL_INFO: self.get_cli_ret(),
                        HOST_MOUNT_INFO: self._mount_info}

        contentParse.setCmdRet4Eval(self._ret_map, json.dumps(check_result),
                                    CHECK_RESULT_KEY_TO_EVAL)
