# -*- coding: UTF-8 -*-
import re
import sys
import traceback
import os
path = os.path.dirname(os.path.abspath(__file__))
path = os.path.join(path, "..\\..")
sys.path.append(path)
from common.util import qryUltrapathVersion, getUltrapathIntVer, stripOriginalStr, genEvalItemObj
from common.contentParse import singleCmdExecute, setCmdRet4Eval
from linux_multipath_bug_non_lvm_handler import makeChain
from com.huawei.ism.tool.infograb.context import EvalResultEnum
from common import constants
from common import util

#version constant
MULTIPATH_VERSION_TUPLE = (8, 6, 63)
KERNEL_VERSION_TUPLE = (3, 8)
LOWER_BOUND_VERSION_TUPLE = (6, 1, 22)
MULTIPATH_MIDDLE_VERSION = (8, 3, 21)

RESULT_2TLUN_KEY = "cmd_display_kernel_risk_check:"
RESULT_2TLUN = "Result: True\n"
RESULT_2TLUN_UNABLE_EVAL = "Result: NA\n"
multipathVersion = None
ITEMKEY = "LinuxKernelRiskCheck"
DISPLAY_KEY = "kernel_risk_check"
#number constant
TB_2 = 2.0
DOUBLE_2048 = 2048.0
NEED_JUDGE_LUNS = []
INT_1000 = 1000
DOUBLE_1024 = 1024.0
CONVERSION_FACTOR = 1000.0 / 1024.0
cliRets = []
LINE_BREAK = "\n"
ALL_EXIST = "eval.host.multipath.reduntpath.self&buildin.allexists.2TLUN"
COMMAND_FAILED = "eval.host.multipath.buglist.failed.without.command"

def executeBugJudge(context):
    '''
    @summary: bug judge execute method
    @param context: execute context 
    '''
    global LOG
    LOG = context.get("Logger")
    cmdDisplay = context.get("ret_map")
    try:
        util.updateItemProgress(context, constants.PROG25)
        executeStatus, executeResult = hasUltrapath(context)
        util.updateItemProgress(context, constants.PROG85)
        if not executeStatus:
            LOG.info("***[query multipath version is failure]***")
            handleEvalResult(cmdDisplay, RESULT_2TLUN_UNABLE_EVAL + getSuggestInfo(), EvalResultEnum.UNABLE_EVAL, 'linux.multipath.bug.judgement.query.multipath.failed')
            return False
        if not executeResult:
            handleEvalResult(cmdDisplay, RESULT_2TLUN + getSuggestInfo(), EvalResultEnum.PASSED, '')
            return True
        executeStatus, executeResult = executeCondtionsJudge(context)
        if not executeStatus:
            LOG.info("***[query kernel version is failure]***")
            handleEvalResult(cmdDisplay, RESULT_2TLUN_UNABLE_EVAL + getSuggestInfo(), EvalResultEnum.UNABLE_EVAL, 'linux.multipath.bug.judgement.query.kernel.failed')
            return False
        if not executeResult:
            LOG.info("***[not to meet the 2TLUN judge execute conditions]***")
            handleEvalResult(cmdDisplay, RESULT_2TLUN + getSuggestInfo(), EvalResultEnum.PASSED, '')
            return True
        executeStatus, executeResult = judgeLunCapacity(context)
        if not executeStatus:
            LOG.info("***[query the luns managed by multipath is failure]***")
            handleEvalResult(cmdDisplay, RESULT_2TLUN_UNABLE_EVAL + getSuggestInfo(), EvalResultEnum.UNABLE_EVAL, 'eval.host.multipath.reduntpath.qry.array.id.empty')
            return False
        if not executeResult:
            handleEvalResult(cmdDisplay, RESULT_2TLUN + getSuggestInfo(), EvalResultEnum.PASSED, '')
            return True

        # 新增加判断是否为自研自带共存
        is_qry_ult_succ, ultrapath_version, _ = util.qryUltrapathVersion(
            context)
        is_qry_mul_succ, is_multipathd_running, pathd_cli_ret = \
            util.qryMultipathdStatus(
                context)
        cliRets.append(pathd_cli_ret)
        if not is_qry_ult_succ or not is_qry_mul_succ \
                or 'may require superuser privileges' in pathd_cli_ret:
            LOG.info("query info is failure.")
            handleEvalResult(cmdDisplay,
                             RESULT_2TLUN_UNABLE_EVAL + getSuggestInfo(),
                             EvalResultEnum.FAILED,
                             COMMAND_FAILED)
            return False

        _, port_numbers, port_cli_ret, _, _ = util.getTargetPortNumByMultipath(
            context,
            True)
        cliRets.append(port_cli_ret)
        if ultrapath_version and is_multipathd_running \
                and "UPTYPE" in port_numbers:
            LOG.info("both ultrapath and multipath exist.")
            handleEvalResult(cmdDisplay,
                             RESULT_2TLUN_UNABLE_EVAL + getSuggestInfo(),
                             EvalResultEnum.FAILED,
                             ALL_EXIST)
            return False

        lunOver2TJudement = LunOver2TJudgement(context, NEED_JUDGE_LUNS, LOG)
        lunOver2TJudement.checkPartition()
        evaluteResult, displayInfo = lunOver2TJudement.obtainResultAndDisplayInfo()
        LOG.info("2TLUN_Result: %s" % str(displayInfo))
        if evaluteResult:
            handleEvalResult(cmdDisplay, RESULT_2TLUN + getSuggestInfo(), EvalResultEnum.PASSED, '')
            return True
        else:
            handleEvalResult(cmdDisplay, displayInfo, EvalResultEnum.FAILED, 'linux.multipath.bug.judgement.notpass')
            return False
    except:
        msg = traceback.format_exc()
        LOG.error("executeBugJudge ==> " + msg)
        return False

def handleEvalResult(retMap, dispalyResult, evalResult, errorMessageKey):
    '''
    @summary: handle evaluate result
    '''
    itemEvalResult = genEvalItemObj(ITEMKEY, evalResult, LINE_BREAK.join(cliRets), errorMessageKey)
    #for evaluating
    setCmdRet4Eval(retMap, itemEvalResult.transferItemEvalResult2Json(), ITEMKEY)
    #for display
    setCmdRet4Eval(retMap, dispalyResult, DISPLAY_KEY)

def hasUltrapath(context):
    '''
    @summary: judge whether or not has ultrapath
    @param context: execute context 
    @return: (executeStatus, executeResult)
    '''
    try:
        versionStr = qryUltrapathVersion(context)
        util.updateItemProgress(context, constants.PROG45)
        queryResult = singleCmdExecute(context, 'upadmin show version', 'multipath_version', True)
        cliRets.append(queryResult[-1])
        if versionStr[1]:
            global multipathVersion
            util.updateItemProgress(context, constants.PROG65)
            multipathVersion = getUltrapathIntVer(context, versionStr[1])
            LOG.info("****[linux multipath version is %s]****" % str(multipathVersion))
            return True, True
        if not versionStr[0]:
            LOG.info("****[query ultra path fail in judging 2t Lun]***")
            return False, False
        return True, False
    except:
        msg = traceback.format_exc()
        LOG.error("hasUltrapath ==> " + msg)
        return False, False

def executeCondtionsJudge(context):
    '''
    @summary: version < 8.06.063 and kernel version >= 3.8  to meet the conditions
    @param context: execute context 
    @return: (executeStatus, executeResult: 'False' can not execute and vice versa) 
    '''
    try:
        kernelTuple = getLinuxKernelVersion(context)
        LOG.info("****[linux version is: %d, %d]****" % kernelTuple)
        if (0, 0) == kernelTuple:
            return False, False
        if multipathVersion < MULTIPATH_VERSION_TUPLE  and multipathVersion >= LOWER_BOUND_VERSION_TUPLE and kernelTuple >= KERNEL_VERSION_TUPLE:
            return True, True
        return True, False
    except:
        msg = traceback.format_exc()
        LOG.error("executeCondtionsJudge ==> " + msg)
        return False, False

def getLinuxKernelVersion(context):
    '''
    @summary: get kernel version
    @param context: execute context
    @return: kernel version tuple, such as (3, 8)
    '''
    try:
        queryResult = singleCmdExecute(context, 'uname -a', 'kernel_version', True)
        cliRets.append(queryResult[1])
        if not queryResult[0]:
            return (0, 0)
        kernelLine = list(filter(lambda line : 'linux' in line.lower() and 'uname -a' not in line.lower(), stripOriginalStr(queryResult[1])))
        if not kernelLine:
            return (0, 0)
        LOG.info("***[kernelLine is: %s]***" % kernelLine[0])
        m = re.search(r'(\d+\.\d+)', kernelLine[0])
        if m:
            return tuple(map(int, m.group(0).split('.')))
        return (0, 0)
    except:
        msg = traceback.format_exc()
        LOG.error("getLinuxKernelVersion ==> " + msg)
        return (0, 0)

def judgeLunCapacity(context):
    '''
    @summary: judge lun capacity whether or not larger than 2TB
    @return: (executeStatus, executeResult)
    '''
    try:
        queryResult = ''
        if multipathVersion > MULTIPATH_MIDDLE_VERSION:
            queryResult = singleCmdExecute(context, 'upadmin show vlun type=all', 'query_lun_list', True)
        else:
            queryResult = singleCmdExecute(context, 'upadmin show vlun', 'query_lun_list', True)
        if not queryResult[0]:
            #命令查询失败
            return False, False
        cliRets.append(queryResult[1])
        LOG.info("***[upadmin show vlun or type=all --> ]***" + queryResult[1])
        if re.search('Vlun ID.*Disk.*Name.*Lun WWN.*Status.*Capacity', queryResult[1], re.I):
            queryStr = re.split('Vlun ID.*Disk.*Name.*Lun WWN.*Status.*Capacity.*', queryResult[1], re.I)
            lines = list(filter(lambda item : len(re.split('\s+', item)) > 5, stripOriginalStr(queryStr[-1])))
            LOG.info("***[filter upadmin show vlun or type=all --> {}]***"
                     .format(len(lines)))
            for line in lines:
                lunDataObj = handleLunInfo(line)
                if lunDataObj:
                    NEED_JUDGE_LUNS.append(lunDataObj)
        LOG.info("***[need judge vluns list size is: %d]***" % len(NEED_JUDGE_LUNS))
        if NEED_JUDGE_LUNS:
            return True, True
        return True, False
    except:
        msg = traceback.format_exc()
        LOG.error("judgeLunCapacity ==> " + msg)
        return False, False

def handleLunInfo(lunInfo):
    '''
    @summary: if capacity is large than 2 TB, return disk name
    '''
    infoArray = re.split('\s+', lunInfo)
    lunData = LunInfo()
    lunData.setLUNName(infoArray[1])
    lunData.setLUNWWN(infoArray[3])
    lunData.setLUNSize(infoArray[5])
    if conversionUnit2GB(infoArray[5]) > DOUBLE_2048:
        return lunData
    return None

def conversionUnit2GB(strUnit):
    '''
    @summary: conversion unit 2 gb
    @return: the number in gb
    '''
    try:
        if not strUnit:
            return 0.0
        if strUnit.lower().endswith('kb'):
            return convert2Float(strUnit[:-2]) / (DOUBLE_1024 ** 2)
        elif strUnit.lower().endswith('mb'):
            return convert2Float(strUnit[:-2]) / DOUBLE_1024
        elif strUnit.lower().endswith('gb'):
            return convert2Float(strUnit[:-2])
        elif strUnit.lower().endswith('tb'):
            return convert2Float(strUnit[:-2]) * DOUBLE_1024
        elif strUnit.lower().endswith('pb') :
            return convert2Float(strUnit[:-2]) * DOUBLE_1024 ** 2
        elif strUnit.lower().endswith('b'):
            return convert2Float(strUnit[:-1]) / (DOUBLE_1024 ** 3)
        else:
            return 0.0
    except:
        msg = traceback.format_exc()
        LOG.error("conversionUnit2GB ==> " + msg)
        return 0.0

def getSuggestInfo():
    '''
    @summary: suggest info
    @return: string
    '''
    dividingLine = '------------------------------------------------\n'
    suggestInfo = '''1.If the result is True, there is no risk. Otherwise, go to the next step.
2.If the UltraPath version is 8.06.034 or 8.06.045, check the storage events. If the capacity of no LUN has been expanded, there is no risk. Otherwise, there are risks.
3.If the UltraPath version is between 6.01.022 and 8.06.022, there are risks.
4.If you have any questions, contact UltraPath engineers.
'''
    return dividingLine + suggestInfo

def convert2Float(strFloat):
    '''
    @summary: convert string to float
    @return: float number
    '''
    if not strFloat:
        return 0.0
    return float(strFloat)

class LunOver2TJudgement(object):
    '''
    @summary: the core class to judge whether disk is risk or not 
    '''

    __noPartitionOrSingle = []
    __noRiskPartition = []
    __riskPartition = []
    __startProcessor = None
    is_system_limited = False

    __template = "LUN:\n    LUNName: %s\n    LUNWWN: %s\n    LUNSize: %s\nPart:\n    PartType: %s\n    EndOfPatition: %s\n    PartNum: %s\n    PartName: %s\nLVM:\n    LVM: %s\n    PESize: %s\n    LastAllocatedPE: %s\nFS:\n    FilesystemType: %s"

    def __init__(self, context, lunDatas, log):
        self.__context = context
        self.__lunDatas = lunDatas
        self.__log = log

    def checkPartition(self):
        '''
        @summary: execute partition check, judge whether the disk is risk or not
        '''
        try:
            for lunData in self.__lunDatas:
                if self.is_system_limited:
                    return
                self.__checkSingleLun(lunData)
        except:
            msg = traceback.format_exc()
            self.__log.error("checkPartition ==> " + msg)

    def __checkSingleLun(self, lunData):
        '''
        @summary: check disk partition type
        '''
        lun = lunData.getLUNName()
        command = "parted /dev/%s print" % lun
        queryResult = singleCmdExecute(self.__context, command, "partition_for_lun_%s" % lun, True)
        cliRets.append(queryResult[1])
        if not queryResult[0]:
            self.__log.info("****[command: %s execute failure]***" % command)
        if re.search('No such file or directory', queryResult[1], re.I):
            self.__log.info("****[no corresponding disk for disk name: %s]***" % lun)
        elif re.search('unrecognised disk label', queryResult[1], re.I):
            self.__log.info("****[disk name: %s no partition]***" % lun)
            self.__noPartitionOrSingle.append(lunData)
            self.handlePartitionLVM(lunData)
        elif re.search('Partition Table.*msdos', queryResult[1], re.I):
            self.__log.info("****[disk name: %s has flag: msdos]***" % lun)
            lunData.setPartType('MSDOS')
            self.__noRiskPartition.append(lunData)
        elif re.search('Partition Table.*gpt', queryResult[1], re.I):
            self.__log.info("****[disk name: %s has flag: gpt]***" % lun)
            lunData.setPartType('GPT')
            self.__handleGptDisk(queryResult[1], lunData)
        else:
            self.__log.info("****[disk name: %s , command execute error]***" % lun)
            self.handlePartitionLVM(lunData)

    def __handleGptDisk(self, queryStr, lunData):
        '''
        @summary: handle gpt type partition, check the partition numbers and the last partition is whether or not large than 2TB
        '''
        LOG.info("***[start execute handleGptDisk]***")
        if re.search('Number.*Start.*End.*Size', queryStr, re.I):
            splitStrs = re.split('Number.*Start.*End.*Size.*', queryStr, re.I)
            lines = list(filter(lambda item : len(re.split('\s+', item)) > 3, stripOriginalStr(splitStrs[-1])))
            endNum = None
            if not lines:
                return
            endPartitionInfo = re.split('\s+', lines[-1])
            lunData.setEndOfPatition(endPartitionInfo[2])
            endNum = self.__conversionUnit2GB(endPartitionInfo[2])
            if endNum and endNum > DOUBLE_2048:
                lunData.setPartNum(str(len(lines)))
                lunData.setPartName(lunData.getLUNName() + endPartitionInfo[0])
                if len(lines) > 1:
                    self.__riskPartition.append(lunData)
                else:
                    self.__noPartitionOrSingle.append(lunData)
                    self.handlePartitionLVM(lunData)
            elif endNum:
                self.__noRiskPartition.append(lunData)

    def __conversionUnit2GB(self, strUnit):
        '''
        @summary: conversion unit 2 GB, 1GB = 1000MB = 1000 ^ 2KB = 1000 ^ 3B
        @return: the number in GB
        '''
        try:
            if not strUnit:
                return 0.0
            if strUnit.lower().endswith('kb'):
                return convert2Float(strUnit[:-2]) * CONVERSION_FACTOR / (DOUBLE_1024 ** 2)
            elif strUnit.lower().endswith('mb'):
                return convert2Float(strUnit[:-2]) * CONVERSION_FACTOR ** 2 / DOUBLE_1024
            elif strUnit.lower().endswith('gb'):
                return convert2Float(strUnit[:-2]) * CONVERSION_FACTOR ** 3
            elif strUnit.lower().endswith('tb'):
                return convert2Float(strUnit[:-2]) * CONVERSION_FACTOR ** 4 * DOUBLE_1024
            elif strUnit.lower().endswith('pb') :
                return convert2Float(strUnit[:-2]) * CONVERSION_FACTOR ** 5 * DOUBLE_1024 ** 2
            elif strUnit.lower().endswith('b'):
                return convert2Float(strUnit[:-1]) / (DOUBLE_1024 ** 3)
            else:
                return 0.0
        except:
            msg = traceback.format_exc()
            LOG.error("__conversionUnit2GB ==> " + msg)
            return 0.0

    def handlePartitionLVM(self, lunData):
        '''
        @summary: handle disk or single partion lun
        '''
        LOG.info("***[start execute handlePartitionLVM]***")
        lvmFlag, executeStatus = self.isLVM(lunData.getPartName() == 'NA' and lunData.getLUNName() or lunData.getPartName())
        if executeStatus and lvmFlag:
            lunData.setLVM('True')
            self.judgeLVM(lunData)
        elif executeStatus and not lvmFlag:
            self.judgeNonLvm(lunData)
        else:
            self.__riskPartition.append(lunData)

    def isLVM(self, lun):
        '''
        @summary: check partition is whether or not lvm
        @return: (flag1, flag2), flag1 is whether or not lvm, flag2: command is whether or not exception
        '''
        LOG.info("***[start execute isLVM]***")
        try:
            command = "pvs /dev/%s" % lun
            queryStr = singleCmdExecute(self.__context, command, "pvs_lun_for_%s" % lun, True)
            cliRets.append(queryStr[1])

            # LUN 超过1K 会触发Linux 系统已知问题，需要根据关键字进行过滤。
            if "too many open files" in str(queryStr[1]).lower():
                self.is_system_limited = True
                self.__log.info(
                    "***[command: {} execute failure]***".format(command))
                return False, False

            if not queryStr[0]:
                self.__log.info("***[command: %s execute failure]***" % command)
                return False, False
            if re.search('No physical volume label read from', queryStr[1], re.I):
                self.__log.info("***[this lun %s is not lvm]***" % lun)
                return False, True
            elif re.search('PV.*VG.*Fmt.*Attr.*PSize.*PFree', queryStr[1], re.I):
                return True, True
            else:
                self.__log.info("***[command error: %s, continue check]***" % command)
                return False, True
        except:
            msg = traceback.format_exc()
            self.__log.error("isLVM ==> " + msg)
            return False, False

    def judgeLVM(self, lunData):
        '''
        @summary: judge lun of lvm is over 2TB  
        '''
        LOG.info("***[start execute judgeLVM]***")
        try:
            lun = lunData.getPartName() == 'NA' and lunData.getLUNName() or lunData.getPartName()
            command = "pvdisplay -m /dev/%s" % lun
            queryStr = singleCmdExecute(self.__context, command, "pvdisplay lun for %s" % lun, True)
            cliRets.append(queryStr[1])

            # LUN 超过1K 会触发Linux 系统已知问题，需要根据关键字进行过滤。
            if "too many open files" in queryStr[1].lower:
                self.is_system_limited = True
                self.__log.info(
                    "***[command: {} execute failure]***".format(command))
                return

            if not queryStr[0]:
                self.__log.info("***[command: %s execute failure]***" % command)
                return
            if self.__isOver2TB(queryStr[1], lunData):
                self.__riskPartition.append(lunData)
            else:
                self.__noRiskPartition.append(lunData)
        except:
            msg = traceback.format_exc()
            self.__log.error("judgeLVM ==> " + msg)

    def __isOver2TB(self, queryStr, lunData):
        '''
        @summary: check the lun is whether or not over 2 TB
        @param queryStr: lun's info
        '''
        LOG.info("***[start execute isOvers2TB]***")
        lvmInfo = None
        if re.search('.*Physical Segments.*', queryStr, re.I):
            lvmInfo = re.split('---.*Physical Segments.*---', queryStr, re.I)
        if lvmInfo and len(lvmInfo) == 2:
            sizeInMB = self.__getPeSizeByMB(lvmInfo[0], lunData)
            if not self.__checkLegality(sizeInMB):
                self.__log.info("***[peSize is illegal]***")
                return False
            return sizeInMB * self.__getLastNonFreeSegmentSize(lvmInfo[1], lunData) / DOUBLE_1024 > DOUBLE_2048
        return False

    def __checkLegality(self, sizeInMB):
        '''
        @summary: legality check
        @return: True or False
        '''
        if sizeInMB:
            sizeInB = sizeInMB * 2 ** 20
            if sizeInB < 512 or sizeInB > 2 ** 40:
                return False;
            #sizeInB divided by 512 is the power of 2
            judgeNum = int(sizeInB / 2 ** 9)
            if (judgeNum & judgeNum - 1) == 0:
                return True
        return False

    def __getPeSizeByMB(self, physicalVolume, lunData):
        '''
        @summary: PE Size transfer to MB
        @param physicalVolume: physical volume
        @return: pe size(MB)
        '''
        LOG.info("***[start execute getPeSizeByMB]***")
        peSizeStr = list(filter(lambda line : 'pe size' in line.lower(), stripOriginalStr(physicalVolume)))
        if not peSizeStr:
            return 0
        peSizeInfo = re.split('\s+', peSizeStr[0])
        if not len(peSizeInfo) == 4:
            return 0
        lunData.setPESize(re.split('\s{2,}', peSizeStr[0])[-1])
        unitFlag = peSizeInfo[-1].lower()
        peSize = float(peSizeInfo[2])
        if unitFlag.startswith('b'):
            return peSize / (DOUBLE_1024 ** 2)
        elif unitFlag.startswith('k'):
            return peSize / DOUBLE_1024
        elif unitFlag.startswith('m'):
            return peSize
        elif unitFlag.startswith('g'):
            return peSize * DOUBLE_1024
        elif unitFlag.startswith('t'):
            return peSize * DOUBLE_1024 ** 2
        elif unitFlag.startswith('p'):
            return peSize * DOUBLE_1024 ** 3
        return 0

    def __getLastNonFreeSegmentSize(self, physicalSegments, lunData):
        '''
        @summary: get last non free segment size, this size multiply pe size can determine whether or not super 2TB
        @param physicalSegments: physical segments
        @return: last free pe size
        '''
        LOG.info("***[start execute getLastNonFreeSegmentSize]***")
        segments = stripOriginalStr(physicalSegments)
        segments.reverse()
        aimStr = ''
        try:

            for index, segment in enumerate(segments):
                if 'logical volume' in segment.lower():
                    aimStr = segments[index + 1]
                    break
            if aimStr:
                aimStr = aimStr.split()[-1][:-1]
                LOG.info("***[last free pe size is: %s]***" % aimStr)
                lunData.setLastAllocatedPE(aimStr)
                return int(aimStr) + 1
            return 0
        except:
            msg = traceback.format_exc()
            self.__log.error("__getLastNonFreeSegmentSize ==> " + msg)
            return 0

    def judgeNonLvm(self, lunData):
        '''
        @summary: judge lun of nonlvm is over 2TB
        '''
        LOG.info("***[start execute judgeNonLvm]***")
        if not self.__startProcessor:
            self.__startProcessor = makeChain(self.__log, self.__context, self.__riskPartition)
        self.__startProcessor.process(lunData)

    def obtainResultAndDisplayInfo(self):
        resultStr = ''
        dividingLine = '------------------------------------------------\n'
        suggestInfo = getSuggestInfo()
        if not self.__riskPartition:
            return True, 'Result: True\n' + suggestInfo + '\n'
        else:
            resultStr = 'Result: False\n'
        for lunData in self.__riskPartition:
            resultStr += dividingLine + self.__template % lunData.getLunInfoTuple() + '\n'
            resultStr += suggestInfo + '\n'
        return False, resultStr

class LunInfo(object):
    '''
    @summary: used to save the 2T lun data
    '''

    def __init__(self):
        self.__lUNName = ''
        self.__lUNWWN = ''
        self.__lUNSize = ''
        self.__partType = 'NA'
        self.__endOfPatition = 'NA'
        self.__partNum = 'NA'
        self.__partName = 'NA'
        self.__lVM = 'False'
        self.__pESize = 'NA'
        self.__lastAllocatedPE = 'NA'
        self.__filesystemType = 'NA'

    def getLUNName(self):
        return self.__lUNName

    def setLUNName(self, name):
        self.__lUNName = name

    def getLUNWWN(self):
        return self.__lUNWWN

    def setLUNWWN(self, wwn):
        self.__lUNWWN = wwn

    def getLUNSize(self):
        return self.__lUNSize

    def setLUNSize(self, size):
        self.__lUNSize = size

    def getPartType(self):
        return self.__partType

    def setPartType(self, typeValue):
        self.__partType = typeValue

    def getEndOfPatition(self):
        return self.__endOfPatition

    def setEndOfPatition(self, endPartition):
        self.__endOfPatition = endPartition

    def getPartNum(self):
        return self.__partNum

    def setPartNum(self, num):
        self.__partNum = num

    def getPartName(self):
        return self.__partName

    def setPartName(self, pName):
        self.__partName = pName

    def getLVM(self):
        return self.__lVM

    def setLVM(self, flag):
        self.__lVM = flag

    def getPESize(self):
        return self.__pESize

    def setPESize(self, eSize):
        self.__pESize = eSize

    def getLastAllocatedPE(self):
        return self.__lastAllocatedPE

    def setLastAllocatedPE(self, lastPE):
        self.__lastAllocatedPE = lastPE

    def getFilesystemType(self):
        return self.__filesystemType

    def setFilesystemType(self, sytemType):
        self.__filesystemType = sytemType

    def getLunInfoTuple(self):
        '''
        @summary: make tuple type data
        @return: tuple () is made of LunInfo's attribute
        '''
        return (self.__lUNName, self.__lUNWWN, self.__lUNSize,
                self.__partType, self.__endOfPatition, self.__partNum,
                self.__partName, self.__lVM, self.__pESize,
                self.__lastAllocatedPE, self.__filesystemType)
