# -*- coding: UTF-8 -*-
import re
import sys
import traceback
import os
path = os.path.dirname(os.path.abspath(__file__))
path = os.path.join(path, "..\\..")
sys.path.append(path)
from common.contentParse import singleCmdExecute

class Processor(object):
    '''
    @summary: base class, Initialize context
    '''
    
    def __init__(self, LOG, context, riskPartition, nextProcessor=None):
        self.LOG = LOG
        self.context = context
        self.riskPartition = riskPartition
        self.nextProcessor = nextProcessor
    
    def setNextProcessor(self, nextProcessor):
        '''
        @summary: setting next processor
        '''
        self.nextProcessor = nextProcessor
    
    def process(self, lunData):
        '''
        @summary: empty implementation, it will be implemented by subclass
        '''
        pass
    
    def hasNextProcessor(self):
        '''
        @summary: judge whether there is a processor or not
        '''
        return self.nextProcessor != None and True or False

class ASMProcessor(Processor):
    '''
    @summary: handle ASM
    '''
    
    def process(self, lunData):
        lun = lunData.getPartName() == 'NA' and lunData.getLUNName() or lunData.getPartName()
        command = "hexdump -C /dev/%s -n 128" % lun
        try:
            queryStr = singleCmdExecute(self.context, command, "hexdump_for_/dev/%s" % lun, True)
            if queryStr[0]:
                if re.search('00000020.*4f 52 43 4c 44 49 53 4b', queryStr[1], re.I):
                    lunData.setFilesystemType('ASM')
                    self.riskPartition.append(lunData)
                elif self.hasNextProcessor():
                    self.nextProcessor.process(lunData)
            elif self.hasNextProcessor():
                self.LOG.info("command: %s execute failure" % command)
                self.nextProcessor.process(lunData)
        except:
            msg = traceback.format_exc()
            self.LOG.error("ASMProcessor.process ==> " + msg)

class Ext4OrReiserFSOrXFSProcessor(Processor):
    '''
    @summary: handle Ext4 XFS ReiserFS
    '''
    
    def process(self, lunData):
        lun = lunData.getPartName() == 'NA' and lunData.getLUNName() or lunData.getPartName()
        command = "file -s /dev/%s" % lun
        try:
            queryStr = singleCmdExecute(self.context, command, "hexdump_for_/dev/%s" % lun, True)
            if queryStr[0]:
                if re.search('ext4', queryStr[1], re.I):
                    lunData.setFilesystemType('Ext4')
                    self.riskPartition.append(lunData)
                elif re.search('XFS', queryStr[1], re.I):
                    lunData.setFilesystemType('XFS')
                    self.riskPartition.append(lunData)
                elif re.search('ReiserFS', queryStr[1], re.I):
                    lunData.setFilesystemType('ReiserFS')
                    self.riskPartition.append(lunData)
                elif self.hasNextProcessor():
                    self.nextProcessor.process(lunData)
            elif self.hasNextProcessor():
                self.LOG.info("command: %s execute failure" % command)
                self.nextProcessor.process(lun)
        except:
            msg = traceback.format_exc()
            self.LOG.error("Ext4Processor.process ==> " + msg)

class OtherProcessor(Processor):
    '''
    @summary: other type processor
    '''
    def process(self, lunData):
        self.LOG.info("file system type is other")
        lunData.setFilesystemType('other')
        self.riskPartition.append(lunData)

def makeChain(LOG, context, riskPartition):
    '''
    @summary: make duty chain
    @param LOG: LOG
    @param context: execute context
    @param riskPartition:   save risk partition
    '''
    asmProcessor = ASMProcessor(LOG, context, riskPartition)
    ext4OrReiserFSOrXFSProcessor = Ext4OrReiserFSOrXFSProcessor(LOG, context, riskPartition)
    otherProcessor = OtherProcessor(LOG, context, riskPartition)
    asmProcessor.setNextProcessor(ext4OrReiserFSOrXFSProcessor)
    ext4OrReiserFSOrXFSProcessor.setNextProcessor(otherProcessor)
    return asmProcessor
    
