#  coding=UTF-8
#  Copyright (c) Huawei Technologies Co., Ltd. 2019-2021. All rights reserved.
# EMC存储脚本工具类
import os
import re
import json
from storages.storage_common import get_resource, get_command_result_lines

CLI = None
LANGUAGE = None
ERR_MSG = 'err_msg'
CMD_DISPLAY = 'cmd_display'
EMC_VNX = 'VNX_SAN'
EMC_VMAX = 'VMAX'
EMC_VPLEX = 'Vplex'
EMC_UNITY = 'Unity'
REGEX = '\\s{2,}'
NA = 'NA'
NEW_LINE = '\r\n'

VPLEX_CLI = 'vplexcli'
VPLEX_CLI_LS_CLUSTERS = "ls /clusters/"
invalid_result = [
    "toolkit_exe_cmd_failed", "toolkit_send_cmd_time_out",
    "command not found", "no such file or directory",
    "unknown command", "no context found for"
]


def cmd_execute(context, cmd_info_id, cmd_info_desc):
    """
    执行EMC cmd命令获取结果
    :param context: 结果内容
    :param cmd_info_id: 命令Key
    :param cmd_info_desc: cmd
    :return: cmd_display
    """
    global CLI
    global LANGUAGE
    CLI = context.get("SSH")
    LANGUAGE = context.get("lang")
    cmd_display = context.get("ret_map")
    fun_err_msg = ''
    for rg in range(len(cmd_info_id)):
        cmd_display_temp = CLI.execCmdHasLog(cmd_info_desc[rg])
        cmd_display.put(CMD_DISPLAY + (cmd_info_id[rg])[8:], cmd_display_temp)
        if check_result_invalid(cmd_display_temp):
            fun_err_msg += get_resource(LANGUAGE, "execute_fail")
        else:
            fun_err_msg += get_resource(LANGUAGE, "execute_success")
    cmd_display.put(ERR_MSG, fun_err_msg)
    return cmd_display


def batch_get_vplex_info(context, info_command_key, info_command):
    """
    批量获取Vplex信息，Rest接口执行
    :param context: 上下文
    :param info_command_key: 命令Key
    :param info_command: 命令
    :return: 结果数据
    """
    rest = context.get("SSH")
    language = context.get("lang")
    cmd_display = context.get("ret_map")
    error_msg = ''
    all_obj_json = rest.sendRestWithoutLog("GET", info_command, "")
    cmd_display.put(CMD_DISPLAY + info_command_key[8:], all_obj_json)
    error_msg += get_rest_result_message(info_command, all_obj_json, language)

    names = get_vplex_object_names(all_obj_json)
    for name in names:
        command = info_command + name
        command_key = CMD_DISPLAY + info_command_key[8:] + name
        object_json = rest.sendRestWithoutLog("GET", command, "")
        cmd_display.put(command_key, object_json)
        error_msg += get_rest_result_message(command, object_json, language)
    cmd_display.put(ERR_MSG, error_msg)
    return cmd_display


def get_rest_result_message(command, json_response, language):
    """
    获取命令执行结果信息
    :param command: 命令
    :param json_response: 执行结果回显
    :param language: 语言类型
    :return: 结果信息
    """
    json_obj = json.loads(json_response)
    return command + get_resource(language, "execute_success") \
        if json_obj else command + get_resource(language, "execute_fail")


def get_vplex_object_names(json_response):
    """
    从总数据中，获取所有对象名称信息
    :param json_response: 所有对象信息
    :return: 对象名称集合
    """
    json_obj = json.loads(json_response)
    contexts = json_obj.get("response", {}).get("context", [])
    names = []
    for context in contexts:
        children = context.get("children", [])
        for child in children:
            name = child.get("name", "")
            if name:
                names.append(name)
    return names


def get_vplex_all_objects(context, command):
    """
    获取Vplex所有对象信息
    :param context: 上下文
    :param command: 命令
    :return: 命令查询的所有对象
    """
    obj_names_json = "".join(get_command_result_lines(context, command))
    obj_names = get_vplex_object_names(obj_names_json)
    objects = []
    for name in obj_names:
        obj_json = "".join(get_command_result_lines(context, command + name))
        object_dicts = get_vplex_objects(obj_json)
        objects.extend(object_dicts)
    return objects


def get_vplex_objects(json_response):
    """
    获取所有对象属性信息
    :param json_response: 所有对象信息
    :return: 对象属性集合
    """
    json_obj = json.loads(json_response)
    contexts = json_obj.get("response", {}).get("context", [])
    objects = []
    for context in contexts:
        obj_dict = {}
        attributes = context.get("attributes", [])
        for attribute in attributes:
            obj_dict[attribute.get("name", "")] = attribute.get("value", "")
        objects.append(obj_dict)
    return objects


def vplex_rest_post_execute(context, command_keys, commands):
    """
    执行多条Rest命令
    :param context: 上下文
    :param command_keys: 命令Key
    :param commands: 命令
    :return: 结果数据
    """
    rest = context.get("SSH")
    language = context.get("lang")
    cmd_display = context.get("ret_map")
    error_msg = ''
    for index, command_key in enumerate(command_keys):
        rest_response = rest.sendRestWithoutLog("POST", commands[index], "")
        cmd_display.put(CMD_DISPLAY + command_key[8:], rest_response)
        error_msg += get_rest_result_message(commands[index], rest_response, language)
    cmd_display.put(ERR_MSG, error_msg)
    return cmd_display


def vplex_get_custom_data(json_response):
    """
    获取json回显中的数据信息
    :param json_response: rest返回的json回显信息
    :return: custom-data字段信息
    """
    json_obj = json.loads(json_response)
    return json_obj.get("response", {}).get("custom-data", "").strip()


def format_vplex_result(result, command):
    """
    将vplex cli回显格式处理，每行前加“|”字符，方式前方空白被strip
    :param result: vplex cli result
    :param command: 执行的命令
    :return: format result
    """
    format_result = ''
    for line in result.splitlines():
        if line.endswith(command):
            line = line[0:line.find(">") + 2] + command
        if command in line:
            format_result += line + NEW_LINE
        elif bool(line):
            format_result += "|" + line + NEW_LINE
    return format_result


def check_vplexcli(cli):
    """
    检查并将CLI执行模式切换为vplexcli模式
    :param cli ssh connection
    """
    result = cli.execCmdHasLogTimout(VPLEX_CLI_LS_CLUSTERS, 60)
    if check_result_invalid(result):
        cli.execCmdHasLogTimout(VPLEX_CLI, 10)


def check_result_invalid(cmd_result):
    if not bool(cmd_result):
        return True
    check_result = cmd_result.lower()
    for invalid_key in invalid_result:
        if invalid_key in check_result:
            return True
    return False


def check_vmax_json(line):
    """
    检查回显的行信息是否是json字符串
    :param line: 回显行信息
    :return: boolean
    """
    return bool(line) and line.startswith("{") and line.endswith("}")


def read_smart_idc_report(context, info_id, info_desc):
    """
    读取SmartIDC采集的报告文件内容
    :param context:  上下文
    :param info_id:  文件关键字
    :param info_desc:  文件名称
    :return:  cmd_display
    """
    global LANGUAGE
    LANGUAGE = context.get("lang")
    report_path = context.get("reportPath")
    cmd_display = context.get("ret_map")
    logger = context.get("Logger")
    device_model = context.get("dev_model")
    txt_dir = os.path.join(report_path, 'Storage')
    find_dir = False
    for report_dir in os.listdir(txt_dir):
        if device_model in report_dir:
            txt_dir = os.path.join(txt_dir, report_dir)
            txt_dir = os.path.join(txt_dir, 'config')
            for grab_path in os.listdir(txt_dir):
                if grab_path.startswith("grab"):
                    txt_dir = os.path.join(txt_dir, grab_path)
                    find_dir = True
                    break
            break
    if find_dir:
        read_grab_txt(cmd_display, logger, txt_dir, info_id, info_desc)
    else:
        cmd_display.put(ERR_MSG, get_resource(LANGUAGE, "invalid_file_path"))
    return cmd_display


def read_grab_txt(cmd_display, logger, txt_dir, info_id, info_desc):
    """
    从grab_目录下读取txt文件内容
    :param cmd_display: 上下文
    :param logger: 日志对象
    :param txt_dir: txt文件目录
    :param info_id: 文件Key
    :param info_desc: 文件名称
    :return: cmd_display
    """
    fun_err_msg = ""
    try:
        for index in range(len(info_desc)):
            result = info_desc[index] + "\r\n"
            with open(os.path.join(txt_dir, info_desc[index]), "r") as file:
                result += file.read()
            fun_err_msg += \
                info_desc[index] + get_resource(LANGUAGE, "execute_success")
            cmd_display.put(CMD_DISPLAY + info_id[index][9:], result)
    except IOError as ioe:
        logger.error("Read grab file error: {}".format(ioe))
        fun_err_msg += get_resource(LANGUAGE, "read_txt_file_fail")
    cmd_display.put(ERR_MSG, fun_err_msg)


def parse_vplex_cli_table_result(result, headers):
    """
    格式化读取vplex cli回显信息，表格类型
    :param result: 回显
    :param headers: 表头
    :return: 字典组
    """
    table_lst = []
    header_indexes = []
    start = False
    table_row = {}
    for re_line in result:
        line = re_line[1:]
        if not start and check_header_line(line, headers):
            update_header_start_index(line, headers, header_indexes)
            start = True
            continue
        if start and "----" not in line and bool(line):
            if not line.startswith("  ") and bool(table_row):
                put_row_to_table(table_lst, table_row, headers)
                table_row = {}
            put_line_value(line, headers, header_indexes, table_row)
    put_row_to_table(table_lst, table_row, headers)
    return table_lst


def put_row_to_table(table_lst, table_row, headers):
    if bool(table_row):
        for header in headers:
            if header not in table_row:
                table_row[header] = ''
        table_lst.append(table_row)


def put_line_value(line, headers, header_indexes, table_row):
    for index in range(len(headers)):
        value = line[header_indexes[index]:]
        if value.startswith("  "):
            continue
        value = re.split(REGEX, value)[0]
        if headers[index] in table_row:
            table_row[headers[index]] += value
        else:
            table_row[headers[index]] = value


def update_header_start_index(header_line, headers, header_indexes):
    for header in headers:
        header_indexes.append(header_line.find(header))


def check_header_line(line, headers):
    for header in headers:
        if header not in line:
            return False
    return True
