# coding=utf-8
#  Copyright (c) Huawei Technologies Co., Ltd. 2023-2024. All rights reserved.
import re

from storages.HUAWEI.nas.util.parse_common import parse_data
from storages.storage_common import get_command_result_lines


def execute(context):
    """
    获取isi 的 quota 信息
    :param context: 上下文数据
    """
    data_header = [
        "Path", "Type", "AppliesTo", "Snapshot",
        "Hard", "Soft", "Adv", "Grace", "Files",
        "Physical", "FileSystem Physical", "FileSystem Logical",
        "AppLogical", "ShadowLogical", "Physical Data", "Protection", "Reduction",
        "Efficiency", "Over", "Enforced", "Container", "Linked"
    ]
    parse_data(context, [data_header], get_quota_content)


def parse_isi_quota_info(headers, content_line):
    """
    根据 isi quota quotas list --verbose的回显信息的表头解析每一行表内容的字典数据
    由于该表格不固定分隔且无规律（多处充斥着单个空格或多个空格的可能）
    以下分 5 部分进行解析
    """
    content = content_line.split()
    dic = {}
    first_part = headers.index("Path")
    second_part = headers.index("ShadowLogical")
    third_part = headers.index("Efficiency") + 1
    last_part = len(headers)
    # 单个空格作为区分
    for i in range(first_part):
        dic[headers[i]] = content[i]
    # 此处是Path，Path可能存在多个及以上空格，这里是用正则做了一个规则来找下一个字段的位置
    # 找到第一个 No + 4个空格 + number/- 或者 第一个 Yes + 3个空格 + number/-
    # 这里如果是名字是这样的规范会出问题，但还未思考出万全之策（出现概率极低）
    rindex = content_line.find(content[2])
    pattern = r"(No\s{4}[-\d]+|Yes\s{3}[-\d]+)"
    m = re.search(pattern, content_line)
    dic[headers[first_part]] = content_line[rindex:m.start()].strip()
    # 此处是用单个空格区分，但需逐个向后处理，避免相同数据且为后续的 xx : xx 格式解析做好数据切割
    rest_content_line = content_line[m.start():]
    rest_content = rest_content_line.split()
    for i in range(second_part - first_part - 1):
        dic[headers[first_part + 1 + i]] = rest_content[i]
        index = rest_content_line.find(rest_content[i])
        rest_content_line = rest_content_line[index + len(rest_content[i]):]
    # 此处有两个格式为 xx : xx 的数据，该段数据也可能为 - ，所以使用两个空格进行解析
    rest_content = re.split(r"\s{2,}", rest_content_line.strip())
    for i in range(third_part - second_part):
        dic[headers[second_part + i]] = rest_content[i]
        index = rest_content_line.find(rest_content[i])
        rest_content_line = rest_content_line[index + len(rest_content[i]):]
    # 最后一部分会有单个空格作为分割符
    rest_content = rest_content_line.split()
    for i in range(last_part - third_part):
        dic[headers[third_part + i]] = rest_content[i]
    return dic


def is_separator_line(line):
    line_val = line.strip().replace(" ", "").replace("=", "").replace("-", "")
    return not line_val


def parse_isi_quota_info_result(cmd_result_lines):
    """
    解析表格类型的回显信息
    返回表头-列信息关系的数组，数组每个字典对象表示一行信息
    :param cmd_result_lines: 命令回显行信息
    :return: [{header: value}, ...]
    """
    table_data = []
    headers = []
    headers_line = ""
    first_flag = True
    for result_line in cmd_result_lines:
        line = result_line.strip()
        if not line or is_separator_line(line):
            continue
        # 排除首行
        if first_flag:
            items = re.split("\\s{2,}", line)
            first_flag = False
        else:
            items = re.split("\\s{1,}", line)
        if len(items) == 1:
            continue
        if not headers:
            headers = items
            headers_line = line
            continue
        # 排除末尾行
        if len(line) < len(headers_line) / 2:
            continue
        table_data.append(parse_isi_quota_info(headers, line))
    return table_data


def get_isi_quota_info_result(context, command, with_blank=True):
    result_lines = get_command_result_lines(context, command, with_blank=with_blank)
    return parse_isi_quota_info_result(result_lines)


def get_quota_content(context):
    """
    该命令格式可能有多个数据存在，且每个数据内分隔符不一致
    获取该数组数据后循环查询
    """
    quota_key_name = [
        "Path", "Type", "AppliesTo", "Snap",
        "Hard", "Soft", "Adv", "Grace", "Files",
        "Physical", "FSPhysical", "FSLogical", "AppLogical",
        "ShadowLogical", "PhysicalData", "Protection", "Reduction",
        "Efficiency", "Over", "Enforced", "Container", "Linked"
    ]
    quota_data = get_isi_quota_info_result(context, "cmd_display_quotas_list_verbose", False)

    return [[data.get(key, "-") for key in quota_key_name] for data in quota_data]
