#  coding=UTF-8
#  Copyright (c) Huawei Technologies Co., Ltd. 2019-2021. All rights reserved.

from storages.storage_common import HostInfo
from storages.EMC.emc_storage_util import NA
from storages.EMC.unity.emc_unity_info_getter import host_info_api_key, initiator_info_api_key
from storages.EMC.unity.unity_info_parser import get_contents, analysis_host_lun_map


def parse_unity_host_group(context):
    """
    解析Unity主机组信息
    :param context: 上下文
    :return: 主机组信息
    """
    initiators = parse_initiator(context)
    host_contents = get_contents(context, host_info_api_key)
    host_info_dict = parse_host(host_contents, initiators)
    map_groups = analysis_host_lun_map(host_contents)
    host_group_infos = parse_host_group_info(host_info_dict, map_groups)
    return transform_host_group_info(host_group_infos)


def transform_host_group_info(host_group_infos):
    """
    转换主机组信息为外部可用的数据结构
    :param host_group_infos: 主机组信息
    :return: 主机组信息数据
    """
    value_lst = []
    for host_group_info in host_group_infos:
        value_lst.extend(host_group_info.transform_host_value())
    return value_lst if value_lst else [NA, NA, NA, NA, NA, NA, NA, NA, NA]


def parse_host_group_info(host_info_dict, map_groups):
    """
    解析主机组信息，主机映射以主机组信息方式返回
    :param host_info_dict: 主机信息字典
    :param map_groups: 映射组信息
    :return: 主机组信息
    """
    host_group_infos = []
    for index, map_group in enumerate(map_groups):
        host_ids = map_group[0]
        if not host_ids:
            continue
        mapping_view_id = str(index + 1)
        # 单主机映射
        if len(host_ids) == 1:
            host_id = host_ids[0]
            host_info = host_info_dict[host_id][0]
            mapping_view_name = "MV_" + host_info.get_host_name()
            host_group_name = "HG_" + host_info.get_host_name()
            host_group_infos.extend(build_host_group_info(host_info_dict[host_id], mapping_view_id,
                                                          mapping_view_name, host_group_name))
        # 多主机共享映射
        else:
            mapping_view_name = "MappingView%03d" % (index + 1)
            host_group_name = "HostGroup%03d" % (index + 1)
            for host_id in host_ids:
                host_group_infos.extend(build_host_group_info(host_info_dict[host_id], mapping_view_id,
                                                              mapping_view_name, host_group_name))
    return host_group_infos


def build_host_group_info(host_infos, mapping_view_id, mapping_view_name, host_group_name):
    """
    构建主机组信息
    :param host_infos: 主机信息
    :param mapping_view_id: 映射视图ID
    :param mapping_view_name: 映射视图名称
    :param host_group_name: 主机组名称
    :return: 主机组信息集合
    """
    host_group_infos = []
    for host_info in host_infos:
        host_group_info = host_info.copy_host()
        host_group_info.set_mapping_view_id(mapping_view_id)
        host_group_info.set_mapping_view_name(mapping_view_name)
        host_group_info.set_host_group_name(host_group_name)
        host_group_infos.append(host_group_info)
    return host_group_infos


def parse_host(host_contents, initiators):
    """
    解析主机信息
    :param host_contents: 主机信息
    :param initiators: 启动器信息
    :return: 主机信息字典
    """
    host_infos = {}
    for host_content in host_contents:
        host_id = host_content.get("id", "Host_NA")
        initiator_ids = get_initiator_ids(host_content)
        if not initiator_ids:
            initiator_ids.append(NA)
        for initiator_id in initiator_ids:
            host_info = build_host_info(host_content, initiators.get(initiator_id, NA))
            # 每个主机启动器生成一个主机信息，主机有多个启动器时，会生成多个主机信息
            if host_id not in host_infos:
                host_infos[host_id] = []
            host_infos[host_id].append(host_info)
    return host_infos


def build_host_info(host_content, port_wwn):
    """
    构建主机信息数据
    :param host_content: 主机数据
    :param port_wwn: 启动器WWN
    :return: 主机信息
    """
    host_info = HostInfo()
    host_info.set_host_name(host_content.get("name", NA))
    host_info.set_host_id(host_content.get("id", "Host_NA").split("_")[1])
    host_type = host_content.get("osType", NA)
    host_info.set_host_type(host_type if host_type else NA)
    host_info.set_initiator_wwn(port_wwn)
    return host_info


def parse_initiator(context):
    """
    解析启动器信息
    :param context: 上下文
    :return: 启动器ID-WWN字典
    """
    initiators = {}
    initiator_contents = get_contents(context, initiator_info_api_key)
    for initiator_content in initiator_contents:
        initiator_id = initiator_content.get("id", "")
        port_wwn = initiator_content.get("portWWN", "")
        if initiator_id and port_wwn:
            initiators[initiator_id] = port_wwn
    return initiators


def get_initiator_ids(host_content):
    """
    获取启动器ID集合
    :param host_content: 主机信息
    :return: 启动器ID集合
    """
    initiator_ids = []
    host_initiators = host_content.get("fcHostInitiators", [])
    for host_initiator in host_initiators:
        initiator_id = host_initiator.get("id", "")
        if initiator_id:
            initiator_ids.append(initiator_id)
    return initiator_ids
