#  coding=UTF-8
#  Copyright (c) Huawei Technologies Co., Ltd. 2019-2021. All rights reserved.

from storages.storage_common import LunInfo
from storages.EMC.emc_storage_util import NA
from storages.EMC.unity.emc_unity_info_getter import host_info_api_key, lun_info_api_key
from storages.EMC.unity.unity_info_parser import get_contents, analysis_host_lun_map

THIN = 'THIN'
HEALTH_ENUM = {0: 'UNKNOWN', 5: 'OK', 7: 'OK_BUT', 10: 'DEGRADED', 15: 'MINOR', 20: 'MAJOR', 25: 'CRITICAL',
               30: 'NON_RECOVERABLE'}
ONE_GB = 1 << 30


def parse_unity_lun_info(context):
    """
    解析LUN信息
    :param context: 上下文
    :return: LUN信息
    """
    host_contents = get_contents(context, host_info_api_key)
    map_groups = analysis_host_lun_map(host_contents)
    host_id_name_dict = get_host_id_name_dict(host_contents)
    lun_info_dict = parse_lun(context)
    lun_group_infos = parse_lun_group_info(lun_info_dict, map_groups, host_id_name_dict)
    return transform_lun_group_info(lun_group_infos)


def transform_lun_group_info(lun_group_infos):
    """
    转换LUN组信息数据值
    :param lun_group_infos: LUN组信息
    :return: LUN组信息数据
    """
    value_lst = []
    for lun_group_info in lun_group_infos:
        value_lst.extend(lun_group_info.transform_lun_value())
    return value_lst if value_lst else [NA, NA, NA, NA, NA, NA, NA, NA, NA, NA, NA, NA]


def get_host_id_name_dict(host_contents):
    """
    获取主机ID和名称的字典
    :param host_contents: 主机信息
    :return: 主机ID-名称字典
    """
    host_id_name_dict = {}
    for host_content in host_contents:
        host_id_name_dict[host_content.get("id", "Host_NA")] = host_content.get("name", NA)
    return host_id_name_dict


def parse_lun_group_info(lun_info_dict, map_groups, host_id_name_dict):
    """
    解析LUN组信息，LUN映射信息以LUN组信息形式返回
    :param lun_info_dict: LUN信息字典
    :param map_groups: 映射组信息
    :param host_id_name_dict: 主机ID-名称字典
    :return: LUN组信息
    """
    lun_group_infos = []
    for index, map_group in enumerate(map_groups):
        mapping_view_id = str(index + 1)
        lun_ids = map_group[1]
        # 主机无LUN映射，构建空数据，方便表格展示
        if not lun_ids:
            host_name = host_id_name_dict.get(map_group[0][0], NA)
            lun_group_info = LunInfo()
            lun_group_info.set_mapping_view_id(mapping_view_id)
            lun_group_info.set_mapping_view_name("MV_" + host_name)
            lun_group_infos.append(lun_group_info)
            continue
        host_ids = map_group[0]
        # 单主机映射
        if len(host_ids) == 1:
            host_name = host_id_name_dict.get(host_ids[0], NA)
            mapping_view_name = "MV_" + host_name
            lun_group_name = "LG_" + host_name
            lun_group_infos.extend(build_lun_group_info(lun_info_dict, lun_ids, mapping_view_id, mapping_view_name,
                                                        lun_group_name))
        # 多主机共享映射
        else:
            mapping_view_name = "MappingView%03d" % (index + 1)
            lun_group_name = "LunGroup%03d" % (index + 1)
            lun_group_infos.extend(build_lun_group_info(lun_info_dict, lun_ids, mapping_view_id, mapping_view_name,
                                                        lun_group_name))
    return lun_group_infos


def build_lun_group_info(lun_info_dict, lun_ids, mapping_view_id, mapping_view_name, lun_group_name):
    """
    构建LUN组信息
    :param lun_info_dict: LUN信息字典
    :param lun_ids: LUN ID集合
    :param mapping_view_id: 映射视图ID
    :param mapping_view_name: 映射视图名称
    :param lun_group_name: LUN组名称
    :return: LUN组信息
    """
    lun_group_infos = []
    for lun_id in lun_ids:
        lun_group_info = lun_info_dict.get(lun_id, LunInfo()).copy_lun()
        lun_group_info.set_mapping_view_id(mapping_view_id)
        lun_group_info.set_mapping_view_name(mapping_view_name)
        lun_group_info.set_lun_group_name(lun_group_name)
        lun_group_infos.append(lun_group_info)
    return lun_group_infos


def parse_lun(context):
    """
    解析LUN信息
    :param context: 上下文
    :return: LUN信息
    """
    lun_infos = {}
    lun_contents = get_contents(context, lun_info_api_key)
    for lun_content in lun_contents:
        lun_info = LunInfo()
        lun_id = lun_content.get("id", "sv_NA")
        lun_info.set_lun_id(lun_id.split("_")[1])
        lun_info.set_lun_name(lun_content.get("name", NA))
        lun_info.set_capacity(convert_byte_to_gb(lun_content))
        lun_info.set_status(get_lun_status(lun_content))
        lun_info.set_lun_wwn(lun_content.get("wwn", NA))
        lun_info.set_lun_type(get_lun_type(lun_content))
        lun_infos[lun_id] = lun_info
    return lun_infos


def convert_byte_to_gb(lun_content):
    """
    获取LUN容量并将字节数转换为GB大小
    :param lun_content: LUN数据
    :return: GB大小
    """
    return "%.2f" % (lun_content.get("sizeTotal", 0) / ONE_GB) + "GB"


def get_lun_status(lun_content):
    """
    获取LUN状态
    :param lun_content: LUN数据
    :return: LUN状态
    """
    return HEALTH_ENUM.get(lun_content.get("health", {}).get("value", -1), NA)


def get_lun_type(lun_content):
    """
    获取LUN类型
    :param lun_content: LUN数据
    :return: 类型：THIN或NA
    """
    return THIN if lun_content.get("isThinEnabled", False) else NA
