#  coding=UTF-8
#  Copyright (c) Huawei Technologies Co., Ltd. 2019-2020. All rights reserved.
# --*-- coding:utf-8 --*--
import re
import json
from common.contentParse import getValueFMap, addValueToMap, get_jsonstr, get_attrcom_dicv2, get_attrvalue_dic, \
    get_jsonarr_name, style
from storages.EMC.emc_storage_util import NA, EMC_VNX, EMC_VMAX, EMC_VPLEX, EMC_UNITY
from storages.EMC.emc_storage_util import check_vmax_json, get_vplex_all_objects
from storages.EMC.unity.unity_host_info_parser import parse_unity_host_group


# 对应excel中的属性字段
key_lst = [
    'mapping_view_id', 'mapping_view_name', 'host_group_id',
    'host_group_name', 'host_id', 'host_name', 'host_type',
    'initiator_wwn', 'port'
]
gp_name_prefix = "HostGroup%03d"


def execute(context):
    """
    解析EMC存储主机信息
    :param context:
    :return:
    """
    command = getValueFMap(context, style, 'command').strip()
    addValueToMap(context, style, 'command', command)
    # 属性名与对应命令列表组成的字典
    attrcom_dic = get_attrcom_dicv2(context)
    # 对应execel属性名
    json_attr_name = get_jsonarr_name(attrcom_dic)
    attr_valuelst_dic = get_attrvalue_dic(context, attrcom_dic)
    # 获取attribute ,value_lst
    attribute_arr = []
    for key in key_lst:
        for attr in attr_valuelst_dic.keys():
            if key.lower().strip() in attr.lower():
                attribute_arr.append(attr)

    # storageGroup信息
    sg_lst = attr_valuelst_dic[attribute_arr[-2]]
    # hba卡信息
    hba_value_lst = attr_valuelst_dic[attribute_arr[-4]]

    dev_model = getValueFMap(context, style, 'dev_model').strip()
    if dev_model == EMC_VNX:
        # 获取传入的mapping_view_id
        mapping_view_id = attrcom_dic['mapping_view_id'][0].strip()
        value_lst = get_vnx_info(sg_lst, hba_value_lst, mapping_view_id)
    elif dev_model == EMC_VMAX:
        # hostGroup信息
        host_gp_lst = attr_valuelst_dic[attribute_arr[-6]]
        value_lst = get_vmax_info(sg_lst, host_gp_lst, hba_value_lst)
    elif dev_model == EMC_VPLEX:
        value_lst = get_vplex_info(context)
    elif dev_model == EMC_UNITY:
        value_lst = parse_unity_host_group(context)
    else:
        value_lst = [NA, NA, NA, NA, NA, NA, NA, NA, NA]
    # 获取json结果字符串
    result = get_jsonstr(json_attr_name, attribute_arr, value_lst)
    addValueToMap(context, style, 'result', result)
    return context


def get_vplex_info(context):
    """
    解析Vplex信息
    :param context: 上下文
    :return: value_lst
    """
    host_map = get_vplex_host_map(context)
    host_gp_entities = parse_vplex_host_info(context, host_map)
    return trans_hg_to_val_lst(host_gp_entities)


def do_vplex_hg_entity(host_group_lst, host_map, port_info,
                       storage_view_name, gp_id):
    for item in port_info.split(","):
        host_name = item.strip()
        if bool(host_name) and host_name in host_map:
            hg_entity = HostGroupEntity()
            hg_entity.set_mapping_name(storage_view_name)
            hg_entity.set_gp_id(str(gp_id))
            hg_entity.set_hg_name(gp_name_prefix % gp_id)
            hg_entity.set_host_name(host_name)
            hg_entity.set_initiator(host_map[host_name][2:])
            host_group_lst.append(hg_entity)


def get_vplex_host_map(context):
    """
    获取Vplex主机端口信息
    :param context: 上下文
    :return: 主机端口名称-启动器WWN 映射
    """
    initiator_dicts = get_vplex_all_objects(context, "cmd_display_initiators_")
    host_map = {}
    for initiator_dict in initiator_dicts:
        name = initiator_dict.get("name", "")
        wwn = initiator_dict.get("port-wwn", "")
        if name and wwn:
            host_map[name] = wwn[2:]
    return host_map


def parse_vplex_host_info(context, host_map):
    """
    解析Vplex映射信息
    :param context: 上下文
    :param host_map: 主机启动器映射
    :return: 主机组信息
    """
    storage_view_dicts = get_vplex_all_objects(context, "cmd_display_storage_views_")
    host_infos = []
    storage_view_id = 0
    for storage_view_dict in storage_view_dicts:
        storage_view_id += 1
        storage_view_name = storage_view_dict.get("name", "")
        host_gp_name = "HostGroup%03d" % storage_view_id
        initiator_names = storage_view_dict.get("initiators", [])
        if not initiator_names:
            # 映射中无LUN时，添加空白LUN信息，以让表格可展示映射中主机信息
            host_info = HostGroupEntity()
            host_info.set_gp_id(str(storage_view_id))
            host_info.set_mapping_name(storage_view_name)
            host_infos.append(host_info)
            continue
        for initiator_name in initiator_names:
            host_info = HostGroupEntity()
            host_info.set_gp_id(str(storage_view_id))
            host_info.set_mapping_name(storage_view_name)
            host_info.set_hg_name(host_gp_name)
            host_info.set_host_name(initiator_name)
            host_info.set_initiator(host_map.get(initiator_name, ""))
            host_infos.append(host_info)
    return host_infos


def get_vmax_info(sg_lst, host_gp_lst, hba_value_lst):
    """
    解析VMAX信息
    :param sg_lst: host gp info
    :param host_gp_lst: host group info
    :param hba_value_lst: host info
    :return: value_lst
    """
    mv_map = get_vmax_mv_map(sg_lst)
    host_entities = get_host_entities(hba_value_lst)
    hg_entities = get_hg_entities(host_gp_lst)
    host_group_list = []
    gp_id = 0
    for mv_entry in mv_map:
        gp_id += 1
        mv_name = mv_entry[0]
        hg_name = mv_entry[1]
        if hg_name != NA and hg_name in hg_entities:
            entities = hg_entities[hg_name]
            append_entities(entities, gp_id, hg_name,
                            host_group_list, mv_name)
            continue
        host_name = mv_entry[2]
        if host_name != NA and host_name in host_entities:
            entities = host_entities[host_name]
            hg_name = gp_name_prefix % gp_id
            append_entities(entities, gp_id, hg_name,
                            host_group_list, mv_name)
    return trans_hg_to_val_lst(host_group_list)


def trans_hg_to_val_lst(host_group_lst):
    if bool(host_group_lst):
        value_lst = []
        for hg_entity in host_group_lst:
            value_lst.append(hg_entity.get_gp_id())
            value_lst.append(hg_entity.get_mapping_name())
            value_lst.append(NA)
            value_lst.append(hg_entity.get_hg_name())
            value_lst.append(NA)
            value_lst.append(hg_entity.get_host_name())
            value_lst.append(hg_entity.get_host_type())
            value_lst.append(hg_entity.get_initiator())
            value_lst.append(NA)
    else:
        value_lst = [NA, NA, NA, NA, NA, NA, NA, NA, NA]
    return value_lst


def append_entities(entities, gp_id, hg_name, host_group_list, mv_name):
    for entity in entities:
        host_entity = copy_host_entity(entity)
        host_entity.set_hg_name(hg_name)
        host_entity.set_mapping_name(mv_name)
        host_entity.set_gp_id(str(gp_id))
        host_group_list.append(host_entity)


def copy_host_entity(host_entity):
    new_entity = HostGroupEntity()
    new_entity.set_mapping_name(host_entity.get_mapping_name())
    new_entity.set_gp_id(host_entity.get_gp_id())
    new_entity.set_hg_name(host_entity.get_hg_name())
    new_entity.set_host_name(host_entity.get_host_name())
    new_entity.set_host_type(host_entity.get_host_type())
    new_entity.set_initiator(host_entity.get_initiator())
    return new_entity


def get_vmax_mv_map(sg_lst):
    """
    解析maskingView信息
    :param sg_lst: maskingView info
    :return: [(mv_name, hg_name, host_name), ...]
    """
    mv_map = []
    for line in sg_lst:
        if check_vmax_json(line):
            masking_view = json.loads(line).get("maskingView", [{}])[0]
            mv_name = masking_view.get("maskingViewId", NA)
            hg_name = masking_view.get("hostGroupId", NA)
            host_name = masking_view.get("hostId", NA)
            mv_map.append((mv_name, hg_name, host_name))
    return mv_map


def get_hg_entities(host_gp_lst):
    """
    读取hostGroup信息，从已解析的host集合中关联获取主机
    :param host_gp_lst: hostGroup信息
    :return: {hg_name: [[host_entity, ...], ...]}
    """
    hg_entities = {}
    for line in host_gp_lst:
        if check_vmax_json(line):
            hg_json = json.loads(line).get("hostGroup", [{}])[0]
            hg_name = hg_json.get("hostGroupId", NA)
            hosts = hg_json.get("host", {})
            host_type = hg_json.get("type", NA)
            for host_json in hosts:
                host_name = host_json.get("hostId", NA)
                initiators = host_json.get("initiator", [NA])
                for initiator in initiators:
                    host_entity = HostGroupEntity()
                    host_entity.set_host_name(host_name)
                    host_entity.set_initiator(initiator)
                    host_entity.set_host_type(host_type)
                    if hg_name not in hg_entities:
                        hg_entities[hg_name] = []
                    hg_entities[hg_name].append(host_entity)
    return hg_entities


def get_host_entities(hba_value_lst):
    """
    解析主机信息
    :param hba_value_lst: 主机信息
    :return: {host_name: [host_entity, ...]}
    """
    host_entities = {}
    for line in hba_value_lst:
        if check_vmax_json(line):
            host_json = json.loads(line).get("host", [{}])[0]
            host_name = host_json.get("hostId", NA)
            host_type = host_json.get("type", NA)
            initiators = host_json.get("initiator", [NA])
            if not bool(initiators):
                continue
            for initiator in initiators:
                host_entity = HostGroupEntity()
                host_entity.set_host_name(host_name)
                host_entity.set_host_type(host_type)
                host_entity.set_initiator(initiator)
                if host_name not in host_entities:
                    host_entities[host_name] = []
                host_entities[host_name].append(host_entity)
    return host_entities


def get_vnx_info(sg_lst, hba_value_lst, mapping_view_id):
    """
    解析VNX信息
    :param sg_lst: storageGroup
    :param hba_value_lst: hba
    :param mapping_view_id: mapping view id
    :return: value_lst
    """
    # 由sg求 storagename & id & hba_lst 之对象集合
    # 1.求Storage_Group_Entity 命令是：cmd_display_storage_group_sg
    storage_group_lst = get_storage_group_lst(sg_lst)
    storage_group = StorageGroupEntity()
    for storage_gp in storage_group_lst:
        if mapping_view_id in storage_gp.get_uid():
            storage_group = storage_gp
    mapping_view_name = storage_group.get_name()
    hba_lst = storage_group.get_lst()
    port_lst = storage_group.get_portlst()

    # 由hba求hba之uid &ServerName即host_name
    # 2.求hba_server，命令是：cmd_display_storage_group_hba
    hba_server_lst = get_hba_server_lst(hba_value_lst)

    return format_value_lst(hba_lst, mapping_view_name, hba_server_lst, port_lst)


def format_value_lst(hba_lst, mapping_view_name, hba_server_lst, port_lst):
    if bool(hba_lst):
        value_lst = []
        for port_idx, hba_wwn in enumerate(hba_lst):
            value_lst.append(NA)
            value_lst.append(mapping_view_name)
            value_lst.extend([NA, NA, NA])
            # 根据hba uid查询host name
            host_name = NA
            for hba_egg in hba_server_lst:
                if hba_wwn in hba_egg.get_uid():
                    host_name = hba_egg.get_hostname()
                    break
            value_lst.append(host_name)
            value_lst.append(NA)
            value_lst.append(hba_wwn)
            value_lst.append(port_lst[port_idx])
    else:
        value_lst = [NA, mapping_view_name, NA, NA, NA, NA, NA, NA, NA]
    return value_lst


def get_storage_group_lst(temp_lst):
    storage_gp_bean = StorageGroupEntity()
    storage_lst = []
    if not bool(temp_lst):
        storage_lst.append(storage_gp_bean)
        return storage_lst
    for i, line in enumerate(temp_lst):
        if 'storage' in line.lower() and 'group' in line.lower() \
                and 'name' in line.lower():
            storage_gp_bean = StorageGroupEntity()
            name = line.split(':')[-1].strip()
            storage_gp_bean.set_name(name)
        elif 'storage' in line.lower() and 'group' in line.lower() \
                and 'uid' in line.lower():
            id_temp_lst = re.split('\\s+', line.strip())
            uid = id_temp_lst[-1].strip()
            storage_gp_bean.set_id(uid)
            storage_lst.append(storage_gp_bean)
        elif 'hba' in line.lower() and 'uid' in line.lower():
            hba_start = i + 2
            hba_id_lst = temp_lst[hba_start:]
            id_lst = []
            port_lst = []
            do_hba_id_lst(hba_id_lst, id_lst, port_lst, storage_gp_bean)
    return storage_lst


def do_hba_id_lst(hba_id_lst, id_lst, port_lst, storage_gp_bean):
    for uid in hba_id_lst:
        if uid.count(':') > 3:
            lst = re.split('\\s+', uid.strip())
            if len(lst) >= 2:
                id_value = lst[0].strip()
                id_lst.append(id_value)
                port_lst.append((lst[1].strip() + lst[-1]).replace(' ', ''))
        else:
            storage_gp_bean.set_lst(id_lst)
            storage_gp_bean.set_portlst(port_lst)
            break


def get_hba_server_lst(temp_lst):
    hba_lst = []
    if temp_lst is None or 0 == len(temp_lst):
        return hba_lst
    for i, line in enumerate(temp_lst):
        if 'hba' in line.lower() and 'uid' in line.lower():
            hba_server = HbaServerEntity()
            id_temp_lst = re.split('\\s+', line)
            uid = id_temp_lst[-1]
            uid = uid.strip()
            hba_server.set_uid(uid)
            name_idx = i + 1
            name_temp = temp_lst[name_idx]
            name = name_temp.split(':')[-1].strip()
            hba_server.set_hostname(name)
            hba_lst.append(hba_server)
    return hba_lst


class StorageGroupEntity:
    __group_name = NA
    __group_uid = NA
    __hbaid_lst = []
    __port_lst = []
    
    def get_name(self):
        return self.__group_name

    def get_uid(self):
        return self.__group_uid

    def get_lst(self):
        return self.__hbaid_lst

    def get_portlst(self):
        return self.__port_lst
    
    def set_name(self, name):
        self.__group_name = name

    def set_id(self, uid):
        self.__group_uid = uid

    def set_lst(self, lst):
        self.__hbaid_lst = lst

    def set_portlst(self, lst):
        self.__port_lst = lst


class HbaServerEntity:
    __hba_uid = NA
    __host_name = NA

    def get_uid(self):
        return self.__hba_uid

    def get_hostname(self):
        return self.__host_name

    def set_uid(self, uid):
        self.__hba_uid = uid

    def set_hostname(self, name):
        self.__host_name = name


class HostGroupEntity:
    __mapping_name = NA
    __hg_name = NA
    __gp_id = NA
    __host_name = NA
    __host_type = NA
    __initiator = NA

    # get
    def get_mapping_name(self):
        return self.__mapping_name

    def get_hg_name(self):
        return self.__hg_name

    def get_gp_id(self):
        return self.__gp_id

    def get_host_name(self):
        return self.__host_name

    def get_host_type(self):
        return self.__host_type

    def get_initiator(self):
        return self.__initiator

    # set
    def set_mapping_name(self, mapping_name):
        self.__mapping_name = mapping_name

    def set_hg_name(self, hg_name):
        self.__hg_name = hg_name

    def set_gp_id(self, gp_id):
        self.__gp_id = gp_id

    def set_host_name(self, host_name):
        self.__host_name = host_name

    def set_host_type(self, host_type):
        self.__host_type = host_type

    def set_initiator(self, initiator):
        self.__initiator = initiator
