#  coding=UTF-8
#  Copyright (c) Huawei Technologies Co., Ltd. 2019-2020. All rights reserved.
# --*-- coding:utf-8 --*--
import re
import json
from common.contentParse import getValueFMap, addValueToMap, get_jsonstr, get_attrcom_dicv2, get_attrvalue_dic,\
    get_jsonarr_name, style
from storages.storage_common import get_command_result_lines
from storages.EMC.emc_storage_util import NA, EMC_VNX, EMC_VMAX, EMC_VPLEX, EMC_UNITY
from storages.EMC.emc_storage_util import vplex_get_custom_data, check_vmax_json
from storages.EMC.unity.emc_unity_info_getter import system_info_api_key, basic_system_info_api_key
from storages.EMC.unity.unity_host_info_parser import get_contents

# 对应excel中的属性字段
key_lst = ['report_date/time', 'infograb_version', 'storage_vendor', 'storage_series',
           'storage_model', 'storage_firmware', 'storage_serial', 'ip_address']


def execute(context):
    """
    获取EMC设备基本信息
    :param context: 上下文
    :return: context
    """
    command = getValueFMap(context, style, 'command').strip()
    addValueToMap(context, style, 'command', command)
    # 属性名与对应命令列表组成的字典
    attrcom_dic = get_attrcom_dicv2(context)
    # 对应execel属性名
    json_attr_name = get_jsonarr_name(attrcom_dic)
    attr_valuelst_dic = get_attrvalue_dic(context, attrcom_dic)
    # 获取attribute_arr
    attribute_arr = []
    for key in key_lst:
        for attr in attr_valuelst_dic.keys():
            if key.lower().strip() in attr.lower():
                attribute_arr.append(attr)

    # storage_model
    model_lst = attr_valuelst_dic[attribute_arr[4]]
    # storage_serial
    serial_lst = attr_valuelst_dic[attribute_arr[6]]

    dev_model = getValueFMap(context, style, 'dev_model').strip()
    if dev_model == EMC_VNX:
        value_lst = get_vnx_info(model_lst, serial_lst)
    elif dev_model == EMC_VMAX:
        value_lst = get_vmax_info(serial_lst)
    elif dev_model == EMC_VPLEX:
        value_lst = get_vplex_info(context)
    elif dev_model == EMC_UNITY:
        value_lst = get_unity_info(context)
    else:
        value_lst = [NA, NA, NA, NA, NA, NA]
    # 获取json结果字符串
    result = get_jsonstr(json_attr_name, attribute_arr, value_lst)
    addValueToMap(context, style, 'result', result)
    return context


def get_unity_info(context):
    """
    获取Unity信息，型号，序列号，版本号
    :param context:
    :return:
    """
    content = get_contents(context, system_info_api_key)[0]
    model = content.get("model", NA)
    sn = content.get("serialNumber", NA)
    content = get_contents(context, basic_system_info_api_key)[0]
    version = content.get("softwareVersion", NA)
    return [NA, NA, "EMC", "Unity Series", model, version, sn, NA]


def get_vplex_info(context):
    """
    获取Vplex的型号、版本号和序列号
    :param context: 上下文信息
    :return: Vplex设备信息
    """
    sn = get_vplex_serial_number(context)
    version = get_vplex_version(context)
    return [NA, NA, "EMC", EMC_VPLEX, EMC_VPLEX, version, sn, NA]


def get_vplex_serial_number(context):
    """
    获取设备序列号
    :param context: 上下文
    :return: sn
    """
    json_response = "".join(get_command_result_lines(context, "cmd_display_cluster_summary"))
    data = vplex_get_custom_data(json_response)
    if not data:
        return 'NA'
    start = False
    for line in data.split("\n"):
        if not start and "Name" in line and "TLA" in line:
            start = True
            continue
        if start and "----" not in line:
            items = line.split()
            return items[2].strip() if len(items) > 2 else 'NA'
    return 'NA'


def get_vplex_version(context):
    """
    获取设备版本号
    :param context: 上下文
    :return: version
    """
    json_response = "".join(get_command_result_lines(context, "cmd_display_device_version"))
    data = vplex_get_custom_data(json_response)
    if not data:
        return 'NA'
    for line in data.split("\n"):
        if "Product Version" in line:
            items = re.split("\\s{2,}", line)
            return items[1].strip() if len(items) > 1 else 'NA'
    return 'NA'


def get_vmax_info(serial_lst):
    """
    获取VMAX的型号、版本号和序列号
    :param serial_lst: 序列号信息
    :return: value_lst
    """
    model, firmware, serial = NA, NA, NA
    for line in serial_lst:
        if check_vmax_json(line):
            symmetrix = json.loads(line).get("symmetrix", [{}])[0]
            model = symmetrix.get("model", NA)
            firmware = symmetrix.get("ucode", NA)
            serial = symmetrix.get("symmetrixId", NA)
            break
    return [NA, NA, "EMC", "VMAX Series", model, firmware, serial, NA]


def get_vnx_info(model_lst, serial_lst):
    """
    获取VNX的型号、版本号和序列号
    :param model_lst: 型号信息
    :param serial_lst: 序列号信息
    :return: value_lst
    """
    # storage_model
    model = NA
    for line in model_lst:
        if ('model' in line.lower()) and (':' in line.lower()):
            lst = re.split(':', line)
            model = lst[1].strip()
            break
    # storage_firmware
    firmware = NA
    for line in model_lst:
        lst = re.split(':', line.strip())
        if 'revision' in lst[0].lower():
            firmware = lst[1].strip()
            break
    # storage_serial
    serial = NA
    for line in serial_lst:
        if ('serial' in line.lower()) and ('no' in line.lower()):
            lst = re.split(':', line)
            serial = lst[1].strip()
            break
    storage_series = "VNX Series"
    if "CX" in model.upper():
        storage_series = "CX Series"
    value_lst = [NA, NA, "EMC", storage_series, model, firmware, serial, NA]
    return value_lst
