#  coding=UTF-8
#  Copyright (c) Huawei Technologies Co., Ltd. 2022-2022. All rights reserved.
from storages.storage_common import get_command_result_lines
from storages.storage_san_constant import HOST_NAME, LUN_GROUP_NAME, MV_ID, HOST_GROUP_NAME, HOST_LIST, LUN_GROUP_ID, \
    HOST_GROUP_ID

FUJISTU = "Fujistu"
HOST_MAPPING_KEY_WORDS = ["CM#", "(Host Affinity Mode Enable)"]
HOST_GROUP_MAPPING_KEY_WORDS = ["Port Group", "Host Group", "LUN Group", "LUN Overlap"]
HOST_MAPPING_KEY = "HOST_MAPPING_KEY"
HOST_GROUP_MAPPING_KEY = "HOST_GROUP_MAPPING_KEY"
KEY_WORD_NO = "No."
KEY_WORD_VOLUMES = "Volumes"
KEY_WORD_VOLUME = "Volume"
KEY_WORD_NAME = "Name"
KEY_WORD_GROUP_NO = "Group No."
SPLIT_WORD = "---"


def get_lun_group_name_list(content_lines):
    """
    获取lun组名
    Affinity Group        LUN Overlap
    No.  Name             Volumes
    ---- ---------------- -----------
       0 lun_group1       Yes
       1 lun_group2       No
       2 lun_group3       No
    :param content_lines: 回文
    :return: lun组名
    """
    content_flag = False
    lun_group_names = []
    for line in content_lines:
        line = line.strip()
        if line.startswith(KEY_WORD_NO) and line.endswith(KEY_WORD_VOLUMES):
            content_flag = True
            continue
        items = line.split()
        if not content_flag or is_split_line(line) or len(items) != 3:
            continue
        lun_group_names.append(items[1])
    return lun_group_names


def get_mapping_data(context):
    """
    主机/主机组 为维度的视图
    :param context: 上下文
    :return: 视图
    """
    host_group_mapping_info, host_mapping_info = get_mapping_info_list(context)
    index, host_group_mapping = build_host_group_mapping(host_group_mapping_info)
    host_mapping = build_host_mapping(host_mapping_info, index)
    return host_mapping, host_group_mapping


def get_mapping_info_list(context):
    """
    获取 主机/主机组 为维度的 视图回文
    Port Group           Host Group           LUN Group             LUN Overlap
    No. Name             No. Name             No.  Name             Volumes
    --- ---------------- --- ---------------- ---- ---------------- -----------
      0 name1             3  g_name1             7 test             No
    <Connection List>
      Port             Host
                       No.  Name
      ---------------- ---- ----------------
      CM#0 CA#0 Port#1    6 host_name1
      CM#0 CA#0 Port#1    7 host_name12
      CM#1 CA#0 Port#0    6 host_name1
      CM#1 CA#0 Port#0    7 host_name12

    CM#0 CA#0 Port#0 (Host Affinity Mode Enable)
    Host                  LUN Group             LUN Overlap LUN Mask
    No.  Name             No.  Name             Volumes     Group No.
    ---- ---------------- ---- ---------------- ----------- ---------
       1 host1               1 lun_group1       No                  -
      20 host12              9 lun_group2       No                  -

    :param context: 上下文
    :return: 主机/主机组 为维度的 视图回文
    """
    host_group_mapping_info, host_mapping_info, one_mapping, mapping_key = [], [], [], ""
    mapping_view_lines = get_command_result_lines(context, "show host-affinity", True)
    for line in mapping_view_lines:
        if not line.strip() and mapping_key and one_mapping:
            if mapping_key == HOST_MAPPING_KEY:
                host_mapping_info.append(one_mapping)
            elif mapping_key == HOST_GROUP_MAPPING_KEY:
                host_group_mapping_info.append(one_mapping)
            one_mapping, mapping_key = [], ""
            continue
        if not mapping_key:
            mapping_key = check_mapping_content(line)
        one_mapping.append(line)
    return host_group_mapping_info, host_mapping_info


def check_mapping_content(line):
    """
    检查 主机/主机组 维度的映射
    :param line: 回文
    :return: 主机/主机组 维度的映射
    """
    if check_line_key_words(line, HOST_GROUP_MAPPING_KEY_WORDS):
        return HOST_GROUP_MAPPING_KEY
    if check_line_key_words(line, HOST_MAPPING_KEY_WORDS):
        return HOST_MAPPING_KEY
    return ""


def check_line_key_words(line, key_words):
    """
    检查 行数据包含指定关键字
    :param line: 回文
    :param key_words: 关键字
    :return: 是否包含指定关键字
    """
    for key_word in key_words:
        if key_word not in line:
            return False
    return True


def build_host_mapping(host_mapping_info, index):
    """
    添加主机维度的映射
    CM#0 CA#0 Port#0 (Host Affinity Mode Enable)
    Host                  LUN Group             LUN Overlap LUN Mask
    No.  Name             No.  Name             Volumes     Group No.
    ---- ---------------- ---- ---------------- ----------- ---------
       1 host1               1 lun_group1       No                  -
      20 host12              9 lun_group2       No                  -
    :param host_mapping_info: 主机的映射信息
    :param index: 视图id
    :return: 主机映射 [{HOST_NAME: host_name, LUN_GROUP_ID: lun_group_id, LUN_GROUP_NAME: lun_group_name, MV_ID: 1},]
    """
    host_mapping = []
    for one_host_info in host_mapping_info:
        one_host_mapping, index = get_one_host_mapping(one_host_info, index)
        host_mapping.extend(one_host_mapping)
    return host_mapping


def get_one_host_mapping(one_host_info, index):
    """
    获取一个主机的映射
    :param one_host_info: 主机的映射信息
    :param index: 视图id
    :return: 主机映射 [{HOST_NAME: host_name, LUN_GROUP_ID: lun_group_id, LUN_GROUP_NAME: lun_group_name, MV_ID: 1},]
    """
    host_name, lun_group_name, host_flag, host_mapping = "", "", False, []
    for line in one_host_info:
        line = line.strip()
        if not host_flag and check_line(line, KEY_WORD_NO, KEY_WORD_GROUP_NO):
            host_flag = True
            continue
        if is_split_line(line):
            continue
        items = line.split()
        if host_flag and len(items) == 6:
            host_name, lun_group_id, lun_group_name = items[1], items[2], items[3]
            if is_exist_host_mapping(host_name, lun_group_name, host_mapping):
                continue
            host_mapping.append(
                {HOST_NAME: host_name, LUN_GROUP_ID: lun_group_id, LUN_GROUP_NAME: lun_group_name, MV_ID: index})
            index += 1
    return host_mapping, index


def is_exist_host_mapping(host_name, lun_group_name, host_mapping):
    """
    映射视图是否已存在
    :param host_name: 主机名
    :param lun_group_name: lun组名
    :param host_mapping: 主机视图信息
    :return: 是否已存在
    """
    for mapping in host_mapping:
        if host_name == mapping.get(HOST_NAME, "") and lun_group_name == mapping.get(LUN_GROUP_NAME, ""):
            return True
    return False


def build_host_group_mapping(host_group_mapping_info):
    """
    添加主机组维度的映射
    :param host_group_mapping_info: 主机组的映射信息
    :return: 视图id
    """
    index, group_mapping = 1, []
    for one_group_info in host_group_mapping_info:
        host_group_data = parse_host_group_data(one_group_info)
        if is_exist_host_group_mapping(host_group_data.get(HOST_GROUP_NAME, ""),
                                       host_group_data.get(LUN_GROUP_NAME, ""), group_mapping):
            continue
        host_group_data[MV_ID] = index
        group_mapping.append(host_group_data)
        index += 1
    return index, group_mapping


def parse_host_group_data(one_group_info):
    """
    解析一个主机组映射数据
    Port Group           Host Group           LUN Group             LUN Overlap
    No. Name             No. Name             No.  Name             Volumes
    --- ---------------- --- ---------------- ---- ---------------- -----------
      0 name1             3  g_name1             7 test             No
    <Connection List>
      Port             Host
                       No.  Name
      ---------------- ---- ----------------
      CM#0 CA#0 Port#1    6 host_name1
      CM#0 CA#0 Port#1    7 host_name12
      CM#1 CA#0 Port#0    6 host_name1
      CM#1 CA#0 Port#0    7 host_name12
    :param one_group_info: 一个主机组的映射回文
    :return: 主机组映射 {HOST_GROUP_NAME: host_group_name, LUN_GROUP_ID: lun_group_id,
                       LUN_GROUP_NAME: lun_group_name, HOST_LIST:[host1,host2,]}
    """
    group_flag, host_flag, host_list, host_group_data = False, False, [], {}
    for line in one_group_info:
        line = line.strip()
        if (not group_flag and not host_flag) and check_line(line, KEY_WORD_NO, KEY_WORD_VOLUMES):
            group_flag = True
            continue
        if is_split_line(line):
            continue
        items = line.split()
        if group_flag and len(items) == 7:
            host_group_id, host_group_name, lun_group_id, lun_group_name = items[2], items[3], items[4], items[5]
            host_group_data = {HOST_GROUP_ID: host_group_id, HOST_GROUP_NAME: host_group_name,
                               LUN_GROUP_ID: lun_group_id, LUN_GROUP_NAME: lun_group_name}
            group_flag = False
            continue
        if (not group_flag and not host_flag) and check_line(line, KEY_WORD_NO, KEY_WORD_NAME):
            host_flag = True
            continue
        if host_flag and len(items) > 2 and items[-1] not in host_list:
            host_list.append(items[-1])
    host_group_data[HOST_LIST] = host_list
    return host_group_data


def is_exist_host_group_mapping(host_group_name, lun_group_name, group_mapping):
    """
    映射视图是否已存在
    :param host_group_name: 主机组名
    :param lun_group_name: lun组名
    :param group_mapping: 主机组视图信息
    :return: 是否已存在
    """
    for mapping in group_mapping:
        if host_group_name == mapping.get(HOST_GROUP_NAME, "") and lun_group_name == mapping.get(LUN_GROUP_NAME, ""):
            return True
    return False


def check_line(line, starter, ender):
    """
    检查回文是否为指定的 开始/结束
    :param line: 回文行
    :param starter: 起始
    :param ender: 结束
    :return: 是否为指定的 开始/结束
    """
    line = line.strip()
    return line.startswith(starter) and line.endswith(ender)


def is_split_line(line):
    """
    是否是分割符行
    :param line: 回文行
    :return: 是否是分割符行
    """
    line = line.strip()
    return line.startswith(SPLIT_WORD) and line.endswith(SPLIT_WORD)
