#  coding=UTF-8
#  Copyright (c) Huawei Technologies Co., Ltd. 2022-2022. All rights reserved.
from storages.Fujistu.fujistu_san_util import get_mapping_data, check_line, KEY_WORD_NO, is_split_line, \
    KEY_WORD_VOLUMES, KEY_WORD_VOLUME
from storages.storage_common import get_storage_format_info, get_mapping_view_name, NA, get_command_result_lines, \
    LunInfo
from storages.storage_san_constant import LUN_GROUP_NAME, MV_ID, LUN_GROUP_ID


def execute(context):
    return get_storage_format_info(context, parse_lun_info, [LunInfo().get_lun_info_dict()], False)


def parse_lun_info(context):
    """
    LUN解析（映射视图中的）
    :param context: 上下文数据
    :return: [{}, ...]
    """
    lun_group_data = get_lun_group_data(context)
    host_mapping, group_mapping = get_mapping_data(context)
    lun_list = []
    lun_list.extend(parse_lun_for_host_mapping(host_mapping, lun_group_data))
    lun_list.extend(parse_lun_for_host_mapping(group_mapping, lun_group_data))
    return lun_list


def get_one_lun_group_data(context, lun_group_name, all_lun_data):
    """
    解析一个lun组信息
    :param context: 上下文
    :param lun_group_name: lun组名
    :param all_lun_data: 所有lun数据
    :return: 一个lun组信息
    """
    lun_group_info_lines = get_command_result_lines(context, "show affinity-groups -ag-name %s" % lun_group_name)
    content_flag, lun_list = False, []
    for line in lun_group_info_lines:
        if check_line(line, KEY_WORD_NO, KEY_WORD_VOLUME):
            content_flag = True
            continue
        if not content_flag or is_split_line(line):
            continue
        items = line.split()
        if len(items) != 6:
            continue
        lun_list.append(all_lun_data.get(items[2], LunInfo()))
    return lun_list


def get_lun_group_data(context):
    """
    lun组信息解析
    :param context: 上下文
    :return: {lun_group_name1:[lun1,lun2..],lun_group_name2..}
    """
    all_lun_data = get_all_lun_data(context)
    lun_group_names = get_lun_group_names(context)
    lun_group_data = {}
    for lun_group_name in lun_group_names:
        lun_group_data[lun_group_name] = get_one_lun_group_data(context, lun_group_name, all_lun_data)
    return lun_group_data


def get_lun_group_names(context):
    """
    获取lun组名字
    :param context: 上下文
    :return: lun组名字
    """
    lun_group_info_lines = get_command_result_lines(context, "show affinity-groups")
    content_flag = False
    for line in lun_group_info_lines:
        if check_line(line, KEY_WORD_NO, KEY_WORD_VOLUMES):
            content_flag = True
            continue
        if not content_flag or is_split_line(line):
            continue
        items = line.split()
        if len(items) != 3:
            continue
        lun_group_info_lines.append(items[1])
    return lun_group_info_lines


def get_all_lun_data(context):
    """
    lun详细信息解析
    :param context: 上下文
    :return: {lun_name1:LunInfo(),lun_name2:}
    """
    lun_info_lines = get_command_result_lines(context, "show volumes -mode uid")
    content_flag, lun_data = False, {}
    for line in lun_info_lines:
        if check_line(line, KEY_WORD_NO, "Mode"):
            content_flag = True
            continue
        if not content_flag or is_split_line(line):
            continue
        items = line.split()
        if len(items) != 11:
            continue
        lun = LunInfo()
        lun.set_lun_id(items[0])
        lun.set_lun_name(items[1])
        lun.set_status(items[2])
        lun.set_lun_type(items[3])
        lun.set_capacity(items[8] + "MB")
        lun.set_lun_wwn(items[9])
        lun_data[lun.get_lun_name()] = lun
    return lun_data


def parse_lun_for_host_mapping(host_mapping, lun_group_data):
    """
    解析主机维度的视图
    :param host_mapping: 映射视图
    :param lun_group_data: 所有lun组数据
    :return: lun数据[{}, ...]
    """
    lun_list = []
    for host_data in host_mapping:
        lun_group_name = host_data.get(LUN_GROUP_NAME, "")
        mv_id = host_data.get(MV_ID, 0)
        if lun_group_name not in lun_group_data:
            continue
        lun_list_in_group = lun_group_data.get(lun_group_name)
        lun_group_id = host_data.get(LUN_GROUP_ID, NA)
        for lun in lun_list_in_group:
            lun_tmp = lun.copy_lun()
            lun_tmp.set_mapping_view_id(mv_id)
            lun_tmp.set_mapping_view_name(get_mapping_view_name(mv_id))
            lun_tmp.set_lun_group_name(lun_group_name)
            lun_tmp.set_lun_group_id(lun_group_id)
            lun_list.append(lun_tmp.get_lun_info_dict())
    return lun_list
