#  coding=UTF-8
#  Copyright (c) Huawei Technologies Co., Ltd. 2019-2023. All rights reserved.

import traceback

from common.cmd_execute import DES, CMD
from storages.HUAWEI.huawei_storage_util import (execute_command, execute_cmd_list, OCEAN_STOR_V6,
                                                 parse_id_list_from_table_type_result)
from storages.HUAWEI.huawei_storage_util import batch_execute_command
from storages.HUAWEI.huawei_storage_util import do_execute_error
from storages.HUAWEI.huawei_storage_util import parse_table_type_result
from storages.HUAWEI.huawei_storage_util import get_device_class

tv1_host_info_id = [
    "cmd_info_host_groups",
    "cmd_info_host_by_group_%s",
    "cmd_info_initiator_by_host_%s",
]
tv1_host_info_cmd = [
    "showhostgroup",
    "showhost -group %s",
    "showhostport -host %s",
]

common_host_info_id = [
    "cmd_info_mapping_view",
    "cmd_info_host_group_by_mapping_%s",
    "cmd_info_host_by_group_%s",
    "cmd_info_initiator_by_host_%s",
]
common_host_info_cmd = [
    "show mapping_view general",
    "show mapping_view host_group mapping_view_id=%s",
    "show host_group host host_group_id=%s",
    "show initiator host_id=%s",
]

common_cmd_list = [
    {DES: "cmd_info_iscsi_initiator", CMD: "show initiator initiator_type=iSCSI"},
    {DES: "cmd_info_fc_initiator", CMD: "show initiator initiator_type=FC"},
]

oceanstor_v6_host_info_id = [
    "cmd_info_host_group_general",
    "cmd_info_hosts_in_host_group_%s",
    "cmd_info_mapping_in_host_group_%s",
    "cmd_info_host_general",
    "cmd_info_mapping_general_host_%s",
    "cmd_info_host_general_host_%s",
]
oceanstor_v6_host_cmd = [
    "show host_group general",
    "show host_group host host_group_id=%s",
    "show mapping general host_group_id=%s",
    "show host general",
    "show mapping general host_id=%s",
    "show host general host_id=%s",
]


def execute(context):
    """
    获取映射信息入口方法
    :param context: 上下文
    :return: 映射信息全部回显
    """
    try:
        dev_class = get_device_class(context)
        if dev_class == "TV1":
            query_tv1_host_group_info(context)
        if dev_class == "COMMON":
            query_common_host_group_info(context)
            grab_port(context)
        if dev_class == OCEAN_STOR_V6:
            query_oceanstor_v6_host_info(context)
            grab_port(context)
        return context.get("ret_map")
    except Exception:
        do_execute_error(context, str(traceback.format_exc()))


def grab_port(context):
    port_info = execute_command(context, "cmd_info_port_general", "show port general")
    port_lines = port_info.splitlines()
    port_types = ['ETH', 'FC', 'FCoE']
    for port_type in port_types:
        grab_port_initiators(context, port_type, port_lines)


def grab_port_initiators(context, port_type, port_lines):
    start = False
    for line in port_lines:
        if not line or not start:
            if port_type + ' port:' in line:
                start = True
            continue
        if ' port:' in line:
            break
        if '----' in line or 'Health Status' in line:
            continue
        port_id = line.split()[0]
        execute_command(context, 'cmd_info_port_initiator_%s_%s' % (port_type, port_id),
                        'show port initiator port_type=%s port_id=%s' % (port_type, port_id))


def query_oceanstor_v6_host_info(context):
    """
    Dorado V6+新融合 主机信息获取
    :param context: 上下文
    """
    execute_cmd_list(context, common_cmd_list)
    host_group_ids = get_id_list(context, oceanstor_v6_host_info_id[0], oceanstor_v6_host_cmd[0], "ID")
    if host_group_ids:
        batch_execute_command(context, host_group_ids, oceanstor_v6_host_info_id[1], oceanstor_v6_host_cmd[1])
        batch_execute_command(context, host_group_ids, oceanstor_v6_host_info_id[2], oceanstor_v6_host_cmd[2])
    host_ids = get_id_list(context, oceanstor_v6_host_info_id[3], oceanstor_v6_host_cmd[3], "ID")
    if host_ids:
        batch_execute_command(context, host_ids, oceanstor_v6_host_info_id[4], oceanstor_v6_host_cmd[4])
        batch_execute_command(context, host_ids, oceanstor_v6_host_info_id[5], oceanstor_v6_host_cmd[5])


def get_id_list(context, cmd_info, cmd, key):
    """
    Dorado V6+新融合 获取host_group_id/host_id
    :param context: 上下文
    :param cmd_info: 命令描述
    :param cmd: 命令
    :param key: 表头
    :return: id列表
    """
    ids = []
    for one_dict in parse_table_type_result(execute_command(context, cmd_info, cmd).splitlines()):
        ids.append(one_dict.get(key, ""))
    return ids


def query_tv1_host_group_info(context):
    """
    TV1设备映射信息获取
    :param context: 上下文
    :return: 命令回显
    """
    host_group_result = execute_command(context, tv1_host_info_id[0], tv1_host_info_cmd[0])
    host_group_dicts = parse_table_type_result(host_group_result.splitlines())
    host_group_ids = []
    for host_group_dict in host_group_dicts:
        host_group_ids.append(host_group_dict.get("Host Group ID", ""))
    host_results = batch_execute_command(context, host_group_ids, tv1_host_info_id[1], tv1_host_info_cmd[1])
    query_tv1_host_initiator(context, host_results)
    execute_command(context, "cmd_info_hosts", "showhost")
    execute_command(context, "cmd_info_port_general", "showport")


def query_tv1_host_initiator(context, all_group_host_results):
    """
    查询主机启动器信息
    :param context: 上下文
    :param all_group_host_results: 所有主机组中主机的回显信息
    :return: none
    """
    host_ids = []
    for group_host_result in all_group_host_results:
        host_dicts = parse_table_type_result(group_host_result.splitlines())
        for host_dict in host_dicts:
            host_ids.append(host_dict.get("Host ID", ""))
    batch_execute_command(context, host_ids, tv1_host_info_id[2], tv1_host_info_cmd[2])


def query_common_host_group_info(context):
    """
    TV2之后非V6设备映射信息获取
    :param context: 上下文
    :return: 命令回显
    """
    mapping_result = execute_command(context, common_host_info_id[0], common_host_info_cmd[0])
    mapping_dicts = parse_table_type_result(mapping_result.splitlines())
    mapping_ids = []
    for mapping_dict in mapping_dicts:
        mapping_ids.append(mapping_dict.get("Mapping View ID", ""))
    query_common_host_info(context, mapping_ids)


def query_common_host_info(context, mapping_ids):
    """
    TV2之后非V6设备主机信息获取
    :param context: 上下文
    :param mapping_ids: 映射视图ID
    :return: none
    """
    hg_results = batch_execute_command(context, mapping_ids, common_host_info_id[1], common_host_info_cmd[1])
    host_group_ids = []
    for hg_result in hg_results:
        host_group_dicts = parse_table_type_result(hg_result.splitlines())
        for host_group_dict in host_group_dicts:
            host_group_id = host_group_dict.get("Host Group ID", "")
            if host_group_id not in host_group_ids:
                host_group_ids.append(host_group_id)

    host_results = batch_execute_command(context, host_group_ids, common_host_info_id[2], common_host_info_cmd[2])
    host_ids = parse_id_list_from_table_type_result(host_results, "ID", True)

    batch_execute_command(context, host_ids, common_host_info_id[3], common_host_info_cmd[3])
    execute_command(context, "cmd_info_host_general", "show host general")
    execute_cmd_list(context, common_cmd_list)
