#  coding=UTF-8
#  Copyright (c) Huawei Technologies Co., Ltd. 2019-2023. All rights reserved.

import traceback

from common.cmd_execute import DES, CMD
from storages.HUAWEI.huawei_storage_host_info import get_id_list
from storages.HUAWEI.huawei_storage_util import (execute_command, execute_cmd_list, OCEAN_STOR_V6,
                                                 parse_id_list_from_table_type_result)
from storages.HUAWEI.huawei_storage_util import batch_execute_command
from storages.HUAWEI.huawei_storage_util import do_execute_error
from storages.HUAWEI.huawei_storage_util import parse_table_type_result
from storages.HUAWEI.huawei_storage_util import get_device_class

tv1_lun_info_id = [
    "cmd_info_lun_map",
    "cmd_info_lun_detail",
    "cmd_info_show_lun_detail_%s"
]
tv1_lun_info_cmd = [
    "showhostmap",
    "showlun",
    "showlun -lun %s"
]

common_mapping_info_id = [
    "cmd_info_mapping_view",
    "cmd_info_lun_group_by_mapping_%s",
    "cmd_info_lun_by_group_%s",
    "cmd_info_lun_general_lun_%s"
]
common_mapping_cmd = [
    "show mapping_view general",
    "show mapping_view lun_group mapping_view_id=%s",
    "show lun_group lun lun_group_id=%s",
    "show lun general lun_id=%s"
]

common_cmd_list = [
    {DES: "cmd_info_lun_takeover", CMD: "show lun_takeover general"},
]

oceanstor_v6_mapping_info_id = [
    "cmd_info_lun_general",
    "cmd_info_lun_group_general",
    "cmd_info_luns_in_lun_group_%s",
]
oceanstor_v6_mapping_cmd = [
    "show lun general",
    "show lun_group general",
    "show lun_group lun lun_group_id=%s",
]


def execute(context):
    """
    获取映射信息入口方法
    :param context: 上下文
    :return: 映射信息全部回显
    """
    try:
        dev_class = get_device_class(context)
        if dev_class == "TV1":
            query_tv1_lun_info(context)
            return
        execute_cmd_list(context, common_cmd_list)
        if dev_class == "COMMON":
            query_common_lun_group_info(context)
        if dev_class == OCEAN_STOR_V6:
            query_oceanstor_v6_lun_info(context)
        return context.get("ret_map")
    except Exception:
        do_execute_error(context, str(traceback.format_exc()))


def query_oceanstor_v6_lun_info(context):
    """
    Dorado V6+新融合 设备映射信息获取
    :param context: 上下文
    """
    lun_results = execute_command(context, oceanstor_v6_mapping_info_id[0], oceanstor_v6_mapping_cmd[0])

    lun_ids = parse_id_list_from_table_type_result(lun_results, "ID", False)
    if lun_ids:
        batch_execute_command(context, lun_ids, common_mapping_info_id[3], common_mapping_cmd[3])

    lun_group_ids = get_id_list(context, oceanstor_v6_mapping_info_id[1], oceanstor_v6_mapping_cmd[1], "ID")
    if lun_group_ids:
        batch_execute_command(context, lun_group_ids, common_mapping_info_id[2], common_mapping_cmd[2])


def query_tv1_lun_info(context):
    """
    TV1设备映射信息获取
    :param context: 上下文
    :return: 命令回显
    """
    execute_command(context, tv1_lun_info_id[0], tv1_lun_info_cmd[0])
    lun_info_result = execute_command(context, tv1_lun_info_id[1], tv1_lun_info_cmd[1])
    lun_ids = parse_id_list_from_table_type_result(lun_info_result, "ID", False)
    batch_execute_command(context, lun_ids, tv1_lun_info_id[2], tv1_lun_info_cmd[2])


def query_common_lun_group_info(context):
    """
    TV2之后非V6设备映射信息获取
    :param context: 上下文
    :return: 命令回显
    """
    mapping_result = execute_command(context, common_mapping_info_id[0], common_mapping_cmd[0])
    mapping_dicts = parse_table_type_result(mapping_result.splitlines())
    mapping_ids = []
    for mapping_dict in mapping_dicts:
        mapping_ids.append(mapping_dict.get("Mapping View ID", ""))
    query_common_lun_info(context, mapping_ids)


def query_common_lun_info(context, mapping_ids):
    """
    TV2之后非V6设备LUN信息获取
    :param context: 上下文
    :param mapping_ids: 映射视图ID
    :return: none
    """
    lun_group_results = batch_execute_command(context, mapping_ids, common_mapping_info_id[1], common_mapping_cmd[1])
    lun_group_ids = []
    for lun_group_result in lun_group_results:
        lun_group_dicts = parse_table_type_result(lun_group_result.splitlines())
        for lun_group_dict in lun_group_dicts:
            lun_group_id = lun_group_dict.get("LUN Group ID", "")
            if lun_group_id not in lun_group_ids:
                lun_group_ids.append(lun_group_id)

    lun_results = batch_execute_command(context, lun_group_ids, common_mapping_info_id[2], common_mapping_cmd[2])
    lun_ids = parse_id_list_from_table_type_result(lun_results, "ID")

    batch_execute_command(context, lun_ids, common_mapping_info_id[3], common_mapping_cmd[3])
