#  coding=UTF-8
#  Copyright (c) Huawei Technologies Co., Ltd. 2019-2023. All rights reserved.

import re
from common.cmd_execute import DES, CMD
from storages.storage_common import get_resource, get_command_result_lines, NA, empty_value

OCEAN_STOR_V6 = "OceanStor_6.x"


def execute_command(context, command_id, command_desc):
    """
    执行华为存储cli命令并获取结果
    :param context: 上下文信息
    :param command_id: 命令Key
    :param command_desc: command
    :return: cmd_display
    """
    cli = context.get("SSH")
    language = context.get("lang")
    cmd_display = context.get("ret_map")
    fun_err_msg = cmd_display.get("err_msg")
    if not fun_err_msg:
        fun_err_msg = ""
    fun_err_msg += command_desc
    cmd_result = cli.execCmdHasLogTimout(command_desc, 50)
    cmd_display.put("cmd_display" + command_id[8:], cmd_result)
    if is_result_invalid(cmd_result):
        fun_err_msg += get_resource(language, "execute_fail")
    else:
        fun_err_msg += get_resource(language, "execute_success")
    cmd_display.put("err_msg", fun_err_msg)
    return cmd_result


def batch_execute_command(context, object_ids, command_id, command):
    """
    批量执行以ID关联的命令
    :param context: 上下文
    :param object_ids: 对象ID
    :param command_id: 命令ID
    :param command: 命令
    :return: 命令回显集合
    """
    cmd_result_list = []
    for obj_id in object_ids:
        if not obj_id:
            continue
        cmd_id = command_id % obj_id
        cmd_desc = command % obj_id
        cmd_result = execute_command(context, cmd_id, cmd_desc)
        cmd_result_list.append(cmd_result)
    return cmd_result_list


def do_execute_error(context, error_info):
    """
    处理执行异常场景回显信息
    :param context: 上下文信息
    :param error_info: 异常信息
    :return: 错误信息
    """
    language = context.get("lang")
    cmd_display = context.get("ret_map")
    logger = context.get("Logger")
    logger.error("get huawei info error, reason: %s" % error_info)
    fun_err_msg = get_resource(language, "execute_fail")
    cmd_display.put("err_msg", fun_err_msg)
    return cmd_display


def is_result_invalid(cmd_result):
    """
    判断命令执行结果是否非法
    :param cmd_result: 命令执行结果
    :return: 是否非法
    """
    return "not exist" in cmd_result


def parse_table_type_result(cmd_result_lines):
    """
    解析表格类型的回显信息
    返回表头-列信息关系的数组，数组每个字典对象表示一行信息
    :param cmd_result_lines: 命令回显行信息
    :return: [{header: value}, ...]
    """
    table_data = []
    header = []
    for result_line in cmd_result_lines:
        line = result_line.strip()
        if not line or is_separator_line(line):
            continue
        items = re.split("\\s{2,}", line)
        if len(items) == 1:
            continue
        if not header:
            header = items
            continue
        row_data = {}
        for i in range(min(len(items), len(header))):
            row_data[header[i]] = items[i]
        table_data.append(row_data)
    return table_data


def is_separator_line(line):
    line_val = line.strip().replace(" ", "").replace("=", "").replace("-", "")
    return not line_val


def parse_key_value_type_result(cmd_result_lines, separator):
    """
    解析键值对类型的回显信息
    :param cmd_result_lines: 命令回显行信息
    :param separator: 分隔符
    :return: 字典
    """
    data_dict = {}
    for line in cmd_result_lines:
        if separator in line:
            info_pair = line.split(separator)
            data_dict[info_pair[0].strip()] = info_pair[1].strip()
    return data_dict


def get_storage_series(dev_type):
    """
    TV2-非V6：获取设备系列
    :param dev_type: 设备类型
    :return: 设备系列
    """
    model_type = dev_type.split()[-1]
    series_format = "OceanStor %s Series"
    if model_type.startswith("S"):
        model_type = model_type.split("-")[0]
        if model_type.endswith("T"):
            return series_format % "T"
        if model_type.endswith("900"):
            return series_format % "Sx900"
        return series_format % "S"
    if model_type.startswith("18"):
        return series_format % "18000"
    if "Dorado" in dev_type:
        return series_format % ("Dorado " + model_type)
    return series_format % model_type


def get_device_class(context):
    """
    获取设备类型，例如：TV1、通用类（TV2、V3、V5）或DoradoV6等
    :param context: 上下文信息
    :return: 设备类型
    """
    dev_type = context.get("dev_type", "")
    version = context.get("version", "").upper()
    storage_series = get_storage_series(dev_type)
    if ("Dorado" in storage_series and "V6" in storage_series) or version.startswith("6."):
        return OCEAN_STOR_V6
    if ("T Series" in storage_series or "Sx900 Series" in storage_series or "S Series" in storage_series) \
            and not version.startswith("V2"):
        return "TV1"
    return "COMMON"


def parse_data_dict(info_keys, info_values):
    """
    将集合数据与key匹配为字典回显
    :param info_keys: 字典key
    :param info_values: 数据
    :return: dict
    """
    data_dict = {}
    if len(info_keys) == len(info_values):
        for i in range(len(info_keys)):
            data_dict[info_keys[i]] = info_values[i]
    return data_dict


def execute_cmd_list(context, common_cmd_list):
    """
    命令列表执行
    :param context: 上下文
    :param common_cmd_list: 命令列表
    """
    for item in common_cmd_list:
        execute_command(context, item.get(DES), item.get(CMD))


def get_ocenstor_v6_mapping_info(context):
    """
    Dorado V6+新融合 映射信息获取
    :param context: 上下文
    :return 映射信息
    """
    lun_group_id_name = get_lun_group_id_name(context)
    index, mapping_infos = get_host_group_mapping_info(context, lun_group_id_name)
    mapping_infos.extend(get_host_mapping_info(context, index, lun_group_id_name))
    return mapping_infos


def get_host_mapping_info(context, index, lun_group_id_name):
    """
    Dorado V6+新融合 主机场景映射信息获取(主机-lun组、主机-luns)
    :param context: 上下文
    :param index: 映射id编号
    :param lun_group_id_name: lun组 ID-Name映射
    :return 映射信息
    """
    host_lines = get_command_result_lines(context, "show host general")
    host_dicts = parse_table_type_result(host_lines)
    mapping_infos = []
    for host_dict in host_dicts:
        mapping_info = OcenstorV6MappingInfo()
        mapping_info.host_id = host_dict.get("ID", NA)
        mapping_info.host_name = host_dict.get("Name", NA)
        mapping_info.type = host_dict.get("Operating System", NA)
        index, mappings = get_one_host_mapping(context, mapping_info, index, lun_group_id_name)
        mapping_infos.extend(mappings)
    return mapping_infos


def get_one_host_mapping(context, mapping_info, index, lun_group_id_name):
    """
    Dorado V6+新融合 一个主机的映射信息（每个lun组就是一个映射，所有lun id一起组成一个映射）
    :param context: 上下文
    :param mapping_info: 该组机组的信息
    :param index: 映射id编号
    :param lun_group_id_name: lun组 ID-Name映射
    :return 映射信息
    """
    host_mapping_lines = get_command_result_lines(context, "show mapping general host_id=%s" % mapping_info.host_id)
    host_mapping_dicts = parse_table_type_result(host_mapping_lines)
    mapping_infos = []
    lun_id_list = []
    for mapping_dict in host_mapping_dicts:
        mapping_type = mapping_dict.get("Mapping Type", "")
        if mapping_type and "Host Group" == mapping_type:
            continue
        lun_group_id = mapping_dict.get("LUN Group ID", "")
        lun_group_name = lun_group_id_name.get(lun_group_id, NA)
        lun_id = mapping_dict.get("LUN ID", "")
        if lun_group_id and empty_value != lun_group_id:
            mapping_info_temp = mapping_info.copy_mapping_info()
            mapping_info_temp.lun_group_id = lun_group_id
            mapping_info_temp.lun_group_name = lun_group_name
            mapping_info_temp.mapping_view_id = index
            mapping_infos.append(mapping_info_temp)
            index += 1
        if lun_id and empty_value != lun_id:
            lun_id_list.append(lun_id)
    if lun_id_list:
        mapping_info_temp = mapping_info.copy_mapping_info()
        mapping_info_temp.mapping_view_id = index
        mapping_info_temp.lun_id_list = lun_id_list
        mapping_infos.append(mapping_info_temp)
        index += 1
    return index, mapping_infos


def get_host_group_mapping_info(context, lun_group_id_name):
    """
    Dorado V6+新融合 主机组场景映射信息获取(主机组的映射，只有主机组-lun组)
    :param context: 上下文
    :param lun_group_id_name: lun组 ID-Name映射
    :return 映射信息
    """
    host_group_lines = get_command_result_lines(context, "show host_group general")
    host_group_dicts = parse_table_type_result(host_group_lines)
    mapping_infos = []
    index = 1
    for host_group_dict in host_group_dicts:
        mapping_info = OcenstorV6MappingInfo()
        mapping_info.host_group_id = host_group_dict.get("ID", NA)
        mapping_info.host_group_name = host_group_dict.get("Name", NA)
        index, mappings = get_one_host_group_mapping(context, mapping_info, index, lun_group_id_name)
        mapping_infos.extend(mappings)
    return index, mapping_infos


def get_one_host_group_mapping(context, mapping_info, index, lun_group_id_name):
    """
    Dorado V6+新融合 一个主机组的映射信息
    :param context: 上下文
    :param mapping_info: 该组机组的信息
    :param index: 映射id编号
    :param lun_group_id_name: lun组 ID-Name映射
    :return 映射信息
    """
    host_group_mapping_lines \
        = get_command_result_lines(context, "show mapping general host_group_id=%s" % mapping_info.host_group_id)
    host_group_mapping_dicts = parse_table_type_result(host_group_mapping_lines)
    mapping_infos = []
    for mapping_dict in host_group_mapping_dicts:
        lun_group_id = mapping_dict.get("LUN Group ID", "")
        lun_group_name = lun_group_id_name.get(lun_group_id, NA)
        if lun_group_id and empty_value != lun_group_id:
            mapping_info_temp = mapping_info.copy_mapping_info()
            mapping_info_temp.lun_group_id = lun_group_id
            mapping_info_temp.lun_group_name = lun_group_name
            mapping_info_temp.mapping_view_id = index
            mapping_infos.append(mapping_info_temp)
            index += 1
    return index, mapping_infos


def get_lun_group_id_name(context):
    """
    Dorado V6+新融合：lun组 ID-Name映射
    :param context: 上下文
    :return:lun组 ID-Name映射
    """
    lines = get_command_result_lines(context, "show lun_group general")
    lun_dicts = parse_table_type_result(lines)
    data = {}
    for lun_dict in lun_dicts:
        data[lun_dict.get("ID", NA)] = lun_dict.get("Name", NA)
    return data


def parse_id_list_from_table_type_result(info_query_result, id_key, is_batch_query_result=True):
    """
    从表结构查询结果中解析id列表
    :param info_query_result: 命令回显结果
    :param id_key: id对应列名称
    :param is_batch_query_result: 是否是批量查询结果
    :return: id列表
    """
    info_result_list = info_query_result if is_batch_query_result else [info_query_result]
    info_ids = []
    for info_result in info_result_list:
        info_dicts = parse_table_type_result(info_result.splitlines())
        for info_dict in info_dicts:
            info_id = info_dict.get(id_key)
            if info_id and info_id not in info_ids:
                info_ids.append(info_id)
    return info_ids


class OcenstorV6MappingInfo:
    def __init__(self):
        self.mapping_view_id = NA
        self.host_group_id = NA
        self.host_group_name = NA
        self.host_id = NA
        self.host_name = NA
        self.type = NA
        self.lun_group_id = NA
        self.lun_group_name = NA
        self.lun_id_list = []

    def copy_mapping_info(self):
        """
        复制数据模型
        :return: OcenstorV6MappingInfo
        """
        mapping_info = OcenstorV6MappingInfo()
        mapping_info.mapping_view_id = self.mapping_view_id
        mapping_info.host_group_id = self.host_group_id
        mapping_info.host_group_name = self.host_group_name
        mapping_info.host_id = self.host_id
        mapping_info.host_name = self.host_name
        mapping_info.type = self.type
        mapping_info.lun_group_id = self.lun_group_id
        mapping_info.lun_group_name = self.lun_group_name
        mapping_info.lun_id_list = self.lun_id_list
        return mapping_info
