#  coding=UTF-8
#  Copyright (c) Huawei Technologies Co., Ltd. 2023-2024. All rights reserved.
import re

from storages.HUAWEI.huawei_storage_util import parse_table_type_result
from storages.storage_common import get_command_result_lines


def parse_data(context, data_header, get_data_list_fun):
    """
    解析数据
    :param context: 上下文
    :param data_header: excel表头
    :param get_data_list_fun: 解析数据的方法
    """
    data_table = []
    data_table.extend(data_header)
    content = get_data_list_fun(context)
    data_table.extend(content)
    ret_map = context.get("map")
    ret_map.put("data", data_table)


def get_vstore_id_name_mapping(context):
    """
    获取租户id和name的映射(admin租户不在此映射内)
    :param context: 上下文
    :return: 数据
    """
    data_dicts = get_vstore_info(context)
    result_data = {}
    for data_dict in data_dicts:
        result_data[data_dict.get("ID", "")] = data_dict.get("Name", "")
    return result_data


def get_vstore_info(context):
    """
    获取租户信息
    :param context: 上下文
    :return: 租户信息
    """
    vstore_info_lines = get_command_result_lines(context, "cmd_display_admin_show_vstore")
    return parse_table_type_result(vstore_info_lines)


def get_file_system_id_vstore_id_mapping(context):
    """
    获取文件系统id和租户id的映射
    :param context: 上下文
    :return: 租户信息
    """
    data_dicts = get_file_system_info(context)
    result_data = {}
    for data_dict in data_dicts:
        result_data[data_dict.get("ID", "")] = data_dict.get("Vstore ID", "")
    return result_data


def get_file_system_id_name_mapping(context):
    """
    获取文件系统id和文件系统名的映射
    :param context: 上下文
    :return: 租户信息
    """
    data_dicts = get_file_system_info(context)
    result_data = {}
    for data_dict in data_dicts:
        result_data[data_dict.get("ID", "")] = data_dict.get("Name", "")
    return result_data


def get_file_system_info(context):
    """
    获取文件系统信息
    :param context: 上下文
    :return: 文件系统信息
    """
    file_system_info_lines = get_command_result_lines(context, "cmd_display_admin_file_system_general")
    return parse_table_type_result(file_system_info_lines)


def get_specific_columns_of_admin(context, cmd_des, column_name):
    """
    获取默认租户admin下的指定列集合 租户id为"--"的是默认租户
    场景一 文件系统
    ID  Name    ...   Vstore ID
    --  ------  ...   ---------
    0   LGU     ...   1
    1   fs001   ...   --
    场景二 逻辑端口
    ID  Name    ...   vstore ID
    --  ------  ...   ---------
    0   LGU     ...   1
    1   fs001   ...   --
    :param context: 上下文
    :param cmd_des: 命令des
    :param column_name: 列名
    :return: 租户信息
    """
    table_info_lines = get_command_result_lines(context, cmd_des)
    data_dicts = parse_table_type_result(table_info_lines)
    column_infos = []
    for data_dict in data_dicts:
        if data_dict.get("vStore ID") == "--" or data_dict.get("Vstore ID") == "--":
            column_infos.append(data_dict.get(column_name, "--"))
    return column_infos


def get_first_number_from_string(input_str):
    """
    获取文件系统信息, 如果传入的信息为空，返回0表示没有
    :param input_str: 上下文
    :return: input_str的第一个数字，若没有数字或input_str为空返回0
    """
    if not input_str:
        return 0
    pat = r'\d+\.\d+|\d+'
    search = re.search(pat, input_str)
    return float(search.group()) if search else 0


def vstore_id_to_sort(element):
    """
    data_content数据根据租户id排序
    :param element:
    :return: 返回整型id
    """
    return int(element[0]) if element[0] and element[0] != "--" else 0


def get_key_value_type_result(context, command, separator, with_blank=True):
    """
    从采集信息txt文件中读取回显信息
    :param context: 上下文数据
    :param command: 读取信息的命令
    :param separator: 分隔符
    :param with_blank:  是否保留前后空格
    :return: 字典
    """
    result_lines = get_command_result_lines(context, command, with_blank=with_blank)
    return parse_key_value_type_result(result_lines, separator)


def get_key_value_type_with_no_fixed_separator_result(context, command, separator, with_blank=True):
    """
    从采集信息txt文件中读取回显信息
    :param context: 上下文数据
    :param command: 读取信息的命令
    :param separator: 分隔符
    :param with_blank:  是否保留前后空格
    :return: 字典
    """
    result_lines = get_command_result_lines(context, command, with_blank=with_blank)
    return parse_key_value_type_with_no_fixed_speartor_result(result_lines, separator)


def get_key_value_type_with_no_fixed_separator_and_more_info_result(context, command, separator, with_blank=True):
    result_lines = get_command_result_lines(context, command, with_blank=with_blank)
    return parse_key_value_type_with_no_fixed_separator_and_more_info_result(result_lines, separator)


def get_key_value_type_with_more_info_result(context, command, separator, with_blank=True):
    result_lines = get_command_result_lines(context, command, with_blank=with_blank)
    return parse_key_value_type_with_more_info_result(result_lines, separator)


def get_table_type_cmd_with_more_info_result(context, command, with_blank=True):
    result_lines = get_command_result_lines(context, command, with_blank=with_blank)
    return parse_table_type_cmd_with_more_info_result(result_lines)


def get_table_type_result(context, command, with_blank=True):
    result_lines = get_command_result_lines(context, command, with_blank=with_blank)
    return parse_table_type_cmd_result(result_lines)


def is_separator_line(line):
    line_val = line.strip().replace(" ", "").replace("=", "").replace("-", "")
    return not line_val


def parse_table_type_cmd_with_more_info_result(cmd_result_lines):
    """
    解析可能有多行同一列表格类型的回显信息
    返回表头-列信息关系的数组，数组每个字典对象表示一行信息
    像下面这种，IP是多个的，需要归到上一层
    6    10gige-1 Up         502      groupnet0.subnet_prod.pool_data     Dynamic     127.0.0.1
                                                                                  127.0.0.1
    :param cmd_result_lines: 命令回显行信息
    :return: [{header: value}, ...]
    """
    table_data = []
    headers = []
    headers_line = ""
    for result_line in cmd_result_lines:
        line = result_line.strip()
        if not line or is_separator_line(line):
            continue
        items = re.split("\\s{2,}", line)
        if not headers and len(items) == 1:
            continue
        if not headers:
            headers = items
            headers_line = line
            continue
        table_data.append(parse_content_line(headers, headers_line, result_line))
    return table_data


def parse_key_value_type_result(cmd_result_lines, separator):
    """
    解析键值对类型的回显信息
    部分回显内容中带换行，会导致一行内容显示成多行，需要拼接多行内容为一行
    该方法默认处理内容格式是分隔符在同一位置
    :param cmd_result_lines: 命令回显行信息
    :param separator: 分隔符
    :return: 字典
    """
    data_dict = {}
    key = ''
    value = ''
    separator_index = 0
    for line in cmd_result_lines[1:]:
        if not line:
            continue
        if not separator_index and separator in line:
            # 获取分隔符的位置
            separator_index = line.index(separator.strip())
        if len(line) <= separator_index + 2:
            # 排除非法行：短于分隔符位置的行
            continue
        if separator in line and line.index(separator.strip()) == separator_index:
            # 分隔符位置匹配，是带Key的行，记录当前的Key/Value
            if key:
                data_dict[key] = value
            key = line[0:separator_index].strip()
            value = line[separator_index + 2:].strip()
        else:
            # 分隔符位置不匹配，属于上一行Value换行后的内容
            value += line[separator_index + 2:].strip()
    if key:
        data_dict[key] = value
    return data_dict


def parse_key_value_type_with_no_fixed_speartor_result(cmd_result_lines, separator):
    """
    解析键值对类型的回显信息
    部分回显内容中带换行，会导致一行内容显示成多行，需要拼接多行内容为一行
    该方法默认处理内容格式是分隔符在同一位置
    :param cmd_result_lines: 命令回显行信息
    :param separator: 分隔符
    :return: 字典
    """
    data_dict = {}
    key = ''
    value = ''
    separator_index = 0
    for line in cmd_result_lines[1:]:
        if not line:
            continue
        if separator not in line:
            continue
        # 获取分隔符的位置
        separator_index = line.index(separator.strip())
        key = line[0:separator_index].strip()
        value = line[separator_index + 2:]
        # 分隔符位置匹配，是带Key的行，记录当前的Key/Value
        if key:
            data_dict[key] = value
    return data_dict


def parse_key_value_type_with_no_fixed_separator_and_more_info_result(cmd_result_lines, separator):
    """
    用于解析有多段数据且有没固定分隔符位置的数据类型,如下：
        Name: System
    Path: /ifs
    --------------------------------------------------------------------------------
        Name: ZONE-MAIN
    Path: /ifs/isilon
    """
    data_dict = {}
    index = 0
    for line in cmd_result_lines[1:]:
        if not line:
            continue
        if separator not in line:
            continue
        separator_index = line.index(separator.strip())
        key = line[0:separator_index].strip()
        value = line[separator_index + 2:]
        if not key:
            continue
        if key + str(index) in data_dict:
            index += 1
        key = key + str(index)
        data_dict[key] = value
    return data_dict


def parse_key_value_type_with_more_info_result(cmd_result_lines, separator):
    """
    用于解析有多段数据且有固定分隔符位置的数据类型,如下：
        Name: System
        Path: /ifs
    --------------------------------------------------------------------------------
        Name: ZONE-MAIN
        Path: /ifs/isilon
    """
    data_dict = {}
    key = ''
    value = ''
    separator_index = 0
    index = 0
    last_len = 0
    for line in cmd_result_lines[1:]:
        if not line:
            continue
        if not separator_index and separator in line:
            # 获取分隔符的位置
            separator_index = line.index(separator.strip())
        if len(line) <= separator_index + 2:
            # 排除非法行：短于分隔符位置的行
            continue
        if separator not in line and len(line) > last_len:
            # 排除非法行，没有分隔符且大于上一行，一般来说是多段数据的分割符
            continue
        if separator in line and line.index(separator.strip()) == separator_index:
            # 分隔符位置匹配，是带Key的行，记录当前的Key/Value
            key = line[0:separator_index].strip()
            value = line[separator_index + 2:]
            if not key:
                continue
            if key + str(index) in data_dict:
                index += 1
            data_dict[key + str(index)] = value
        else:
            # 分隔符位置不匹配，属于上一行Value换行后的内容
            value += line[separator_index + 2:]
        last_len = len(line)
    if not key + str(index) in data_dict:
        data_dict[key + str(index)] = value
    return data_dict


def parse_table_type_cmd_result(cmd_result_lines):
    """
    解析表格类型的回显信息
    返回表头-列信息关系的数组，数组每个字典对象表示一行信息
    :param cmd_result_lines: 命令回显行信息
    :return: [{header: value}, ...]
    """
    table_data = []
    headers = []
    headers_line = ""
    for result_line in cmd_result_lines:
        line = result_line.strip()
        if not line or is_separator_line(line):
            continue
        items = re.split("\\s{2,}", line)
        if len(items) == 1:
            continue
        if not headers:
            headers = items
            headers_line = line
            continue
        table_data.append(parse_content_line(headers, headers_line, line))
    return table_data


def get_index_list_of_list(list_data):
    """
    解析表头，返回表头表项的下标地址
    eg:
    Zone ID    ID    Status
    这种无法直接通过index来获取Zone ID 和 ID的下标地址
    我们通过每次增加初始查询位置来对他进行操作
    """
    item_list = re.split("\\s{2,}", list_data)
    start = 0
    res_data = []
    for item in item_list:
        indexs = list_data.index(item, start)
        start = indexs + len(item)
        res_data.append(indexs)
    return res_data


def parse_content_line(headers, headers_line, content_line):
    """
    根据表格类型的回显信息的表头解析每一行表内容的字典数据
     ID          Zone ID  Description  External IP
     24326_root  root                  localhost
    :param headers:表头集合,例如：['ID', 'Zone ID', 'Description', 'External IP']
    :param headers_line:表头字符串，例如：'ID          Zone ID  Description  External IP'
    :param content_line:每一行数据，例如：'24326_root  root                  localhost'
    :return:解析后的每行数据的字典，例如：{'ID': '24326_root', 'Zone ID': 'root', 'Description': '', 'External IP': 'localhost'}
    """
    content = []
    header_indexes = get_index_list_of_list(headers_line)
    headers_length = len(headers)
    for index, header in enumerate(headers):
        line_bytes = content_line.encode('utf-8', 'ignore')
        if index != headers_length - 1:
            column = line_bytes[header_indexes[index]:header_indexes[index + 1]].decode('utf-8', 'ignore').strip()
            content.append(column)
        else:
            content.append(line_bytes[header_indexes[index]:].decode('utf-8', 'ignore').strip())
    return {headers[i]: content[i] for i in range(headers_length)}


def grouping_list_by_predicate(items, grouping_predicate):
    """
    对list内的对象进行分组
    以满足条件的对象进行分隔，第一个满足分组条件前面的数据为一个分组
    @param items:需要分组的集合
    @param grouping_predicate:分组条件
    @return:分组和的组列表
    """
    groups = []
    group = []
    for item in items:
        if grouping_predicate(item):
            groups.append(group)
            group = []
        group.append(item)
    groups.append(group)
    return groups


def get_sid_from_lines(info_lines):
    """
    从采集的信息行里解析Sid的前缀和后缀
    @param info_lines:信息行
    @return: Sid前缀和后缀元组,未找到时返回空串元组
    """
    sid = get_value_from_xml_lines(info_lines, "<sid>")
    index = sid.rfind("-")
    return sid[:index], sid[index + 1:] if sid else "", ""


def get_value_from_xml_lines(info_lines, xml_pre_label):
    """
    从采集的信息行里解析指定标签中的值
    @param xml_pre_label: xml行的前标签，例如<sid>
    @param info_lines:信息行。例如：['<result>', '<cmd>find-sid</cmd>', '<id>1</id>', '<sid>S-1</sid>', '</result>']
    @return: 标签行中的值
    """
    label_len = len(xml_pre_label)
    for info_line in info_lines:
        info_line = info_line.strip()
        if info_line.startswith(xml_pre_label):
            return info_line[label_len:-(label_len + 1)]
    return ""
