#  coding=UTF-8
#  Copyright (c) Huawei Technologies Co., Ltd. 2022-2023. All rights reserved.
from common.cmd_execute import DES, CMD, BaseCmdExecute
from storages.HUAWEI.huawei_storage_util import parse_table_type_result

DETAIL_DES = "detail_cmd_id"
DETAIL_CMD = "detail_cmd"
DETAIL_ID_KEY = "detail_id_key"


class HuaweiV3V5GrabInfo(BaseCmdExecute):
    def __init__(self, context):
        super(HuaweiV3V5GrabInfo, self).__init__(context)

    def execute_admin_cmd_with_detail(self, cmd_dicts, curr_step, each_step_width, is_developer_mode=False,
                                      timeout=60):
        """
        可以在开发者模式和非开发者模式下执行默认租户命令
        :param cmd_dicts: 命令列表
        :param curr_step: 当前阶段起始进度
        :param each_step_width: 当前阶段进度宽度
        :param is_developer_mode: 是否进入开发者模式执行命令
        :param timeout: 执行命令超过这个时间抛出异常超时
        """
        each_step_width = each_step_width / len(cmd_dicts) if cmd_dicts else each_step_width
        if is_developer_mode:
            self.change_to_developer()
        self.execute_each_cmd_with_detail(cmd_dicts, curr_step=curr_step, each_step_width=each_step_width,
                                          timeout=timeout)
        if is_developer_mode:
            self.exit_developer()

    def execute_vstore_cmd_with_detail(self, cmd_dicts, curr_step, each_step_width, is_developer_mode=False,
                                       timeout=60):
        """
        可以在开发者模式和非开发者模式下执行租户命令
        :param cmd_dicts: 命令列表
        :param curr_step: 当前阶段起始进度
        :param each_step_width: 当前阶段进度宽度
        :param is_developer_mode: 是否进入开发者模式执行命令
        :param timeout: 执行命令超过这个时间抛出异常超时, 默认60s
        """
        vstores = self.get_vstore()
        # 更新每步宽度，进度条每一步宽度先除以租户数，然后再除以每个租户下的的命令条数
        each_step_width = each_step_width / len(vstores) if vstores else each_step_width
        each_step_width = each_step_width / len(cmd_dicts) if cmd_dicts else each_step_width
        for vstore_id in vstores.values():
            self.change_vstore(vstore_id)
            if is_developer_mode:
                self.change_to_developer()
            curr_step = self.execute_each_cmd_with_detail(cmd_dicts, curr_step=curr_step,
                                                          each_step_width=each_step_width,
                                                          vstore_id=vstore_id, timeout=timeout)
            if is_developer_mode:
                self.exit_developer()
            self.exit_to_admin()

    def execute_each_cmd_with_detail(self, cmd_dicts, curr_step, each_step_width, vstore_id=None, timeout=60):
        """
        遍历命令字典执行每条详情命令
        :param cmd_dicts: 命令字典
        :param vstore_id: 租户ID，可为空
        :param curr_step: 当前阶段起始进度
        :param each_step_width: 当前阶段进度宽度
        :param timeout: 执行命令超过这个时间抛出异常超时，默认60s
        """
        for cmd_dict in cmd_dicts:
            self.execute_with_detail_cmd(cmd_dict, vstore_id=vstore_id, start_progress=curr_step,
                                         progress_width=each_step_width, timeout=timeout)
            curr_step += each_step_width
        return curr_step

    def execute_vstore_cmd_without_detail(self, cmd_dicts, cur_step, each_step, has_cmd_param=True,
                                          is_developer_mode=False):
        """
        执行不需要采集详情的租户命令，admin也是租户
        :param cmd_dicts: 命令字典
        :param cur_step: 当前阶段起始进度
        :param each_step: 当前阶段进度宽度
        :param has_cmd_param: 命令是否含有参数
        :param is_developer_mode: 是否进入开发者模式
        """
        vstores = self.get_vstore()
        each_step = each_step / len(vstores) if vstores else each_step
        for vstore_id in vstores.values():
            self.change_vstore(vstore_id)
            if is_developer_mode:
                self.change_to_developer()
            if has_cmd_param:
                cur_step = self.execute_cmds_with_params(cmd_dicts, (vstore_id,), (vstore_id,), cur_step, each_step)
            else:
                cur_step = self.execute_cmds_with_params(cmd_dicts, (vstore_id,), None, cur_step, each_step)
            if is_developer_mode:
                self.exit_developer()
            self.exit_to_admin()

    def execute_with_detail_cmd(self, cmd_dict, vstore_id=None, start_progress=0, progress_width=0, timeout=60):
        """
        {
            des: cmd_id
            cmd: cmd
            detail_des: detail_cmd_id
            detail_cmd: detail_cmd
            detail_id_key: 详细属性命令ID对应的表头Key
        }
        :param cmd_dict: 命令字典
        :param vstore_id: 租户ID，可为空
        :param start_progress: 当前阶段起始进度
        :param progress_width: 当前阶段进度宽度
        :param timeout: 执行命令超过这个时间抛出异常超时,默认60s
        """
        parent_des = cmd_dict[DES]
        parent_cmd = cmd_dict[CMD]
        child_des = cmd_dict[DETAIL_DES]
        child_cmd = cmd_dict[DETAIL_CMD]
        id_key = cmd_dict[DETAIL_ID_KEY]
        if vstore_id:
            parent_des = parent_des % vstore_id
        # 进度条1/10执行cmd，9/10执行detail_cmd
        curr_progress = start_progress
        curr_pgs_width = progress_width / 10.0
        self.execute_pure_cmd_list(commands=[{DES: parent_des, CMD: parent_cmd}], cur_step=curr_progress,
                                   progress_width=curr_pgs_width, timeout=timeout)
        curr_progress += curr_pgs_width
        curr_pgs_width = progress_width - curr_pgs_width
        data = parse_table_type_result(self.display.get(parent_des).splitlines())
        cmds = []
        for row in data:
            child_id = row.get(id_key)
            # key为空则退出
            if not child_id:
                break
            if vstore_id:
                cmds.append({DES: child_des % (vstore_id, child_id), CMD: child_cmd % child_id})
            else:
                cmds.append({DES: child_des % child_id, CMD: child_cmd % child_id})
        self.execute_pure_cmd_list(commands=cmds, cur_step=curr_progress, progress_width=curr_pgs_width,
                                   timeout=timeout)

    def execute_cmds_with_params(self, cmd_dicts, desc_param, cmd_param, curr_step, each_step_width):
        """
        执行带参数的命令，并更新进度条
        :param cmd_dicts: 命令列表
        :param desc_param: 命令描述占位参数，类型为元组
        :param cmd_param: 命令占位参数，类型为元组
        :param curr_step: 当前阶段起始进度
        :param each_step_width: 当前阶段进度宽度
        """
        for cmd_dict in cmd_dicts:
            if cmd_param:
                self.execute_cmd_and_put_result(cmd_dict[DES] % desc_param, cmd_dict[CMD] % cmd_param)
            else:
                self.execute_cmd_and_put_result(cmd_dict[DES] % desc_param, cmd_dict[CMD])
            curr_step = self.update_progress(curr_step, each_step_width)
            curr_step += each_step_width
        return curr_step

    def execute_cmds_without_params(self, cmd_dicts, curr_step, each_step_width, is_developer_mode=False):
        """
        执行不带参数的命令，并更新进度条
        :param cmd_dicts: 命令列表
        :param curr_step: 当前阶段起始进度
        :param each_step_width: 当前阶段进度宽度
        :param is_developer_mode: 是否进入开发者模式执行命令
        """
        # 更新每步宽度,除以要执行的的命令条数
        each_step_width = each_step_width / len(cmd_dicts) if cmd_dicts else each_step_width
        if is_developer_mode:
            self.change_to_developer()
        for cmd_dict in cmd_dicts:
            self.execute_cmd_and_put_result(cmd_dict[DES], cmd_dict[CMD])
            curr_step = self.update_progress(curr_step, each_step_width)
            curr_step += each_step_width
        if is_developer_mode:
            self.exit_developer()

    def execute_cmd_and_put_result(self, desc, cmd):
        cmd_display_temp = self.exec_cmd(cmd)
        self.display.put(desc, cmd_display_temp)
        self.check_result(cmd_display_temp, cmd)
        return cmd_display_temp

    @staticmethod
    def get_cmd_specific_column_info(content, column_name):
        data_dicts = parse_table_type_result(content.splitlines())
        column_infos = []
        for data_dict in data_dicts:
            column_infos.append(data_dict.get(column_name, "--"))
        return column_infos

    @staticmethod
    def filter_cmd_specific_column_info(content, column_name):
        """
        筛选出vstore_id为"--"的指定列的列名
        :param content: 内容
        :param column_name: 列名
        :return column_infos:
        """
        data_dicts = parse_table_type_result(content.splitlines())
        column_infos = []
        for data_dict in data_dicts:
            if data_dict.get("vStore ID") == "--" or data_dict.get("Vstore ID") == "--":
                column_infos.append(data_dict.get(column_name, "--"))
        return column_infos

    def get_vstore(self):
        result = self.exec_cmd("show vstore")
        data_dicts = parse_table_type_result(result.splitlines())
        result_data = {}
        for data_dict in data_dicts:
            result_data[data_dict.get("Name", "")] = data_dict.get("ID", "")
        return result_data

    def change_vstore(self, vstore_id):
        if vstore_id:
            self.exec_cmd("change vstore view id=" + vstore_id)

    def change_to_developer(self):
        result = self.cli.execCmdNoLogTimout("change user_mode current_mode user_mode=developer", 10)
        if "Have you read danger alert message carefully?(y/n)" in result:
            result = self.cli.execCmdNoLogTimout("y", 10)
            if "Are you sure you really want to perform the operation?(y/n)" in result:
                self.cli.execCmdNoLogTimout("y", 10)

    def exit_to_admin(self):
        result = self.exec_cmd("exit")
        if "exit?(y/n):" in result:
            self.exec_cmd("n")

    def exit_developer(self):
        """
        退出developer模式 执行空命令得到回显，然后判断当前租户是否在developer模式下
        在developer模式下就执行"exit"
        """
        result = self.exec_cmd("")
        if result.startswith("developer") and ":/>" in result:
            self.exec_cmd("exit")
