#  coding=UTF-8
#  Copyright (c) Huawei Technologies Co., Ltd. 2019-2023. All rights reserved.
import traceback
from storages.storage_common import format_analysis_result, get_mapping_view_name
from storages.storage_common import get_command_result_lines
from storages.storage_common import HostInfo
from storages.HUAWEI.huawei_storage_util import (parse_table_type_result, OCEAN_STOR_V6, get_ocenstor_v6_mapping_info,
                                                 parse_key_value_type_result)
from storages.HUAWEI.huawei_storage_util import get_device_class

NA = 'NA'
# TV2-非V6 access_mode取值映射
COMMON_ACCESS_MODE_DICT = {
    '--': 'AA',
    'No ALUA': 'AA',
    'Common ALUA': 'ALUA'
}
# Dorado V6+新融合 access_mode取值映射
V6_ACCESS_MODE_DICT = {
    "Asymmetric": "ALUA",
    "Balanced": "AA"
}
# TV1 access_mode取值映射
TV1_ACCESS_MODE_DICT = {
    "Default": "AA",
    "ALUA": "ALUA",
}



def execute(context):
    """
    入口执行方法
    :param context: 上下文数据
    :return: context
    """
    try:
        dev_class = get_device_class(context)
        if dev_class == "TV1":
            host_info_dicts = tv1_analysis_host_info(context)
        elif dev_class == "COMMON":
            host_info_dicts = common_analysis_host_info(context)
        elif dev_class == OCEAN_STOR_V6:
            host_info_dicts = oceanstor_v6_analysis_host_info(context)
        else:
            host_info_dicts = get_default_empty_host_info()
        return format_analysis_result(context, host_info_dicts)
    except Exception:
        context["error"] = "huawei analysis host info error: %s" % str(traceback.format_exc())
        return format_analysis_result(context, get_default_empty_host_info())


def get_default_empty_host_info():
    """
    获取默认的空白主机信息
    :return: [{...}]
    """
    return [HostInfo().get_host_info_dict(True)]


def oceanstor_v6_analysis_host_info(context):
    """
    Dorado V6+新融合：主机解析
    :param context: 上下文
    :return: [HostInfo, ...]
    """
    host_infos = get_oceanstore_v6_hosts(context)
    host_initiator_dict = get_host_initiator(context)
    hosts = []
    for host_info in host_infos:
        host_general_lines = get_command_result_lines(context, "cmd_display_host_general_host_%s"
                                                      % host_info.get_host_id())
        host_general_dict = parse_key_value_type_result(host_general_lines, " : ")
        access_mode_str = host_general_dict.get('Access Mode', NA)
        host_info.set_access_model(V6_ACCESS_MODE_DICT.get(access_mode_str) or access_mode_str)
        hosts.extend(common_analysis_initiator_for_host(host_initiator_dict, host_info))
    return convert_hosts_to_dicts(hosts)


def get_oceanstore_v6_hosts(context):
    """
    Dorado V6+新融合：主机解析
    :param context: 上下文
    :return: [HostInfo, ...]
    """
    mapping_infos = get_ocenstor_v6_mapping_info(context)
    hosts = []
    for mapping_info in mapping_infos:
        hosts.extend(get_one_mapping_hosts(context, mapping_info))
    return hosts


def get_one_mapping_hosts(context, mapping_info):
    """
    Dorado V6+新融合：一个映射下的主机解析
    :param context: 上下文
    :param mapping_info: 映射
    :return: [HostInfo, ...]
    """
    if NA != mapping_info.host_group_id:
        return get_one_host_group_mapping_hosts(context, mapping_info)
    if NA != mapping_info.host_id:
        return get_one_host_mapping_hosts(mapping_info)
    return []


def get_one_host_mapping_hosts(mapping_info):
    """
    Dorado V6+新融合：主机映射场景下的主机解析
    :param mapping_info: 上下文
    :return: [HostInfo, ...]
    """
    host_info = HostInfo()
    host_info.set_host_id(mapping_info.host_id)
    host_info.set_host_name(mapping_info.host_name)
    host_info.set_host_type(mapping_info.type)
    mapping_view_id = mapping_info.mapping_view_id
    host_info.set_mapping_view_id(mapping_view_id)
    host_info.set_mapping_view_name(get_mapping_view_name(mapping_view_id))
    return [host_info]


def get_one_host_group_mapping_hosts(context, mapping_info):
    """
    Dorado V6+新融合：主机组映射场景下的主机解析
    :param context: 上下文
    :param mapping_info: 上下文
    :return: [HostInfo, ...]
    """
    lines = get_command_result_lines(context, "show host_group host host_group_id=%s" % mapping_info.host_group_id)
    host_dicts = parse_table_type_result(lines)
    host_infos = []
    for host_dict in host_dicts:
        host_info = HostInfo()
        host_info.set_host_id(host_dict.get("ID", NA))
        host_info.set_host_name(host_dict.get("Name", NA))
        host_info.set_host_type(host_dict.get("Operating System", NA))
        host_info.set_host_group_id(mapping_info.host_group_id)
        host_info.set_host_group_name(mapping_info.host_group_name)
        mapping_view_id = mapping_info.mapping_view_id
        host_info.set_mapping_view_id(mapping_view_id)
        host_info.set_mapping_view_name(get_mapping_view_name(mapping_view_id))
        host_infos.append(host_info)
    return host_infos


def get_host_initiator(context):
    """
    主机启动器解析
    :param context: 上下文
    :return: [HostInfo, ...]
    """
    host_initiator_dict = {}
    common_analysis_host_initiator(context, "cmd_display_fc_initiator", "WWN", host_initiator_dict)
    common_analysis_host_initiator(context, "cmd_display_iscsi_initiator", "iSCSI IQN", host_initiator_dict)
    return host_initiator_dict


def tv1_analysis_host_info(context):
    """
    TV1设备，主机组映射信息解析
    :param context: 上下文数据
    :return: [{}, ...]
    """
    host_initiators = tv1_analysis_host_initiator(context)
    host_info_dicts = []
    for host_initiator in host_initiators:
        host_info_dicts.append(host_initiator.get_host_info_dict(True))
    private_host_maps = tv1_extend_private_host_map(context, host_initiators)
    for private_host_map in private_host_maps:
        host_info_dicts.append(private_host_map.get_host_info_dict(True))
    return host_info_dicts


def tv1_analysis_host_initiator(context):
    """
    TV1: 解析主机启动器信息
    :param context: 上下文
    :return: [HostInfo, ...]
    """
    host_infos = tv1_analysis_host(context)
    host_initiators = []
    for host_info in host_infos:
        host_initiator_cmd = "cmd_display_initiator_by_host_%s" % host_info.get_host_id()
        initiator_lines = get_command_result_lines(context, host_initiator_cmd)
        initiator_dicts = parse_table_type_result(initiator_lines)
        if not initiator_dicts:
            host_initiators.append(host_info)
            continue
        for initiator_dict in initiator_dicts:
            host_initiator = host_info.copy_host()
            host_initiator.set_initiator_wwn(initiator_dict.get("Port Information", NA))
            host_initiator.set_port(initiator_dict.get("Port Name", NA))
            multipath_type = initiator_dict.get("Multipath Type", NA)
            host_initiator.set_access_model(TV1_ACCESS_MODE_DICT.get(multipath_type) or multipath_type)
            host_initiators.append(host_initiator)
    return host_initiators


def tv1_extend_private_host_map(context, host_initiators):
    """
    TV1: 扩展私有主机LUN映射信息
    :param context: 上下文
    :param host_initiators: 已解析的主机启动器信息, [HostInfo, ...]
    :return: [HostInfo, ...]
    """
    lun_map_lines = get_command_result_lines(context, "cmd_display_lun_map")
    lun_map_dicts = parse_table_type_result(lun_map_lines)
    mapped_host_ids = []
    for lun_map_dict in lun_map_dicts:
        map_type = lun_map_dict.get("Mapped to", "")
        map_to_id = map_type.split()[-1].strip()
        if map_type.startswith("Host: ") and map_to_id not in mapped_host_ids:
            mapped_host_ids.append(map_to_id)
    private_host_maps = []
    for host_initiator in host_initiators:
        if host_initiator.get_host_id() in mapped_host_ids:
            private_host_map = host_initiator.copy_host()
            # 主机私有LUN映射ID以10000起步，以与主机组ID做区分
            private_host_map.set_mapping_view_id("1%04d" % int(host_initiator.get_host_id()))
            private_host_map.set_mapping_view_name("MV_H_" + host_initiator.get_host_id())
            private_host_map.set_host_group_id(NA)
            private_host_map.set_host_group_name("HG_H_" + host_initiator.get_host_id())
            private_host_maps.append(private_host_map)
    return private_host_maps


def get_added_index(index):
    """
    增加并按返回增加后的计数
    :param index: 上次计数
    :return: +1后计数的字符串
    """
    index += index
    return str(index)


def tv1_analysis_host(context):
    """
    TV1: 解析主机信息
    :param context: 上下文
    :return: [HostInfo, ...]
    """
    host_gp_infos = tv1_analysis_host_group(context)
    host_infos = []
    for host_gp_info in host_gp_infos:
        group_host_cmd = "cmd_display_host_by_group_%s" % host_gp_info.get_host_group_id()
        host_info_lines = get_command_result_lines(context, group_host_cmd)
        host_info_dicts = parse_table_type_result(host_info_lines)
        if not host_info_dicts:
            host_infos.append(host_gp_info)
            continue
        for host_info_dict in host_info_dicts:
            host_info = host_gp_info.copy_host()
            host_info.set_host_id(host_info_dict.get("Host ID", NA))
            host_info.set_host_name(host_info_dict.get("Host Name", NA))
            host_info.set_host_type(host_info_dict.get("Os Type", NA))
            host_infos.append(host_info)
    return host_infos


def tv1_analysis_host_group(context):
    """
    TV1: 解析主机组信息
    :param context: 上下文
    :return: [HostInfo, ...]
    """
    host_group_lines = get_command_result_lines(context, "cmd_display_host_groups")
    host_group_dicts = parse_table_type_result(host_group_lines)
    host_gp_infos = []
    for host_group_dict in host_group_dicts:
        host_group_id = host_group_dict.get("Host Group ID", NA)
        host_group_name = host_group_dict.get("Name", NA)
        host_gp_info = HostInfo()
        host_gp_info.set_mapping_view_id(host_group_id)
        host_gp_info.set_mapping_view_name("MV_HG_" + host_group_id)
        host_gp_info.set_host_group_id(host_group_id)
        host_gp_info.set_host_group_name(host_group_name)
        host_gp_infos.append(host_gp_info)
    return host_gp_infos


def common_analysis_host_info(context):
    """
    TV2及之后等非V6设备，映射视图信息解析
    :param context: 上下文数据
    :return: [{}, ...]
    """
    return convert_hosts_to_dicts(common_analysis_host(context))


def convert_hosts_to_dicts(host_infos):
    """
    主机信息转换为字典结构
    :param host_infos: 主机信息
    :return: [{}, ...]
    """
    host_info_dicts = []
    for host_info in host_infos:
        host_info_dicts.append(host_info.get_host_info_dict(True))
    return host_info_dicts


def common_analysis_host(context):
    """
    TV2-非V6：主机解析
    :param context: 上下文
    :return: [HostInfo, ...]
    """
    host_group_infos = common_analysis_host_group(context)
    host_initiator_dict = get_host_initiator(context)

    host_infos = []
    for host_group_info in host_group_infos:
        group_host_cmd = "cmd_display_host_by_group_%s" % host_group_info.get_host_group_id()
        host_lines = get_command_result_lines(context, group_host_cmd)
        host_dicts = parse_table_type_result(host_lines)
        if not host_dicts:
            host_infos.append(host_group_info)
            continue
        for host_dict in host_dicts:
            initiators_info = common_parse_host_info(context, host_dict, host_group_info, host_initiator_dict)

            host_infos.extend(initiators_info)
    return host_infos


def common_parse_host_info(context, host_dict, host_group_info, host_initiator_dict):
    """
    TV2-非V6：解析主机回显内容
    :param context: 上下文
    :param host_dict: 一个主机的命令回显解析字典
    :param host_group_info: 一个主机组信息对象
    :param host_initiator_dict: 主机启动器解析字典
    :return: [HostInfo, ...]
    """
    host_info = host_group_info.copy_host()
    host_info.set_host_id(host_dict.get("ID", NA))
    host_info.set_host_name(host_dict.get("Name", NA))
    host_info.set_host_type(host_dict.get("Operating System", NA))
    initiators_info = common_analysis_initiator_for_host(host_initiator_dict, host_info)
    # 通过host_id拿到回显数据，构造字典{(host_id, wwn): access_mode, ...}
    initiator_by_host_cmd = "cmd_display_initiator_by_host_%s" % host_info.get_host_id()
    initiator_lines = get_command_result_lines(context, initiator_by_host_cmd)
    initiator_dicts = parse_table_type_result(initiator_lines)
    host_wwn_to_access_mode_dict = dict()
    for initiator_dict in initiator_dicts:
        host_id = initiator_dict.get('Host ID')
        wwn = initiator_dict.get('WWN') or initiator_dict.get('iSCSI IQN')
        failover_mode = initiator_dict.get('Failover Mode', NA)
        access_mode = COMMON_ACCESS_MODE_DICT.get(failover_mode) or failover_mode
        host_wwn_to_access_mode_dict[(host_id, wwn)] = access_mode
    # 将access_mode值赋值到initiator_info对象上
    for initiator_info in initiators_info:
        access_mode = host_wwn_to_access_mode_dict.get(
            (initiator_info.get_host_id(), initiator_info.get_initiator_wwn())) or NA
        initiator_info.set_access_model(access_mode)
    return initiators_info


def common_analysis_initiator_for_host(host_initiator_dict, host_info):
    """
    TV2-非V6：解析主机的启动器信息
    :param host_initiator_dict: 所有主机启动器信息
    :param host_info: 主机信息
    :return: {hostId: [initiator, ...], ...}
    """
    if host_info.get_host_id() not in host_initiator_dict:
        return [host_info]
    host_initiators = []
    for initiator_wwn in host_initiator_dict.get(host_info.get_host_id(), []):
        host_initiator = host_info.copy_host()
        host_initiator.set_initiator_wwn(initiator_wwn)
        host_initiators.append(host_initiator)
    return host_initiators


def common_analysis_host_initiator(context, cmd_display, wwn_symbol, host_initiator_dict):
    """
    TV2-非V6：主机启动器机械
    :param context: 上下文
    :param cmd_display: 命令回文标签
    :param wwn_symbol: 主机启动器表头
    :param host_initiator_dict: 主机启动器数据
    """
    initiator_lines = get_command_result_lines(context, cmd_display)
    initiator_dicts = parse_table_type_result(initiator_lines)
    for initiator_dict in initiator_dicts:
        host_id = initiator_dict.get("Host ID", "--")
        if host_id == "--":
            continue
        initiator_wwn = initiator_dict.get(wwn_symbol, NA)
        if host_id in host_initiator_dict:
            host_initiator_dict[host_id].append(initiator_wwn)
        else:
            host_initiator_dict[host_id] = [initiator_wwn]


def common_analysis_host_group(context):
    """
    TV2-非V6：主机组解析
    :param context: 上下文
    :return: [HostInfo, ...]
    """
    mapping_view_lines = get_command_result_lines(context, "cmd_display_mapping_view")
    mapping_view_dicts = parse_table_type_result(mapping_view_lines)
    host_group_infos = []
    for mapping_view_dict in mapping_view_dicts:
        mapping_view_id = mapping_view_dict.get("Mapping View ID", NA)
        mapping_view_name = mapping_view_dict.get("Mapping View Name", NA)
        host_group_lines = get_command_result_lines(context, "cmd_display_host_group_by_mapping_%s" % mapping_view_id)
        host_group_dicts = parse_table_type_result(host_group_lines)
        if not host_group_dicts:
            mapping_view_info = HostInfo()
            mapping_view_info.set_mapping_view_id(mapping_view_id)
            mapping_view_info.set_mapping_view_name(mapping_view_name)
            host_group_infos.append(mapping_view_info)
            continue
        for host_group_dict in host_group_dicts:
            host_group_info = HostInfo()
            host_group_info.set_mapping_view_id(mapping_view_id)
            host_group_info.set_mapping_view_name(mapping_view_name)
            host_group_info.set_host_group_id(host_group_dict.get("Host Group ID", NA))
            host_group_info.set_host_group_name(host_group_dict.get("Host Group Name", NA))
            host_group_infos.append(host_group_info)
    return host_group_infos
