#  coding=UTF-8
#  Copyright (c) Huawei Technologies Co., Ltd. 2019-2023. All rights reserved.

import traceback

from storages.storage_common import format_analysis_result, get_mapping_view_name
from storages.storage_common import get_command_result_lines
from storages.storage_common import LunInfo
from storages.HUAWEI.huawei_storage_util import (parse_table_type_result, OCEAN_STOR_V6, get_ocenstor_v6_mapping_info,
                                                 parse_key_value_type_result)
from storages.HUAWEI.huawei_storage_util import get_device_class

NA = 'NA'


def execute(context):
    """
    入口执行方法
    :param context: 上下文数据
    :return: context
    """
    try:
        dev_class = get_device_class(context)
        if dev_class == "TV1":
            lun_info_dicts = tv1_analysis_lun_info(context)
        elif dev_class == "COMMON":
            lun_info_dicts = common_analysis_lun_info(context)
        elif dev_class == OCEAN_STOR_V6:
            lun_info_dicts = oceanstor_v6_analysis_lun_info(context)
        else:
            lun_info_dicts = get_default_empty_lun_info()
        return format_analysis_result(context, lun_info_dicts)
    except Exception:
        context["error"] = "huawei analysis lun info error: %s" % str(traceback.format_exc())
        return format_analysis_result(context, get_default_empty_lun_info())


def oceanstor_v6_analysis_lun_info(context):
    """
    Dorado V6+新融合：LUN映射信息解析
    :param context: 上下文
    :return:lun信息
    """
    mapping_infos = get_ocenstor_v6_mapping_info(context)
    takeover_wwn_dict = get_takeover_wwn_dict(context)
    all_lun_data = get_all_lun(context)
    luns = []
    for mapping_info in mapping_infos:
        luns.extend(get_one_mapping_luns(context, mapping_info, takeover_wwn_dict, all_lun_data))
    return luns


def get_one_mapping_luns(context, mapping_info, takeover_wwn_dict, all_lun_data):
    """
    Dorado V6+新融合：一个映射下的lun解析
    :param context: 上下文
    :param mapping_info: 映射
    :param takeover_wwn_dict: takeover_wwn字典
    :param all_lun_data: 所有lun数据
    :return: [HostInfo, ...]
    """
    if NA != mapping_info.lun_group_id:
        return get_lun_group_mapping_luns(context, mapping_info, takeover_wwn_dict)
    if len(mapping_info.lun_id_list) > 0:
        return get_lun_list_mapping_luns(context, mapping_info, takeover_wwn_dict, all_lun_data)
    return []


def get_lun_group_mapping_luns(context, mapping_info, takeover_wwn_dict):
    """
    Dorado V6+新融合：lun组映射下的lun解析
    :param context: 上下文
    :param mapping_info: 映射
    :param takeover_wwn_dict: takeover_wwn字典
    :return: [HostInfo, ...]
    """
    lines = get_command_result_lines(context, "show lun_group lun lun_group_id=%s" % mapping_info.lun_group_id)
    lun_dicts = parse_table_type_result(lines)
    lun_infos = []
    lun_info = LunInfo()
    mapping_view_id = mapping_info.mapping_view_id
    lun_info.set_mapping_view_id(mapping_view_id)
    lun_info.set_mapping_view_name(get_mapping_view_name(mapping_view_id))
    lun_info.set_lun_group_id(mapping_info.lun_group_id)
    lun_info.set_lun_group_name(mapping_info.lun_group_name)
    for lun_dict in lun_dicts:
        lun_info_tmp = lun_info.copy_lun()
        lun_id = lun_dict.get("ID", NA)
        lun_info_tmp.set_lun_id(lun_id)
        lun_info_tmp.set_lun_name(lun_dict.get("Name", NA))
        lun_info_tmp.set_capacity(lun_dict.get("Capacity", NA))
        health_status = lun_dict.get("Health Status", NA)
        running_status = lun_dict.get("Running Status", NA)
        lun_info_tmp.set_status(health_status + " | " + running_status)
        lun_info_tmp.set_lun_type(lun_dict.get("Type", NA))
        lun_wwn = lun_dict.get("WWN", NA)
        lun_info_tmp.set_lun_wwn(lun_wwn)
        lun_info_tmp.set_takeover_lun_wwn(takeover_wwn_dict.get(lun_wwn, NA))
        set_lun_replication_and_metro(context, lun_id, lun_info_tmp)
        lun_infos.append(lun_info_tmp.get_lun_info_dict(True))
    return lun_infos


def get_lun_list_mapping_luns(context, mapping_info, takeover_wwn_dict, all_lun_data):
    """
    Dorado V6+新融合：（主机-luns）映射下的lun解析
    :param context: 上下文
    :param mapping_info: 映射
    :param takeover_wwn_dict: takeover_wwn字典
    :param all_lun_data: 所有lun数据
    :return: [HostInfo, ...]
    """
    lun_infos = []
    lun_info = LunInfo()
    mapping_view_id = mapping_info.mapping_view_id
    lun_info.set_mapping_view_id(mapping_view_id)
    lun_info.set_mapping_view_name(get_mapping_view_name(mapping_view_id))
    for lun_id in mapping_info.lun_id_list:
        lun_data = all_lun_data.get(lun_id, LunInfo())
        lun_info_tmp = lun_info.copy_lun()
        lun_info_tmp.set_lun_id(lun_id)
        lun_info_tmp.set_lun_name(lun_data.get_lun_name())
        lun_info_tmp.set_capacity(lun_data.get_capacity())
        lun_info_tmp.set_status(lun_data.get_status())
        lun_info_tmp.set_lun_type(lun_data.get_lun_type())
        lun_wwn = lun_data.get_lun_wwn()
        lun_info_tmp.set_lun_wwn(lun_wwn)
        lun_info_tmp.set_takeover_lun_wwn(takeover_wwn_dict.get(lun_wwn, NA))
        set_lun_replication_and_metro(context, lun_id, lun_info_tmp)
        lun_infos.append(lun_info_tmp.get_lun_info_dict(True))
    return lun_infos


def set_lun_replication_and_metro(context, lun_id, lun_info):
    """
    设置lun对象的remote_replication和hyper_metro值(V3/V5/V6系列)
    :param context: 上下文
    :param lun_id: lun id
    :param lun_info: lun对象
    :return
    """
    if lun_id == NA:
        return
    lun_info_lines = get_command_result_lines(context, "cmd_display_lun_general_lun_%s" % lun_id)
    lun_info_dict = parse_key_value_type_result(lun_info_lines, " : ")
    remote_replication_id = lun_info_dict.get("Remote Replication ID(s)", "")
    hyper_metro_id = lun_info_dict.get("HyperMetro ID(s)", "")

    remote_replication = parse_lun_value_by_id(remote_replication_id)
    hyper_metro = parse_lun_value_by_id(hyper_metro_id)

    lun_info.set_remote_replication(remote_replication)
    lun_info.set_hyper_metro(hyper_metro)


def get_all_lun(context):
    """
    Dorado V6+新融合：所有lun信息
    :param context: 上下文
    :return:lun信息
    """
    lines = get_command_result_lines(context, "show lun general")
    lun_dicts = parse_table_type_result(lines)
    lun_info_dict = {}
    for lun_dict in lun_dicts:
        lun_info = LunInfo()
        lun_did = lun_dict.get("ID", NA)
        lun_info.set_lun_id(lun_did)
        lun_info.set_lun_name(lun_dict.get("Name", NA))
        lun_info.set_capacity(lun_dict.get("Capacity", NA))
        health_status = lun_dict.get("Health Status", NA)
        running_status = lun_dict.get("Running Status", NA)
        lun_info.set_status(health_status + " | " + running_status)
        lun_info.set_lun_type(lun_dict.get("Type", NA))
        lun_info.set_lun_wwn(lun_dict.get("WWN", NA))
        lun_info_dict[lun_did] = lun_info
    return lun_info_dict


def get_default_empty_lun_info():
    """
    获取默认的空白LUN信息
    :return: [{key: NA, ...}]
    """
    empty_lun_info = LunInfo()
    return [empty_lun_info.get_lun_info_dict(True)]


def parse_lun_value_by_id(id_value):
    """
    解析LUN增值特性值
    :param id_value: 回显id值
    :return: 解析结果
    """
    if not id_value:
        return NA
    if id_value == '--':
        return "No"
    return "Yes"


def tv1_analysis_lun_info(context):
    """
    TV1: LUN映射信息解析
    :param context: 上下文
    :return: [LunInfo, ...]
    """
    lun_detail_dict = tv1_analysis_lun_detail(context)
    lun_map_lines = get_command_result_lines(context, "cmd_display_lun_map")
    lun_map_dicts = parse_table_type_result(lun_map_lines)
    lun_info_dicts = []
    for lun_map_dict in lun_map_dicts:
        mapped_to = lun_map_dict.get("Mapped to", NA)
        lun_id = lun_map_dict.get("Dev LUN ID", NA)
        if mapped_to == NA or lun_id not in lun_detail_dict:
            continue
        lun_info = lun_detail_dict.get(lun_id, LunInfo()).copy_lun()
        lun_info.set_lun_wwn(lun_map_dict.get("LUN WWN", NA))

        lun_info_lines = get_command_result_lines(context, "cmd_display_show_lun_detail_%s" % lun_id)
        lun_info_dict = parse_key_value_type_result(lun_info_lines, " | ")
        remote_replication_id = lun_info_dict.get("Remote Replication ID", "")
        remote_replication = parse_lun_value_by_id(remote_replication_id)
        lun_info.set_remote_replication(remote_replication)

        mapped_to_id = mapped_to.split()[-1].strip()
        if mapped_to.startswith("Group: "):
            lun_info.set_mapping_view_id(mapped_to_id)
            lun_info.set_mapping_view_name("MV_HG_" + mapped_to_id)
            lun_info.set_lun_group_name("LG_HG_" + mapped_to_id)
        if mapped_to.startswith("Host: "):
            lun_info.set_mapping_view_id("1%04d" % int(mapped_to_id))
            lun_info.set_mapping_view_name("MV_H_" + mapped_to_id)
            lun_info.set_lun_group_name("LG_H_" + mapped_to_id)
        lun_info_dicts.append(lun_info.get_lun_info_dict(True))
    return lun_info_dicts


def tv1_analysis_lun_detail(context):
    """
    TV1: 解析LUN详细信息
    :param context: 上下文
    :return: {lun_id: LunInfo, ...}
    """
    lun_detail_lines = get_command_result_lines(context, "cmd_display_lun_detail")
    lun_detail_dicts = parse_table_type_result(lun_detail_lines)
    lun_info_dict = {}
    for lun_detail_dict in lun_detail_dicts:
        lun_info = LunInfo()
        lun_info.set_lun_id(lun_detail_dict.get("ID", NA))
        lun_info.set_lun_name(lun_detail_dict.get("LUN Name", NA))
        lun_info.set_capacity(lun_detail_dict.get("Visible Capacity(MB)", NA) + "MB")
        lun_info.set_status(lun_detail_dict.get("Status", NA))
        lun_info.set_lun_type(lun_detail_dict.get("Lun Type", NA))
        lun_info_dict[lun_info.get_lun_id()] = lun_info
    return lun_info_dict


def common_analysis_lun_info(context):
    """
    TV2-非V6：LUN映射信息解析
    :param context:
    :return:
    """
    lun_info_dicts = []
    lun_group_infos = common_analysis_lun_group(context)
    takeover_wwn_dict = get_takeover_wwn_dict(context)
    for lun_group_info in lun_group_infos:
        lun_group_id = lun_group_info.get_lun_group_id()
        group_lun_lines = get_command_result_lines(context, "cmd_display_lun_by_group_%s" % lun_group_id)
        group_lun_dicts = parse_table_type_result(group_lun_lines)
        if not group_lun_dicts:
            lun_info_dicts.append(lun_group_info.get_lun_info_dict(True))
            continue
        for group_lun_dict in group_lun_dicts:
            lun_info = lun_group_info.copy_lun()
            lun_id = group_lun_dict.get("ID", NA)
            lun_info.set_lun_id(lun_id)
            lun_info.set_lun_name(group_lun_dict.get("Name", NA))
            lun_info.set_capacity(group_lun_dict.get("Capacity", NA))
            health_status = group_lun_dict.get("Health Status", NA)
            running_status = group_lun_dict.get("Running Status", NA)
            lun_info.set_status(health_status + " | " + running_status)
            lun_wwn = group_lun_dict.get("WWN", NA)
            lun_info.set_lun_wwn(lun_wwn)
            lun_info.set_takeover_lun_wwn(takeover_wwn_dict.get(lun_wwn, NA))
            lun_info.set_lun_type(group_lun_dict.get("Type", NA))
            set_lun_replication_and_metro(context, lun_id, lun_info)
            lun_info_dicts.append(lun_info.get_lun_info_dict(True))
    return lun_info_dicts


def get_takeover_wwn_dict(context):
    """
    获取lun wwn与takeover lun wwn的映射字典
    :param context:上下文
    :return: 映射字典
    """
    result = {}
    content = get_command_result_lines(context, "show lun_takeover general")
    data_dicts = parse_table_type_result(content)
    for one_data in data_dicts:
        result[one_data.get("WWN", NA)] = one_data.get("Takeover LUN WWN", NA)
    return result


def common_analysis_lun_group(context):
    """
    TV2-非V6：LUN组信息解析
    :param context:
    :return:
    """
    mapping_view_lines = get_command_result_lines(context, "cmd_display_mapping_view")
    mapping_view_dicts = parse_table_type_result(mapping_view_lines)
    lun_group_infos = []
    for mapping_view_dict in mapping_view_dicts:
        mapping_view_id = mapping_view_dict.get("Mapping View ID", NA)
        mapping_view_name = mapping_view_dict.get("Mapping View Name", NA)
        lun_group_lines = get_command_result_lines(context, "cmd_display_lun_group_by_mapping_%s" % mapping_view_id)
        lun_group_dicts = parse_table_type_result(lun_group_lines)
        if not lun_group_dicts:
            mapping_view_info = LunInfo()
            mapping_view_info.set_mapping_view_id(mapping_view_id)
            mapping_view_info.set_mapping_view_name(mapping_view_name)
            lun_group_infos.append(mapping_view_info)
            continue
        for lun_group_dict in lun_group_dicts:
            lun_group_id = lun_group_dict.get("LUN Group ID", NA)
            lun_group_name = lun_group_dict.get("LUN Group Name", NA)
            lun_group_info = LunInfo()
            lun_group_info.set_mapping_view_id(mapping_view_id)
            lun_group_info.set_mapping_view_name(mapping_view_name)
            lun_group_info.set_lun_group_id(lun_group_id)
            lun_group_info.set_lun_group_name(lun_group_name)
            lun_group_infos.append(lun_group_info)
    return lun_group_infos
