#  coding=UTF-8
#  Copyright (c) Huawei Technologies Co., Ltd. 2022-2022. All rights reserved.
from storages.PureStorage.pure_storage_san_util import get_mapping_data
from storages.storage_common import get_storage_format_info, get_mapping_view_name, NA
from storages.storage_common import get_command_result_lines
from storages.storage_common import LunInfo
from storages.storage_san_constant import LUN_LIST, MV_ID


def execute(context):
    return get_storage_format_info(context, parse_lun_info, [LunInfo().get_lun_info_dict()], False)


def parse_lun_info(context):
    """
    LUN解析（映射视图中的）
    :param context: 上下文数据
    :return: [{}, ...]
    """
    all_lun_list = get_all_lun_list(context)
    host_mapping, group_mapping = get_mapping_data(context)
    lun_list = []
    lun_list.extend(parse_lun_for_host_mapping(host_mapping, all_lun_list))
    lun_list.extend(parse_lun_for_host_group_mapping(group_mapping, all_lun_list))
    return lun_list


def get_all_lun_list(context):
    """
    获取所有lun的详情：常规lun、虚拟lun（不含有容量）
    :param context: 上下文
    :return: lun_list
    """
    lun_list = []
    lun_list.extend(get_general_lun_list(context))
    lun_list.extend(get_virtual_lun_list(context))
    return lun_list


def get_general_lun_list(context):
    """
    lun详细信息解析（常规lun）
    Name                Size  Source  Created                  Serial
    lun1                12G   -       2021-09-15 00:09:43 PDT  4DBE916*******5C000A45CF
    lun2                12G   -       2021-09-15 00:09:43 PDT  4DBE916*******5C000A45CE
    :param context: 上下文
    :return: lun_list
    """
    lun_info_lines = get_command_result_lines(context, "purevol list")
    return get_lun_list(lun_info_lines)


def get_virtual_lun_list(context):
    """
    lun详细信息解析（虚拟lun），没有容量大小
    Name            Source  Created                  Serial
    lun1            -       2020-02-05 02:43:11 PST  4DBE****FC7F455****18147
    lun2            -       2019-07-31 01:24:47 PDT  4DBE****FC7F455****12688
    :param context: 上下文
    :return: lun_list
    """
    lun_info_lines = get_command_result_lines(context, "purevol list --protocol-endpoint")
    return get_lun_list(lun_info_lines, True)


def get_lun_list(content, is_virtual_lun=False):
    """
    lun详细信息解析
    :param content: 回文
    :param is_virtual_lun: 是否是虚拟lun
    :return: lun_list
    """
    content_flag, lun_list = False, []
    for line in content:
        if line.startswith("Name") and line.endswith("Serial"):
            content_flag = True
            continue
        if not content_flag:
            continue
        items = line.split()
        if is_not_right_lun_line(len(items), is_virtual_lun):
            continue
        lun_list.append(get_lun_detail(items, is_virtual_lun))
    return lun_list


def is_not_right_lun_line(length, is_virtual_lun):
    """
    检查是否是非lun的数据
    :param length: 行分割后的长度
    :param is_virtual_lun: 是否是虚拟lun
    :return: 是-无用数据，否-有用数据
    """
    if is_virtual_lun and length < 4:
        return True
    if not is_virtual_lun and length < 5:
        return True
    return False


def get_lun_detail(items, is_virtual_lun):
    """
    获取lun详情(虚拟lun，没有容量数据)
    :param items: 行数据
    :param is_virtual_lun: 是否是虚拟lun
    :return: lun_list
    """
    lun = LunInfo()
    lun.set_lun_name(items[0])
    if not is_virtual_lun:
        lun.set_capacity(items[1])
    lun.set_lun_wwn(items[-1])
    return lun


def parse_lun_detail(lun_name):
    """
    解析lun_name是否存在lun组的数据  lun_group/lun_name
    :param lun_name: 映射视图
    :return: lun名字，lun组名字
    """
    if "/" not in lun_name:
        return lun_name, NA
    items = lun_name.split("/")
    return items[-1], items[0]


def get_one_lun_data(lun, mv_id):
    """
    组装一个lun的数据
    :param lun: lun数据
    :param mv_id: 所属的映射ID
    :return: lun数据
    """
    lun_temp = lun.copy_lun()
    lun_temp.set_mapping_view_id(mv_id)
    lun_temp.set_mapping_view_name(get_mapping_view_name(mv_id))
    lun_name, lun_group_name = parse_lun_detail(lun_temp.get_lun_name())
    lun_temp.set_lun_name(lun_name)
    lun_temp.set_lun_group_name(lun_group_name)
    return lun_temp.get_lun_info_dict()


def get_only_name_lun_data(lun_name, mv_id):
    """
    对于映射视图中的lun，不存在lun详情时，只生成带名字的lun
    :param lun_name: lun名字
    :param mv_id: 映射ID
    :return: lun数据
    """
    lun = LunInfo()
    lun_name_tmp, lun_group_name = parse_lun_detail(lun_name)
    lun.set_lun_name(lun_name_tmp)
    lun.set_lun_group_name(lun_group_name)
    lun.set_mapping_view_id(mv_id)
    lun.set_mapping_view_name(get_mapping_view_name(mv_id))
    return lun.get_lun_info_dict()


def get_no_detail_luns(with_detail_lun_names, lun_list_in_mapping, mv_id):
    """
    解析映射视图中的lun，但是该lun没有详细数据
    :param with_detail_lun_names: 映射视图中，存在详情的lun名字
    :param lun_list_in_mapping: 映射视图中lun
    :param mv_id: 映射ID
    :return: lun数据[{}, ...]
    """
    without_detail_luns = []
    for lun_name in lun_list_in_mapping:
        if lun_name not in with_detail_lun_names:
            without_detail_luns.append(get_only_name_lun_data(lun_name, mv_id))
    return without_detail_luns


def parse_lun_for_host_mapping(host_mapping, all_lun_list):
    """
    解析主机维度的视图
    :param host_mapping: 映射视图
    :param all_lun_list: 所有lun数据
    :return: lun数据[{}, ...]
    """
    lun_list = []
    for host_name, host_data in host_mapping.items():
        mv_id = host_data.get(MV_ID, 0)
        lun_list_in_mapping = host_data.get(LUN_LIST, [])
        with_detail_lun_names = []
        for lun in all_lun_list:
            if lun.get_lun_name() in lun_list_in_mapping:
                lun_list.append(get_one_lun_data(lun, mv_id))
                with_detail_lun_names.append(lun.get_lun_name())
        lun_list.extend(get_no_detail_luns(with_detail_lun_names, lun_list_in_mapping, mv_id))
    return lun_list


def parse_lun_for_host_group_mapping(group_mapping, all_lun_list):
    """
    解析主机组维度的视图
    :param group_mapping: 映射视图
    :param all_lun_list: 所有lun数据
    :return: lun数据[{}, ...]
    """
    lun_list = []
    for host_group_name, host_group_data in group_mapping.items():
        mv_id = host_group_data.get(MV_ID, 0)
        lun_list_in_group = host_group_data.get(LUN_LIST, [])
        for lun in all_lun_list:
            if lun.get_lun_name() in lun_list_in_group:
                lun_list.append(get_one_lun_data(lun, mv_id))
    return lun_list
