#  coding=UTF-8
#  Copyright (c) Huawei Technologies Co., Ltd. 2019-2023. All rights reserved.
# 存储脚本使用的资源信息

import json
import os
import traceback
import copy
from common.contentParse import get_return_from_txt

NA = 'NA'
# OceanStor  V3/V5采集解析常量
SYSTEM_VSTORE_ID = "--"
SYSTEM_VSTORE_NAME = "System Vstore"
resource = {
    "invalid_file_path": {
        "zh": u"读取文件:\t执行失败:\t文件路径不正确\r\n",
        "en": "read file:\texecute failed:\tinvalid file path\r\n",
    },
    "execute_success": {
        "zh": u":\t执行成功\r\n",
        "en": ":\texecute success\r\n",
    },
    "execute_fail": {
        "zh": u":\t执行失败\r\n",
        "en": ":\texecute failed\r\n",
    },
    "read_txt_file_fail": {
        "zh": u"读取TXT文件:\t执行失败\r\n",
        "en": "read txt file:\texecute failed\r\n",
    },
    "read_csv_file_fail": {
        "zh": u"读取csv文件:\t执行失败\r\n",
        "en": "read csv file:\texecute failed\r\n",
    },
}
host_info_keys = ["mapping_view_id", "mapping_view_name", "host_group_id", "host_group_name", "host_id", "host_name",
                  "host_type", "initiator_wwn", "port"]
lun_info_keys = ["mapping_view_id", "mapping_view_name", "lun_group_id", "lun_group_name", "lun_id", "lun_name",
                 "capacity", "host_id", "hostgroup_id", "status", "lun_wwn", "lun_type"]

empty_resource = {}
empty_value = "--"
STORAGE_DEV_MODEL = "dev_model"
STORAGE_EMC_ISILON = "Isilon"
STORAGE_HUAWEI_V3_V5 = "OceanStor_V3/V5"
STORAGE_HUAWEI_9000 = "OceanStor_9000"

SYSTEM_INFO_KEYS = ["report_date/time", "infograb_version", "storage_vendor", "storage_series", "storage_model",
                    "storage_firmware", "storage_serial", "ip_address"]


def get_mapping_view_name(mv_id):
    """
    组装映射视图名
    :param mv_id: mapping_view ID
    :return: mapping_view Name
    """
    return "MappingView%03d" % mv_id


def get_storage_format_info(context, get_data_func, default_data, is_system_data=False):
    """
    获取格式化后的数据
    :param context: 上下文
    :param get_data_func: 数据获取的函数
    :param default_data: 异常时的默认数据
    :param is_system_data: 是否为系统数据
    :return: 格式化后的数据
    """
    try:
        data = get_data_func(context)
        if is_system_data:
            data = [parse_data_dict(SYSTEM_INFO_KEYS, data)]
        return format_analysis_result(context, data)
    except Exception:
        context["error"] = "analysis info error: %s" % str(traceback.format_exc())
        return format_analysis_result(context, default_data)


def get_storage_name_info(context):
    """
    默认返回空数据，以适应存储信息解析框架
    :param context: 上下文
    :return: context
    """
    data_dicts = [{'mapping_view_id': NA, 'mapping_view_name': NA}]
    data_key = context.get("data_key", "mapping_view_info")
    context["result"] = json.dumps({data_key: data_dicts})
    return context


def get_default_empty_system_info():
    """
    获取默认的空白系统信息
    :return: [{key: NA, ...}]
    """
    empty_sys_infos = {}
    for sys_key in SYSTEM_INFO_KEYS:
        empty_sys_infos[sys_key] = NA
    return [empty_sys_infos]


def get_resource(language, key):
    """
    根据语言场景获取资源信息
    :param language: 语言：zh、en
    :param key: 资源key
    :return: 资源value
    """
    return resource.get(key, empty_resource).get(language[0:2], empty_value)


def get_command_result_lines(context, command, with_blank=False):
    """
    从采集信息txt文件中读取回显信息
    :param context: 上下文数据
    :param command: 读取信息的命令
    :param with_blank:  是否保留前后空格
    :return: 命令回显行信息，已strip
    """
    context["command"] = command
    return get_return_from_txt(context, with_blank)


def format_analysis_result(context, data_dicts):
    """
    格式化解析的结果数据并返回
    :param context: 上下文数据
    :param data_dicts: 数据字典集合: [{}, ...]
    :return: context
    """
    data_key = context.get("data_key")
    result_data = {data_key: data_dicts}
    context["result"] = json.dumps(result_data)
    return context


def parse_data_dict(info_keys, info_values):
    """
    将集合数据与key匹配为字典回显
    :param info_keys: 字典key
    :param info_values: 数据
    :return: dict
    """
    data_dict = {}
    if len(info_keys) == len(info_values):
        for i in range(len(info_keys)):
            data_dict[info_keys[i]] = info_values[i]
    return data_dict


def remove_bad_end_content_for_hp(txt_list):
    """
    处理hp储存 结尾多余的回文 （最后两行是多余的，版本原因不一定存在）
    364 qj003           003
    ----------------------------------------------------
    52 total                   1679
    :param txt_list: 回文
    :return: 正确回文
    """
    if txt_list and '---------------------' in txt_list[-1]:
        return txt_list[:-1]
    if len(txt_list) > 1 and 'total' in txt_list[-1] and '---------------------' in txt_list[-2]:
        return txt_list[:-2]
    return txt_list


def find_file_parent_dir(report_path, target_file):
    """
    在报告目录中找到指定文件所在的目录
    :param report_path: 报告目录
    :param target_file: 指定文件
    :return: 指定文件所在目录
    """
    if os.path.basename(report_path) == target_file:
        return os.path.dirname(report_path)
    if not os.path.isdir(report_path):
        return ""
    files = os.listdir(report_path)
    if target_file in files:
        return report_path
    for file in files:
        file_path = os.path.join(report_path, file)
        if os.path.isdir(file_path):
            result_path = find_file_parent_dir(file_path, target_file)
            if result_path:
                return result_path
    return ""


def deduplicate(obj_list):
    """
    主机/Lun 最终数据的去重
    :param obj_list: 主机/Lun 列表
    :return: 去重后的数据
    """
    all_data_str = []
    result = []
    for obj_data in obj_list:
        obj_str = "".join(obj_data.to_str())
        if obj_str in all_data_str:
            continue
        result.append(obj_data)
        all_data_str.append(obj_str)
    return result


class HostInfo:

    __mapping_view_id = NA
    __mapping_view_name = NA
    __host_group_id = NA
    __host_group_name = NA
    __host_id = NA
    __host_name = NA
    __host_type = NA
    __initiator_wwn = NA
    __port = NA
    __access_model = NA

    def set_mapping_view_id(self, mapping_view_id):
        self.__mapping_view_id = mapping_view_id

    def set_mapping_view_name(self, mapping_view_name):
        self.__mapping_view_name = mapping_view_name

    def set_host_group_id(self, host_group_id):
        self.__host_group_id = host_group_id

    def set_host_group_name(self, host_group_name):
        self.__host_group_name = host_group_name

    def set_host_id(self, host_id):
        self.__host_id = host_id

    def set_host_name(self, host_name):
        self.__host_name = host_name

    def set_host_type(self, host_type):
        self.__host_type = host_type

    def set_initiator_wwn(self, initiator_wwn):
        self.__initiator_wwn = initiator_wwn

    def set_port(self, port):
        self.__port = port

    def set_access_model(self, access_model):
        self.__access_model = access_model

    def get_mapping_view_id(self):
        return self.__mapping_view_id

    def get_mapping_view_name(self):
        return self.__mapping_view_name

    def get_host_group_id(self):
        return self.__host_group_id

    def get_host_group_name(self):
        return self.__host_group_name

    def get_host_id(self):
        return self.__host_id

    def get_host_name(self):
        return self.__host_name

    def get_host_type(self):
        return self.__host_type

    def get_initiator_wwn(self):
        return self.__initiator_wwn

    def get_port(self):
        return self.__port

    def get_access_model(self):
        return self.__access_model

    def get_host_info_dict(self, is_huawei=False):
        """
        将数据转换为字典回显
        :param is_huawei: 是否华为存储，默认不是
        :return:
        """
        host_info = [self.__mapping_view_id, self.__mapping_view_name, self.__host_group_id,
                     self.__host_group_name, self.__host_id, self.__host_name, self.__host_type,
                     self.__initiator_wwn, self.__port]
        if not is_huawei:
            return parse_data_dict(host_info_keys, host_info)
        huawei_host_info_keys = copy.deepcopy(host_info_keys)
        huawei_host_info_keys.append("access_model")
        host_info.append(self.__access_model)
        return parse_data_dict(huawei_host_info_keys, host_info)

    def copy_host(self):
        """
        复制数据模型
        :return: HostInfo
        """
        host_info = HostInfo()
        host_info.set_mapping_view_id(self.__mapping_view_id)
        host_info.set_mapping_view_name(self.__mapping_view_name)
        host_info.set_host_group_id(self.__host_group_id)
        host_info.set_host_group_name(self.__host_group_name)
        host_info.set_host_id(self.__host_id)
        host_info.set_host_name(self.__host_name)
        host_info.set_host_type(self.__host_type)
        host_info.set_initiator_wwn(self.__initiator_wwn)
        host_info.set_port(self.__port)
        host_info.set_access_model(self.__access_model)
        return host_info

    def transform_host_value(self):
        """
        转换属性值为数组，默认转换方式，特殊场景需单独实现
        :return: 属性值数组
        """
        return [self.__mapping_view_id, self.__mapping_view_name, self.__host_group_id, self.__host_group_name,
                self.__host_id, self.__host_name, self.__host_type, self.__initiator_wwn, self.__port]


class LunInfo:

    __mapping_view_id = NA
    __mapping_view_name = NA
    __lun_group_id = NA
    __lun_group_name = NA
    __lun_id = NA
    __lun_name = NA
    __capacity = NA
    __host_id = NA
    __host_group_id = NA
    __status = NA
    __lun_wwn = NA
    __lun_type = NA
    __takeover_lun_wwn = NA
    __remote_replication = NA
    __hyper_metro = NA

    def set_mapping_view_id(self, mapping_view_id):
        self.__mapping_view_id = mapping_view_id

    def set_mapping_view_name(self, mapping_view_name):
        self.__mapping_view_name = mapping_view_name

    def set_lun_group_id(self, lun_group_id):
        self.__lun_group_id = lun_group_id

    def set_lun_group_name(self, lun_group_name):
        self.__lun_group_name = lun_group_name

    def set_lun_id(self, lun_id):
        self.__lun_id = lun_id

    def set_lun_name(self, lun_name):
        self.__lun_name = lun_name

    def set_capacity(self, capacity):
        self.__capacity = capacity

    def set_host_id(self, host_id):
        self.__host_id = host_id

    def set_host_group_id(self, host_group_id):
        self.__host_group_id = host_group_id

    def set_status(self, status):
        self.__status = status

    def set_lun_wwn(self, lun_wwn):
        self.__lun_wwn = lun_wwn

    def set_takeover_lun_wwn(self, takeover_lun_wwn):
        self.__takeover_lun_wwn = takeover_lun_wwn

    def set_remote_replication(self, remote_replication):
        self.__remote_replication = remote_replication

    def set_hyper_metro(self, hyper_metro):
        self.__hyper_metro = hyper_metro

    def set_lun_type(self, lun_type):
        self.__lun_type = lun_type

    def get_mapping_view_id(self):
        return self.__mapping_view_id

    def get_mapping_view_name(self):
        return self.__mapping_view_name

    def get_lun_group_id(self):
        return self.__lun_group_id

    def get_lun_group_name(self):
        return self.__lun_group_name

    def get_lun_id(self):
        return self.__lun_id

    def get_lun_name(self):
        return self.__lun_name

    def get_capacity(self):
        return self.__capacity

    def get_host_id(self):
        return self.__host_id

    def get_host_group_id(self):
        return self.__host_group_id

    def get_status(self):
        return self.__status

    def get_lun_wwn(self):
        return self.__lun_wwn

    def get_takeover_lun_wwn(self):
        return self.__takeover_lun_wwn

    def get_remote_replication(self):
        return self.__remote_replication

    def get_hyper_metro(self):
        return self.__hyper_metro

    def get_lun_type(self):
        return self.__lun_type

    def get_lun_info_dict(self, is_huawei=False):
        """
        将数据转换为字典回显
        :param is_huawei: 是否华为存储，默认不是
        :return:
        """
        lun_info = [self.__mapping_view_id, self.__mapping_view_name, self.__lun_group_id, self.__lun_group_name,
                    self.__lun_id, self.__lun_name, self.__capacity, self.__host_id, self.__host_group_id,
                    self.__status, self.__lun_wwn, self.__lun_type]
        if not is_huawei:
            return parse_data_dict(lun_info_keys, lun_info)
        huawei_lun_info_keys = copy.deepcopy(lun_info_keys)
        huawei_lun_info_keys.append("takeover_lun_wwn")
        huawei_lun_info_keys.append("remote_replication")
        huawei_lun_info_keys.append("hyper_metro")
        lun_info.append(self.__takeover_lun_wwn)
        lun_info.append(self.__remote_replication)
        lun_info.append(self.__hyper_metro)
        return parse_data_dict(huawei_lun_info_keys, lun_info)

    def copy_lun(self):
        """
        复制数据模型
        :return: LunInfo
        """
        lun_info = LunInfo()
        lun_info.set_mapping_view_id(self.__mapping_view_id)
        lun_info.set_mapping_view_name(self.__mapping_view_name)
        lun_info.set_lun_group_id(self.__lun_group_id)
        lun_info.set_lun_group_name(self.__lun_group_name)
        lun_info.set_lun_id(self.__lun_id)
        lun_info.set_lun_name(self.__lun_name)
        lun_info.set_capacity(self.__capacity)
        lun_info.set_host_id(self.__host_id)
        lun_info.set_host_group_id(self.__host_group_id)
        lun_info.set_status(self.__status)
        lun_info.set_lun_wwn(self.__lun_wwn)
        lun_info.set_lun_type(self.__lun_type)
        lun_info.set_takeover_lun_wwn(self.__takeover_lun_wwn)
        lun_info.set_remote_replication(self.__remote_replication)
        lun_info.set_hyper_metro(self.__hyper_metro)
        return lun_info

    def transform_lun_value(self):
        """
        转换属性值为数组，默认转换方式，特殊场景需单独实现
        :return: 属性值数组
        """
        return [self.__mapping_view_id, self.__mapping_view_name, self.__lun_group_id, self.__lun_group_name,
                self.__lun_id, self.__lun_name, self.__capacity, self.__host_id, self.__host_group_id, self.__status,
                self.__lun_wwn, self.__lun_type]
