/*
 * Decompiled with CFR 0.152.
 */
package com.huawei.ism.tool.certbase.utils;

import com.huawei.ism.tool.certbase.enums.CertificateExceptionEnum;
import com.huawei.ism.tool.certbase.enums.SafeEncodeAlgorithmEnum;
import com.huawei.ism.tool.certbase.exception.CertAlgorithmUnsafeException;
import com.huawei.ism.tool.certbase.exception.CertificateVerifyException;
import com.huawei.ism.tool.certbase.utils.CertPublicKeyValidator;
import com.huawei.ism.tool.certbase.utils.CertificateChainSorter;
import java.security.InvalidKeyException;
import java.security.Principal;
import java.security.PublicKey;
import java.security.SignatureException;
import java.security.cert.CertificateException;
import java.security.cert.CertificateExpiredException;
import java.security.cert.CertificateNotYetValidException;
import java.security.cert.X509Certificate;
import java.util.Arrays;
import java.util.Collection;
import java.util.Collections;
import java.util.LinkedList;
import java.util.Locale;
import java.util.Objects;
import java.util.Optional;
import java.util.regex.Matcher;
import java.util.regex.Pattern;

public class CertificateValidator {
    private static final String SIGN_ALG_SEPARATOR = "WITH";
    private static final Pattern SIGN_ALG_PATTERN = Pattern.compile("([A-Z]+)(\\d+)");

    public static void checkCertificateChain(Collection<X509Certificate> trustedCerts, X509Certificate ... certChain) throws CertificateException {
        if (CertificateValidator.isEmptyCertChain(certChain)) {
            throw new CertificateVerifyException(CertificateExceptionEnum.NO_CERTIFICATE);
        }
        CertificateValidator.checkCertificateChain(CertificateValidator.getCompletesCertChain(Objects.isNull(trustedCerts) ? Collections.emptySet() : trustedCerts, CertificateChainSorter.sortCertificateChain(certChain)));
    }

    private static boolean isEmptyCertChain(X509Certificate[] certChain) {
        return Objects.isNull(certChain) || certChain.length == 0;
    }

    private static X509Certificate[] getCompletesCertChain(Collection<X509Certificate> trustedCerts, X509Certificate[] certificateChain) {
        Optional<X509Certificate> upperLevelCert;
        LinkedList<X509Certificate> certificates = new LinkedList<X509Certificate>(Arrays.asList(certificateChain));
        while (!CertificateValidator.isRootCA((X509Certificate)certificates.get(0)) && (upperLevelCert = CertificateValidator.getUpperLevelCert(trustedCerts, ((X509Certificate)certificates.get(0)).getIssuerDN())).isPresent()) {
            certificates.add(0, upperLevelCert.get());
        }
        return certificates.toArray(new X509Certificate[0]);
    }

    private static Optional<X509Certificate> getUpperLevelCert(Collection<X509Certificate> trustedCerts, Principal issuer) {
        return trustedCerts.stream().filter(certificate -> certificate.getSubjectDN().equals(issuer)).findFirst();
    }

    public static void checkCertificateChain(X509Certificate ... certChain) throws CertificateException {
        if (CertificateValidator.isEmptyCertChain(certChain)) {
            throw new CertificateVerifyException(CertificateExceptionEnum.NO_CERTIFICATE);
        }
        X509Certificate[] certificateChain = CertificateChainSorter.sortCertificateChain(certChain);
        CertificateValidator.checkRootCA(certificateChain[0]);
        CertificateValidator.checkIntermediateCA(certificateChain);
        CertificateValidator.checkLastCertificate(certificateChain[certificateChain.length - 1]);
        CertificateValidator.verifySignature(certificateChain);
        CertificateValidator.checkCertChainAlgorithm(certificateChain);
    }

    private static void checkRootCA(X509Certificate rootCA) throws CertificateException {
        if (!CertificateValidator.isRootCA(rootCA)) {
            throw new CertificateVerifyException(CertificateExceptionEnum.NO_ROOT_CA);
        }
        CertificateValidator.checkCertValidityPeriod(rootCA);
        if (rootCA.getVersion() > 2) {
            CertificateValidator.checkKeyUsage(rootCA);
        }
    }

    private static void checkIntermediateCA(X509Certificate[] certificateChain) throws CertificateException {
        for (int i = 1; i < certificateChain.length - 1; ++i) {
            X509Certificate intermediateCA = certificateChain[i];
            if (intermediateCA.getVersion() < 3) {
                throw new CertificateVerifyException(CertificateExceptionEnum.NOT_X509_V3_CERT);
            }
            CertificateValidator.checkCertValidityPeriod(intermediateCA);
            CertificateValidator.checkExtensions(intermediateCA);
        }
    }

    private static void checkLastCertificate(X509Certificate lastCertificate) throws CertificateException {
        if (lastCertificate.getVersion() < 3) {
            throw new CertificateVerifyException(CertificateExceptionEnum.NOT_X509_V3_CERT);
        }
        CertificateValidator.checkCertValidityPeriod(lastCertificate);
    }

    private static void checkCertValidityPeriod(X509Certificate certificate) throws CertificateException {
        try {
            certificate.checkValidity();
        }
        catch (CertificateNotYetValidException e) {
            throw new CertificateVerifyException(CertificateExceptionEnum.INVALID_VALIDITY_PERIOD, (Throwable)e);
        }
        catch (CertificateExpiredException e) {
            throw new CertificateVerifyException(CertificateExceptionEnum.CERTIFICATE_EXPIRED, (Throwable)e);
        }
    }

    private static void checkCertChainAlgorithm(X509Certificate[] certificateChain) throws CertificateException {
        for (X509Certificate certificate : certificateChain) {
            CertificateValidator.checkAlgorithm(certificate);
        }
    }

    private static void verifySignature(X509Certificate[] certificateChain) throws CertificateException {
        PublicKey upperCaPublicKey = certificateChain[0].getPublicKey();
        for (X509Certificate certificate : certificateChain) {
            CertificateValidator.verifySignature(certificate, upperCaPublicKey);
            upperCaPublicKey = certificate.getPublicKey();
        }
    }

    private static void verifySignature(X509Certificate certificate, PublicKey publicKey) throws CertificateException {
        try {
            certificate.verify(publicKey);
        }
        catch (InvalidKeyException e) {
            throw new CertificateVerifyException(CertificateExceptionEnum.INVALID_PUBLIC_KEY, (Throwable)e);
        }
        catch (SignatureException e) {
            throw new CertificateVerifyException(CertificateExceptionEnum.INVALID_SIGNATURE, (Throwable)e);
        }
        catch (Exception e) {
            throw new CertificateVerifyException(CertificateExceptionEnum.VERIFY_SIGNATURE_ERROR, (Throwable)e);
        }
    }

    public static void checkAlgorithm(X509Certificate certificate) throws CertificateException {
        CertificateValidator.checkCertSignatureAlgorithm(certificate);
        CertPublicKeyValidator.checkPublicKeyLength(certificate.getPublicKey());
    }

    private static void checkCertSignatureAlgorithm(X509Certificate certificate) throws CertificateException {
        String signatureAlgorithm = certificate.getSigAlgName().toUpperCase(Locale.ENGLISH);
        String publicKeyAlgorithm = certificate.getPublicKey().getAlgorithm().toUpperCase(Locale.ENGLISH);
        if (!signatureAlgorithm.contains(SIGN_ALG_SEPARATOR + publicKeyAlgorithm)) {
            throw new CertAlgorithmUnsafeException("SignatureAlgorithm not contains publicKeyAlgorithm: " + signatureAlgorithm);
        }
        CertificateValidator.checkSignatureAlgorithmBits(signatureAlgorithm);
    }

    private static void checkSignatureAlgorithmBits(String signatureAlgorithm) throws CertificateException {
        String signAlgName = signatureAlgorithm.substring(0, signatureAlgorithm.indexOf(SIGN_ALG_SEPARATOR));
        Matcher signAlgMatcher = SIGN_ALG_PATTERN.matcher(signAlgName);
        if (!signAlgMatcher.matches()) {
            throw new CertAlgorithmUnsafeException("Invalid SignatureAlgorithm format: " + signatureAlgorithm);
        }
        if (!SafeEncodeAlgorithmEnum.isSafe(signAlgMatcher.group(1), Integer.parseInt(signAlgMatcher.group(2)))) {
            throw new CertAlgorithmUnsafeException(signatureAlgorithm);
        }
    }

    public static void checkExtensions(X509Certificate cert) throws CertificateException {
        CertificateValidator.checkBasicConstraints(cert);
        CertificateValidator.checkKeyUsage(cert);
    }

    private static void checkBasicConstraints(X509Certificate certificate) throws CertificateException {
        if (certificate.getBasicConstraints() < 0) {
            throw new CertificateVerifyException(CertificateExceptionEnum.END_USER_TRY_ACT_CA);
        }
    }

    private static void checkKeyUsage(X509Certificate certificate) throws CertificateException {
        boolean[] theKeyUsageInfo = certificate.getKeyUsage();
        if (!(theKeyUsageInfo == null || theKeyUsageInfo.length >= 6 && theKeyUsageInfo[5])) {
            throw new CertificateVerifyException(CertificateExceptionEnum.KEY_USAGE_WITHOUT_CERT_SIGN);
        }
    }

    public static boolean isRootCA(X509Certificate certificate) {
        String subjectDN = certificate.getSubjectX500Principal().getName();
        String issuerDN = certificate.getIssuerX500Principal().getName();
        return subjectDN.equals(issuerDN);
    }

    private CertificateValidator() {
    }
}

