#!/bin/bash
###########################################################################
#
# Description:  Patch Install Tool for Campus R21
# Version：        V1.0
# Author:       w00284198
#
###########################################################################

# set -x

# Get current path
curr_path="/opt/oss/spc121_upgrade_patch"
patch_Path="$curr_path/Patch"


timestamp=`date "+%Y%m%d%H%M%S"`
echo $timestamp > $curr_path/timestamp.txt


# create backup_dir
backup_dir="$curr_path/backup"
mkdir -p $backup_dir
chmod 770 $backup_dir


update_success="$curr_path/updateSuccess.txt"


# micro service root path
micro_service_path="/opt/oss/NCECAMPUS/apps"


new_version="V300R021C00SPC121"
old_version="V300R021C00SPC120"


new_time="20221025"
old_time="20220826"


#==========================================================================
#   Name:           log
#   Description:    log to console and file
#   Input: String message
#   Return:
#==========================================================================
function log()
{
    timestamp=`date "+%Y-%m-%d %H:%M:%S"`
    message="$timestamp $@"
    echo -e $message >> $curr_path/patch_upgrade.log
}




#==========================================================================
#   Name:           check_need_update
#   Description:    check need update
#   Input: 
#   Return:
#==========================================================================
function check_need_update()
{
    if [ -f "$update_success" ];then
        log "no need to update."
        echo "no need to update."
        exit 1
    fi
}



#==========================================================================
#   Name:           backup
#   Description:    backup jar
#   Input:          $1 jar name  $2 micro service name 
#   Return:    
#==========================================================================
function backup()
{
     if [ ! -f "$backup_dir/$1" ];then #如果之前没有备份过，则备份
        log "[INFO]Begin to backup $1."
        #未改权限可执行成功
        sudo -u ossuser mv ${micro_service_path}/$2/lib/$1 ${backup_dir}/
        log "[INFO]Backup $1 successfully."
        echo "[INFO]Backup $1 successfully."
     else
        log "[INFO]No need to backup $1, because has backuped before."
     fi
}



#==========================================================================
#   Name:           upgrade_patch
#   Description:    update jar
#   Input:          $1 jar name  $2 micro service name
#   Return:
#==========================================================================
function upgrade_patch()
{
    if [ -f "$backup_dir/$1" ];then #只有备份过，才替换
        log "[INFO]Begin to upgrade $1."
        sudo -u ossuser chmod 770 ${micro_service_path}/$2/lib/$1
        sudo -u ossuser cp ${patch_Path}/$1 ${micro_service_path}/$2/lib/
        if [ "$1" = "license-branch-service-21.0120.0560.0200-SNAPSHOT.jar" ]; then
            sudo -u ossuser chmod 550 ${micro_service_path}/$2/lib/$1
        else
            sudo -u ossuser chmod 400 ${micro_service_path}/$2/lib/$1
        fi
        
       
        log "[INFO]upgrade $1 successfully."
        echo "[INFO]upgrade $1 successfully."
    else
        log "[ERROR]No need to upgrade $1, because backup failed."
    fi
}




#==========================================================================
#   Name:           upgrade
#   Description:    upgrade jar
#   Input:          $1 jar name  $2 micro service name
#   Return:
#==========================================================================
function upgrade()
{   
    if [ -d "${micro_service_path}/$2" ];then #判断节点有服务再替换打补丁
        backup $1 $2
        upgrade_patch $1 $2
    else
        log "[INFO]No upgrade was required $1."
        echo "[INFO]No upgrade was required $1."
    fi
}


function upgrade_PONMgmtService()
{
    #DTS2022092716071    
    original_path="/opt/oss/NCECAMPUS/apps/PONMgmtService/webapps/ROOT/WEB-INF/classes/com/huawei/u2020/ponmgmtservice/business/campus"

    if [ -d "/opt/oss/NCECAMPUS/apps/PONMgmtService" ];then
        if [ ! -d "$backup_dir/reconciliation" ];then #如果之前没有备份过，则备份
        
            # 原文件为可读权限，需要先改为可读可写执行权限
            sudo -u ossuser chmod 770 $original_path/reconciliation
        
            sudo -u ossuser cp -r $original_path/reconciliation ${backup_dir}/
            
            #将备份文件改为可读可写可执行权限
            sudo -u ossuser chmod 770 -R $backup_dir/reconciliation
            
            log "[INFO]backup PONMgmtService successfully."
            echo "[INFO]backup PONMgmtService successfully."
        else
            log "[INFO]No need to backup PONMgmtService, because has backuped.."
        fi
        
        if [ -d "$backup_dir/reconciliation" ];then #如果备份成功，则替换
            
            #改权限
            sudo -u ossuser chmod 770 -R $original_path/reconciliation
            
            sudo -u ossuser cp -r ${patch_Path}/reconciliation $original_path
            
            #恢复权限
            sudo -u ossuser chmod 440 -R $original_path/reconciliation
            sudo -u ossuser chmod 550 $original_path/reconciliation
            
            log "[INFO]update PONMgmtService successfully."
            echo "[INFO]update PONMgmtService successfully."
        else
            log "[ERROR]update PONMgmtService failed, because backup PONMgmtService failed."
        fi
    else
       echo "[INFO]No need to update PONMgmtService, because has no PONMgmtService."
    fi
}

function upgrade_ap_IotConfig()
{
    #DTS2022092807292    
    
    original_path="/opt/oss/envs/Product-NginxService/21.0120.0300.0001/etc/nginx/html/webroot/deviceConfig-app/themes/default/eview_modules/ap_config"

    if [ -d "/opt/oss/envs/Product-NginxService" ];then
        if [ ! -d "$backup_dir/ap_IotConfig" ];then #如果之前没有备份过，则备份
            #未改权限可备份成功
            sudo -u ossuser mv $original_path/ap_IotConfig  ${backup_dir}/
            
            echo "[INFO]Backup ap_IotConfig successfully."
            log "[INFO]Backup ap_IotConfig successfully."
        else
            log "[INFO]No need to backup ap_IotConfig, because has backuped."
        fi
        
        if [ -d "$backup_dir/ap_IotConfig" ];then #如果备份成功，则替换
            #不改权限可替换成功
            sudo -u ossuser cp -r ${patch_Path}/ap_IotConfig $original_path
            
            log "[INFO]Update ap_IotConfig successfully."
            echo "[INFO]Update ap_IotConfig successfully."
        else 
            log "[ERROR]Update ap_IotConfig failed, because backup failed."
        fi
    else
       echo "[INFO]No need to update ap_IotConfig, because has no NginxService."
    fi
}


function upgrade_RadiusServerService_configuration()
{
    #DTS2022090913948    
    
    original_path="/opt/oss/NCECAMPUS/apps/RadiusServerService/controller/configuration"

    if [ -d "/opt/oss/NCECAMPUS/apps/RadiusServerService" ];then
	    # 修改dbconfig
		filename=access_authen_common_dbm.xml
        if [ ! -f "$backup_dir/$filename" ];then #如果之前没有备份过，则备份
            #未改权限可备份成功
            sudo -u ossuser mv $original_path/dbconfig/$filename  ${backup_dir}/
            
            echo "[INFO]Backup $filename successfully."
            log "[INFO]Backup $filename successfully."
        else
            log "[INFO]No need to backup $filename, because has backuped."
        fi
        
        if [ -f "$backup_dir/$filename" ];then #如果备份成功，则替换
            #不改权限可替换成功
            sudo -u ossuser cp -r ${patch_Path}/$filename $original_path/dbconfig/
            
            log "[INFO]Update $filename successfully."
            echo "[INFO]Update $filename successfully."
        else 
            log "[ERROR]Update $filename failed, because backup failed."
        fi
		
		
		#修改threadpools
		filename=ac_campus_radiusauth_radiusserver_service-cfg.xml
		if [ ! -f "$backup_dir/$filename" ];then #如果之前没有备份过，则备份
            #未改权限可备份成功
            sudo -u ossuser mv $original_path/threadpools/$filename ${backup_dir}/
            
            echo "[INFO]Backup $filename successfully."
            log "[INFO]Backup $filename successfully."
        else
            log "[INFO]No need to backup $filename, because has backuped."
        fi
        
        if [ -f "$backup_dir/$filename" ];then #如果备份成功，则替换
            #不改权限可替换成功
            sudo -u ossuser cp -r ${patch_Path}/$filename $original_path/threadpools/
            
            log "[INFO]Update $filename successfully."
            echo "[INFO]Update $filename successfully."
        else 
            log "[ERROR]Update $filename failed, because backup failed."
        fi
    else
       echo "[INFO]No need to update $filename, because has no RadiusServerService."
    fi
}



#==========================================================================
#   Name:           update_version
#   Description:    update version
#   Input: 
#   Return:
#==========================================================================
function update_version()
{
    updated=0
    if [ -d "$1" ];then
        for file in `find $1/manager/DeployAgent/svcs-pub/NCECAMPUS-Product-CampusBaseService-campusbaseservice/CampusSolutionService-*  -name "productioninfo.properties"`
        do
            file_name=`basename $file`
            if [[ "$file_name" == "productioninfo.properties" ]]; then
                sed -i "s/product.info.version=.*/product.info.version=$new_version/g" $file
                sed -i "s/product.info.date=.*/product.info.date=$new_time/g" $file
                if [ $? -eq 0 ]; then
                    updated=1
                else
                    log "Update $1/$file_name failed"
                    echo "Update $1/$file_name failed"
                fi
            fi
        done
        if [[ $updated -eq 1 ]]; then
            log "[INFO]Update version successfully."
            echo "[INFO]Update version successfully."
        fi
    fi
    
    
    # 标记升级成功
    touch $update_success
}


#==========================================================================
#   Name:           restart_service
#   Description:    restart the service which need restart
#   Input: 
#   Return:
#==========================================================================
function restart_service()
{
    if [ -d "${micro_service_path}/$1" ];then #判断节点有服务再替换打补丁
        bash /opt/oss/manager/agent/bin/ipmc_adm -cmd restartapp -app $1
        log "[INFO]restarted $1"
    else
        log "[INFO]No upgrade was required $1."
    fi
}

#==========================================================================
#   Name:           restart_services
#   Description:    restart the services which need restart
#   Input: 
#   Return:
#==========================================================================
function restart_services()
{

    #涉及的服务统一在这里填写，用空格分隔，重启服务会调用
    service_list=(CampusCfgService CampusFabricService RadiusServerService CampusAccountService CampusBaseService PONMgmtService)
    
    log "Restart service begin."
    for service in ${service_list[@]}
    do
        restart_service ${service}
    done
   
}



#==========================================================================
#   Name:           Entrance
#   Description:    Main Function
#   Input: 
#   Return:
#==========================================================================
function main()
{   
    log "-------- Update begin for note--------"
    check_need_update
    
    # $1 :jar名 
    # $2 :服务名
    
    #DTS2022091414712
    upgrade "fabric-util-21.0120.0300.0001.jar" "CampusFabricService"
    upgrade "virtual-network-service-21.0120.0300.0001.jar" "CampusFabricService"
    upgrade "virtual-network-ui-21.0120.0300.0001.jar" "CampusFabricService"
    
    #DTS2022092218076
    upgrade "fabric-manager-service-21.0120.0300.0001.jar" "CampusFabricService"
    upgrade "security-access-service-21.0120.0300.0001.jar" "CampusFabricService"
    
    #DTS2022090712470
    upgrade "net-fabric-service-21.0120.0300.0001.jar" "CampusFabricService"

    #DTS2022092402258 DTS2022090911018 DTS2022092805441 DTS2022093001030
    upgrade "radius-authentication-service-21.0120.0300.0001.jar" "RadiusServerService"
    
    #DTS2022092402258 DTS2022090913948
    upgrade "radius-baseservices-21.0120.0300.0001.jar" "RadiusServerService"
    
	#DTS2022090913948 DTS2022091314312 
	upgrade "accesscommon-cbb-utils-21.0120.0300.0001.jar" "RadiusServerService"
	upgrade "account-res-21.0120.0300.0001.jar" "RadiusServerService"
	upgrade_RadiusServerService_configuration
	
    #DTS2022092409171
    upgrade "ldap-mgr-module-21.0120.0300.0001.jar" "CampusAccountService"
    
    #DTS2022092409257
    upgrade "accountservice-iaereconciliation-module-21.0120.0300.0001.jar" "CampusAccountService"
    
    #DTS2022092201372
    upgrade "license-branch-service-21.0120.0560.0200-SNAPSHOT.jar" "CampusBaseService"
        
    #DTS2022092807292 
    upgrade "netcfg-device-adapter-pkg-service-21.0120.0300.0001.jar" "CampusCfgService"
    upgrade_ap_IotConfig
    
    #DTS2022092716071
    upgrade_PONMgmtService
    
    #更新版本
    update_version "/opt/oss/share"
    update_version "/var/share/oss"

    restart_services
    
    log "-------- Update end for note--------"
    
}

main "$@"