#! /bin/bash
curpath=$(cd `dirname $0`; pwd)
env_file=${curpath}/../env/env.properties
RETURN_AUTH_FAILED=-1
RETURN_SUCCESS=0
RETURN_ERROR=1

TIME_STAMP=$(date '+%s')
BASH_PID=$$

if [ -e ${env_file} ]
then
    env_file=/opt/arbitration/env/env.properties
fi
LOG_PATH=/var/log/arbitration/${INDEX}
if [ ! -d ${LOG_PATH} ]
then
    mkdir -p ${LOG_PATH}
fi

PARP_ERROR_MSG="Parameters invalid. For Example :
    bash arbiter_commissioning.sh -cmd modifyip -oldip xxx -newip xxx -nic bondx [-oldnetmask xxx -newnetmask xxx]
    bash arbiter_commissioning.sh -cmd configip -oldip xxx -newip xxx"

#################################
# 是否已经存在日志，绕接3份日志
#################################
function is_exist_log()
{
    LOG_FILE="${LOG_PATH}/arbiter_commissioning.log"
    if [ -f "${LOG_FILE}" ]
    then
        local file_size=$(du -sk "${LOG_FILE}" | awk '{print $1}')
        if [ ${file_size} -lt 10240 ]
        then
            return
        fi
    fi

    if [ -f "${LOG_FILE}.3" ]
    then
        rm -f "${LOG_FILE}.3"
    fi

    for (( i = 2; i > 0; i-- ))
    do
        if [ -f "${LOG_FILE}.${i}" ]
        then
            local j=$(expr ${i} + 1)
            mv "${LOG_FILE}.${i}" "${LOG_FILE}.${j}"
        fi
    done

    if [ -f "${LOG_FILE}" ]
    then
        tar zcvf "${LOG_FILE}.1" "${LOG_FILE}"
        rm -f "${LOG_FILE}"
    fi

}

#################################
# 脚本退出
#################################
function do_exit()
{
    local errorCode="$1"
    local logType="$2"
    local logInfo="$3"

    if [ "${logType}" = "ERROR" ]
    then
        echo "${logInfo}"
    fi
    write_logger "${logType}" "${logInfo}"
    exit ${errorCode}
}

#################################
# 日志记录
#################################
function write_logger()
{
    local logType="$1"
    local logInfo="$2"

    echo ${logType} | grep -i "ERROR" &>/dev/null && Level="ERROR"
    echo ${logType} | grep -i "WARN" &>/dev/null && Level="WARN"
    [ -z "${Level}" ] && Level="INFO"

    echo "${logInfo}" | grep "parameter is incorrect" >/dev/null
    if [ $? -eq 0 ]
    then
        echo "${PARP_ERROR_MSG}"
    fi
    echo "[$(date '+%Y-%m-%d %H:%M:%S')] [${BASH_PID}] | ${Level} ${logInfo}" >> "${LOG_FILE}"
}

#################################
# 检查参数
#################################
function check_param()
{
    local param_num=$#
    if [ ${param_num} -lt 6 -o ${param_num} -gt 12 ]
    then
        do_exit ${RETURN_ERROR} "ERROR" "The parameter is incorrect."
    fi
    for i in {1..6}
    do
        [[ ${1#-} == "cmd" ]] && { CMD=$2;shift 2;continue; }
        [[ ${1#-} == "oldip" ]] && { OLD_IP=$2;shift 2;continue; }
        [[ ${1#-} == "newip" ]] && { NEW_IP=$2;shift 2;continue; }
        [[ ${1#-} == "nic" ]] && { NIC=$2;shift 2;continue; }
        [[ ${1#-} == "oldnetmask" ]] && { OLD_MASK=$2;shift 2;continue; }
        [[ ${1#-} == "newnetmask" ]] && { NEW_MASK=$2;shift 2;continue; }
    done
    if [ ${param_num} -gt 6 ]
    then
        [ -z "${NIC}" ] && do_exit ${RETURN_ERROR} "ERROR" "The nic parameter is incorrect."
    fi
    if [ ${param_num} -gt 8 ]
    then
        [ -z "${OLD_MASK}" ] && do_exit ${RETURN_ERROR} "ERROR" "The oldnetmask parameter is incorrect."
        [ -z "${NEW_MASK}" ] && do_exit ${RETURN_ERROR} "ERROR" "The newnetmask parameter is incorrect."
    fi
    [ -z "${CMD}" ] && do_exit ${RETURN_ERROR} "ERROR" "The cmd parameter is incorrect."
    [ -z "${OLD_IP}" ] && do_exit ${RETURN_ERROR} "ERROR" "The oldip parameter is incorrect."
    [ -z "${NEW_IP}" ] && do_exit ${RETURN_ERROR} "ERROR" "The newip parameter is incorrect."
    
    # 参数类型校验
    check_cmd_type ${param_num}
    # 参数格式校验
    check_param_format
    
    write_logger "INFO" "[cmd=${CMD}] [oldip=${OLD_IP}] [newip=${NEW_IP}] [nic=${NIC}] [oldnetmask=${OLD_MASK}] [newnetmask=${NEW_MASK}]"
}

#################################
# 参数类型校验
#################################
function check_cmd_type()
{
    local param_num=$1
    if [ "${CMD}" = "modifyip" -a "${NIC}" = "" ]
    then
        do_exit ${RETURN_ERROR} "ERROR" "The nic is null, parameter is incorrect."
    fi
    if [ "${CMD}" = "configip" -a ${param_num} -ne 6 ]
    then
        do_exit ${RETURN_ERROR} "ERROR" "The number of parameter is incorrect."
    fi
}

#################################
# 校验IP合法性
#################################
function check_param_format()
{
    # 检查输入ip格式
    check_ip "${OLD_IP}" || let ret=ret+1
    if [ $? -ne 0 ]
    then
        do_exit ${RETURN_ERROR} "ERROR" "The old IP address format is incorrect. Please confirm the input parameter."
    fi
    check_ip "${NEW_IP}" || let ret=ret+1
    if [ $? -ne 0 ]
    then
        do_exit ${RETURN_ERROR} "ERROR" "The new IP address format is incorrect. Please confirm the input parameter."
    fi
    if [ "${OLD_IP}" = "${NEW_IP}" ]
    then
        do_exit ${RETURN_ERROR} "ERROR" "The IP address does not change. Please check the input."
    fi
    # 检查输入子网掩码格式
    if [ -n "${OLD_MASK}" ]
    then
        check_netmask_format "${OLD_MASK}" "${OLD_IP}"
        check_netmask_format "${NEW_MASK}" "${NEW_IP}"
    fi
    if [ -n "${NIC}" ]
    then
        check_network_port
    fi
}

#################################
# 检查执行用户
#################################
function check_user()
{
    userid=$(id | awk -F "[()]" '{print $2}')
    if [ "${userid}" != "root"  ]
    then
        do_exit ${RETURN_ERROR} "ERROR" "The script must be run with root, exit."
    fi
}

#################################
# 检查软件版本,R22C00及以上版本支持
#################################
function check_version()
{
    version_file="${curpath}/../conf/version.flag"
    if [ ! -f ${version_file} ]
    then
        do_exit ${RETURN_AUTH_FAILED} "ERROR" "The current version does not support modify, exit."
    fi
    version="$(cat ${version_file})"
    if [[ ! "${version}" =~ ^V100R02[2-9]+ ]]
    then
        do_exit ${RETURN_ERROR} "ERROR" "The current version does not support modify, exit."
    fi
    write_logger "INFO" "The current version is ${version}"
}

#################################
#  获取OS版本
#################################
function get_os_type()
{
    if [ -e /etc/SuSE-release ]
    then
        linux_type="SUSE"
    fi
    if [ -e /etc/centos-release ]
    then
        cat /etc/centos-release |grep "CentOS" >/dev/null
        if [ $? -eq 0 ]
        then
            linux_type="CentOS"
        fi
    fi
    if [ -e /etc/redhat-release ]
    then
        cat /etc/redhat-release |grep "Red Hat" >/dev/null
        if [ $? -eq 0 ]
        then
            linux_type="Redhat"
        fi
    fi
    if [ -e /etc/euleros-release ]
    then
        cat /etc/euleros-release  |grep "Euler" >/dev/null
        if [ $? -eq 0 ]
        then
            linux_type="EulerOS"
        fi
    fi
    write_logger "INFO" "The OS version is ${linux_type}"
}

#################################
#  检查本地是否存在待修改的ip
#################################
function check_local_ip() {
    if [ "${CMD}" = "modifyip" ]
    then
        ifconfig ${NIC} | grep -w ${OLD_IP} > /dev/null
        if [ $? -ne 0 ]
        then
            do_exit ${RETURN_ERROR} "ERROR" "The local IP address does not match ${OLD_IP} on ${NIC}. Please check the input parameters."
        fi
    fi
    # 检查是否旧ip是否是仲裁软件使用的ip
    local ret=1
    arbitrations=$(ls /opt | egrep "^arbitration$|^arbitration_[1-9]$")
    for arb_file in ${arbitrations}
    do
        cat /opt/${arb_file}/conf/arbitration_conf.properties | grep -w "${OLD_IP}" > /dev/null
        if [ $? -eq 0 ]
        then
            ret=0
            break
        fi
    done
    if [ ${ret} -ne 0 ]
    then
        do_exit ${RETURN_ERROR} "ERROR" "The old IP address is not the IP address for communication between arbitration nodes."
    fi
}

#################################
#  检查本地是否存在待修改的ip
#################################
function check_local_mask() {
    if [ "${CMD}" = "modifyip" -a "${OLD_MASK}" != "" ]
    then
        path_prefix="/etc/sysconfig/network-scripts/ifcfg-${NIC}"
        [[ "${linux_type}" = "SUSE"  ]] && path_prefix="/etc/sysconfig/network/ifcfg-${NIC}"
        local_mask=$(cat ${path_prefix} | grep NETMASK | awk -F'=' '{print $2}')
        if [ "${local_mask}" != "${OLD_MASK}" ]
        then
            do_exit ${RETURN_ERROR} "ERROR" "The local subnet mask does not match ${OLD_MASK} on ${NIC}. Please check the input parameters."
        fi
    fi
}

#################################
#  检查目标ip是否被占用
#################################
function check_dest_ip_occupy() {
    if [ "${CMD}" = "modify" ]
    then
        local ping_cmd="ping -c 3 ${NEW_IP}"
        if [ "${NEW_IP_TYPE}" = "IPv6" ]
        then
            ping6 -c 3 ${NEW_IP} >> "${LOG_FILE}"
        else
            ping -c 3 ${NEW_IP} >> "${LOG_FILE}"
        fi
        if [ $? -eq 0 ]
        then
            do_exit ${RETURN_ERROR} "ERROR" "The IP address to be changed has been occupied. Please replan the IP address."
        fi
    fi
}

##################################################
# 校验IP合法性
##################################################
function check_ip() {
    [[ $1 =~ ^([1-9][0-9]?|1[0-9][0-9]|2[0-4][0-9]|25[0-5])(\.([1-9]?[0-9]|1[0-9][0-9]?|2[0-4][0-9]|25[0-5])){3}$ ]] ||
    [[ $1 =~ ^(([0-9a-fA-F]{1,4}:){7,7}[0-9a-fA-F]{1,4}|([0-9a-fA-F]{1,4}:){1,7}:|([0-9a-fA-F]{1,4}:){1,6}:[0-9a-fA-F]{1,4}|([0-9a-fA-F]{1,4}:){1,5}(:[0-9a-fA-F]{1,4}){1,2}|([0-9a-fA-F]{1,4}:){1,4}(:[0-9a-fA-F]{1,4}){1,3}|([0-9a-fA-F]{1,4}:){1,3}(:[0-9a-fA-F]{1,4}){1,4}|([0-9a-fA-F]{1,4}:){1,2}(:[0-9a-fA-F]{1,4}){1,5}|[0-9a-fA-F]{1,4}:((:[0-9a-fA-F]{1,4}){1,6})|:((:[0-9a-fA-F]{1,4}){1,7}|:)|fe80:(:[0-9a-fA-F]{0,4}){0,4}%[0-9a-zA-Z]{1,}|::(ffff(:0{1,4}){0,1}:){0,1}((25[0-5]|(2[0-4]|1{0,1}[0-9]){0,1}[0-9])\.){3,3}(25[0-5]|(2[0-4]|1{0,1}[0-9]){0,1}[0-9])|([0-9a-fA-F]{1,4}:){1,4}:((25[0-5]|(2[0-4]|1{0,1}[0-9]){0,1}[0-9])\.){3,3}(25[0-5]|(2[0-4]|1{0,1}[0-9]){0,1}[0-9]))$ ]]
}

##################################################
# Description:  检查掩码格式
##################################################
function check_netmask_format() {
    local netmask=$1
    local ip=$2
    echo "${ip}" | grep ":" > /dev/null
    if [ $? -eq 0 ]
    then
        check_network_ipv6
    else
        if [[ ! ${netmask} =~ ^([1-9][0-9]?|1[0-9][0-9]|2[0-4][0-9]|25[0-5])(\.([1-9]?[0-9]|1[0-9][0-9]?|2[0-4][0-9]|25[0-5])){3}$ ]]
        then
            do_exit ${RETURN_ERROR} "ERROR" "Incorrect subnet mask format. Please confirm the input parameter."
        fi
    fi
}

##################################################
# Description:  检查ipv6掩码格式
##################################################
function check_network_ipv6()
{
    if ! egrep -q '^[0-9]+$' <<< "${netmask}"; then
        do_exit ${RETURN_ERROR} "ERROR" "Incorrect subnet mask format. Please confirm the input parameter."
    fi

    if [ "${netmask}" -lt 0 -o "${netmask}" -gt 128 ]; then
        do_exit ${RETURN_ERROR} "ERROR" "Incorrect subnet mask format. Please confirm the input parameter."
    fi
}

##################################################
# Description:  检查网口格式
##################################################
function check_network_port()
{
    if [[ ! ${NIC} =~ ^bond[0-9]+ ]] && [[ ! ${NIC} =~ ^eth[0-9]+ ]]
    then
        do_exit ${RETURN_ERROR} "ERROR" "Incorrect nic format. Please confirm the input parameter."
    fi
}

##################################################
# Description:  检查修改前后的ip协议是否一致
##################################################
function check_ip_protocol_consistent()
{
    OLD_IP_TYPE="IPv4"
    echo ${OLD_IP} | grep ":" > /dev/null
    if [ $? -eq 0 ]
    then
        OLD_IP_TYPE="IPv6"
    fi
    
    NEW_IP_TYPE="IPv4"
    echo ${NEW_IP} | grep ":" > /dev/null
    if [ $? -eq 0 ]
    then
        NEW_IP_TYPE="IPv6"
    fi
    if [ "${OLD_IP_TYPE}" !=  "${NEW_IP_TYPE}" ]
    then
        do_exit ${RETURN_ERROR} "ERROR" "The IP network protocol cannot be modified."
    fi
}

##################################################
# Description:  检查站点是否支持修改系统ip
##################################################
function check_if_modify_ip_allowed()
{
    site_type=$(get_arb_site_type)
    if [ "${site_type}" != "third" -a "${CMD}" = "modifyip" ]
    then
        do_exit ${RETURN_ERROR} "ERROR" "The system IP address cannot be changed at the current site."
    fi
}

#################################
#  获取仲裁站点类型
#################################
function get_arb_site_type()
{
    cat /opt/arbitration-monitor/conf/aam.properties | grep "aam.local.role" | awk -F'=' '{print $2}'
}

#################################
#  修改仲裁配置信息
#################################
function modify_arb_config()
{
    site_type=$(get_arb_site_type)
    modify_arb_config_file "/opt/arbitration-monitor/conf/aam.properties"
    modify_arb_config_file "/opt/arbitration-monitor/conf/persistkv.conf"
    modify_arb_config_file "/opt/arbitration/conf/arbitration_conf.properties"
    if [ "${site_type}" != "third" ]
    then
        echo "Succeeded in modifying the arbitration software IP address."
        return
    fi
    
    # 三方站点可能存在多套仲裁需要全部匹配
    arbitrations=$(ls /opt | grep "^arbitration_[1-9]$")
    for arb_file in ${arbitrations}
    do
        modify_arb_config_file "/opt/${arb_file}/conf/arbitration_conf.properties"
    done
    
    arbitration_monitors=$(ls /opt | grep "^arbitration-monitor_[1-9]$")
    for monitor_file in ${arbitration_monitors}
    do
        modify_arb_config_file "/opt/${monitor_file}/conf/aam.properties"
        modify_arb_config_file "/opt/${monitor_file}/conf/persistkv.conf"
    done
    echo "Succeeded in modifying the arbitration software IP address."
}

#################################
#  刷新known_hosts文件信息
#################################
function refrech_known_hosts()
{
    local old_ip=$1
    local new_ip=$2
    if [ -f /root/.ssh/known_hosts ]
    then
        sed -i "s#\<${old_ip}\>#${ip_addr}#g" /root/.ssh/known_hosts 2>/dev/null
    fi
    
    if [ -f /home/ossadm/.ssh/known_hosts ]
    then
        sed -i "s#\<${old_ip}\>#${ip_addr}#g" /home/ossadm/.ssh/known_hosts 2>/dev/null
    fi
}

#################################
#  修改仲裁配置文件信息
#################################
function modify_arb_config_file()
{
    file_name=$1
    if [ ! -f ${file_name} ]
    then
        do_exit ${RETURN_ERROR} "ERROR" "Modify arbitration configuration Failed, the arbitration config file ${file_name} does not exist., exit."
    fi
    sed -i "s#\<${OLD_IP}\>#${NEW_IP}#g" ${file_name}
    write_logger "INFO" "Modify Modify the IP address configuration of the ${file_name} file, oldIp is ${OLD_IP}, newIp is ${NEW_IP}."
}

#################################
# 修改三方站点系统ip
#################################
function modify_ip()
{
    get_os_type

    path_prefix=""
    path_route_prefix=""
    ip_name="IPADDR"
    if [ "${linux_type}" = "SUSE" ]
    then
        path_prefix="/etc/sysconfig/network/ifcfg-"
        [[ "${NEW_IP_TYPE}" = "IPv6" ]] && ip_name="IPADDR6"
    elif [ "${linux_type}" = "CentOS" -o "${linux_type}" = "Redhat" -o "${linux_type}" = "EulerOS" ]
    then
        path_prefix="/etc/sysconfig/network-scripts/ifcfg-"
        [[ "${NEW_IP_TYPE}" = "IPv6" ]] && ip_name="IPV6ADDR"
    fi

    ip_config_file=${path_prefix}${NIC}
    if [ ! -f ${ip_config_file} ]
    then
        do_exit ${RETURN_AUTH_FAILED} "ERROR" "The arbitration config file ${file_name} does not exist, exit."
    fi
    local ret=0
    sed -i "s#^${ip_name}.*#IPADDR='${NEW_IP}'#g" $ip_config_file 2>/dev/null || let ret=ret+1
    if [ -n "${NEW_MASK}" ]
    then
        sed -i "s#^NETMASK.*#NETMASK='${NEW_MASK}'#g" $ip_config_file 2>/dev/null || let ret=ret+1
    fi
    sed -i "s#\<${OLD_IP}\>#${NEW_IP}#g" /etc/ssh/sshd_config 2>/dev/null || let ret=ret+1
    if [ ${ret} -ne 0 ]
    then
        do_exit ${RETURN_ERROR} "ERROR" "Failed to change the system IP address of the Arbitration node."
    fi
    if [ "${linux_type}" = "SUSE" ]
    then
        service network restart >> "${LOG_FILE}" 2>&1 || let ret=ret+1
    elif [ "${linux_type}" = "CentOS" -o "${linux_type}" = "Redhat" -o "${linux_type}" = "EulerOS" ]
    then
        systemctl restart network.service >> "${LOG_FILE}" 2>&1 || let ret=ret+1
    fi
    restart_sshd
    if [ ${ret} -ne 0 ]
    then
        echo "Failed to start the Arbitration node network service."
        do_exit ${RETURN_ERROR} "ERROR" "Failed to start the Arbitration node network service."
    fi
    echo "Succeeded in Changing the System IP Address of the Arbitration Node."
}

#################################
# 重启sshd服务
#################################
function restart_sshd() {
    if [ ${linux_type} = "SUSE" ]
    then
        service sshd restart >> "${LOG_FILE}" 2>&1
    elif [ ${linux_type} = "CentOS" -o ${linux_type} = "Redhat" -o ${linux_type} = "EulerOS" ]
    then
        systemctl restart sshd.service >> "${LOG_FILE}" 2>&1
    fi
}

#################################
# 修改三方站点系统ip
#################################
function start_commissioning_arb_ip()
{
    modify_arb_config
    refrech_known_hosts
    if [ "${CMD}" = "configip" ]
    then
        do_exit ${RETURN_SUCCESS} "INFO" "Succeeded in modifying the arbitration software IP address."
    fi
    modify_ip
    do_exit ${RETURN_SUCCESS} "INFO" "Succeeded in Changing the System IP Address of the Arbitration Node."
}

#################################
# 调整仲裁ip入口函数
#################################
function main()
{
    is_exist_log
    check_user
    check_param "$@"
    check_ip_protocol_consistent
    check_local_ip
    check_local_mask
    check_dest_ip_occupy
    check_if_modify_ip_allowed
    start_commissioning_arb_ip
}

main "$@"

