#!/bin/bash
SCRIPT_PATH=$(cd $(dirname $0);pwd)
SELF_FILE=$(basename $0)
TAIL=$(date +%Y%m%d%H%M%S)
LOG_FILE="/var/log/${SELF_FILE//.sh/}_${TAIL}.log"
TEMP_LOG_FILE="/tmp/sek_root_log.log"
NODE_ID_FILE="/opt/oss/manager/var/agent/mcagentid.conf"

# 工程临时文件目录
TMP_DIR="/opt/engr_tmp"
# 成功执行sek_root.sh脚本后生成的标志文件
TMP_SUCCESS="${TMP_DIR}/sek_root_success"


function LOG()
{
    if [ "$1" = "ERROR" -o "$1" = "error" ]
    then
        Level="ERROR"
        echo "[$(date +%Y)-$(date +%m)-$(date +%d) $(date +%H):$(date +%M):$(date +%S)] [${SELF_FILE}] ${Level} | $2" 1>&2
        echo "[$(date +%Y)-$(date +%m)-$(date +%d) $(date +%H):$(date +%M):$(date +%S)] [${SELF_FILE}] ${Level} | $2" >> "${LOG_FILE}"
        logger -p local0.error "[${SELF_FILE}] ${Level} | $2"
    else
        Level="INFO"
        echo "[$(date +%Y)-$(date +%m)-$(date +%d) $(date +%H):$(date +%M):$(date +%S)] [${SELF_FILE}] ${Level} | $2" | tee -a "${LOG_FILE}"
        logger -p local0.info "[${SELF_FILE}] ${Level} | $2"
    fi
}

#create action.json check file
createCheckFile()
{
    echo "#!/bin/bash" >/opt/install/check_sek_root_result.sh
    echo "echo  RESULT:1" >>/opt/install/check_sek_root_result.sh
    chmod 750 /opt/install/check_sek_root_result.sh
}

getcheckResult()
{
   if [ $1 -eq 0 ]
   then
       echo "#!/bin/bash
echo  RESULT:100 && rm -rf /opt/install/*
   ">/opt/install/check_sek_root_result.sh
   else
       echo "#!/bin/bash
echo  RESULT:255 && rm -rf /opt/install/check_sek_root_result.sh
   ">/opt/install/check_sek_root_result.sh
   fi
   chmod 750 /opt/install/check_sek_root_result.sh > /dev/null 2>&1
   return $1
}

#获取操作系统类型
function checkOS()
{
    if [ -f /etc/SuSE-release ];then
        OSType="suse"
    elif [ -f /etc/centos-release ] && [ "$(grep CentOS /etc/centos-release)" ]
    then
        OSType="centos"
    elif [ -f /etc/euleros-release ] && [ "$(grep EulerOS /etc/euleros-release)" ]
    then
        OSType="EulerOS"
    elif [ -f /etc/kylin-release ] && [ "$(grep Kylin /etc/kylin-release)" ]
    then
        OSType="Kylin"
    fi
    LOG "INFO" "OSType is ${OSType}"
}

# 设置特定用户密码不过期
function set_expiration()
{
    LOG "INFO" "Set the user password validity period."
    chage -M 99999 dbuser >/dev/null 2>&1
    chage -M 99999 iscript >/dev/null 2>&1
    chage -M 99999 omm >/dev/null 2>&1
    chage -M 99999 ommdba >/dev/null 2>&1
}

# 执行后置刷权限
function postdo()
{
    LOG "INFO" "Executing Function: postdo"
    #修改su配置，禁用su之前的用户的环境变量
    if [ -f "/etc/default/su" ];then
        sed -i "/ALWAYS_SET_PATH/ s/no/yes/g" /etc/default/su
    fi

    #控制文件权限

    find /opt/install -type f -name "*.sh" | xargs -i echo '"{}"' | xargs chmod 550
    find /opt/tools -type f -name "*.sh" | xargs -i echo '"{}"' | xargs chmod 550

    for targetDir in "/opt/install" "/opt/SEK" "/opt/NCEICMR" "/var/ICMR/" "/var/log/tools/"
    do
        if [ -d "${targetDir}" ]
        then
            find "${targetDir}" -type f -name "*.log" -o -name "*.sh.log*" | xargs -i echo '"{}"' | xargs chmod 640
            find "${targetDir}" -type f -name "*.xml" -o -name "*.properties" -o -name "*.conf" -o -name "*.ini" -o -name "*.cfg" -o -name "*.gif" -o -name "*.png" | xargs -i echo '"{}"' | xargs chmod 400
            find "${targetDir}" -type f -name "*.json" -o -name "*.xml" -o -name "*.txt" -o -name "*.conf" -o -name "*.properties" -o -name "*.csv" | xargs -L 1 chmod 400 >/dev/null 2>&1
        fi
    done
}

# 整理节点信息
function init_data()
{
    tmpPath="/tmp/tmp_iplist"
    [ -d "${tmpPath}" ] && rm -rf "${tmpPath}"
    mkdir -p "${tmpPath}"
    chown ossadm:ossgroup "${tmpPath}"
    chmod 750 "${tmpPath}"
    local count=0
    while true
    do
        # 判断是否获取到节点信息，没有获取到重试两次
        su - ossadm -c "cd /opt/oss/manager/tools/resmgr;./queryproduct.sh -pn all -output ${tmpPath}" > /dev/null 2>&1
        result=$(ls "${tmpPath}" 2>/dev/null | wc -l)
        if [ ${result} -eq 0 ]
        then
            sleep 30
            let count++
            if [ ${count} -eq 10 ]
            then
                LOG "ERROR" "Failed to query product nodes."
                return 1
            fi
        else
            break
        fi
    done
    nodelist=$(cat "${tmpPath}"/nodes_*.json 2>/dev/null | sed 's/,/\n/g'| grep nodemgrip | awk -F\" '{print $4}' 2>/dev/null)
    omplist=$(cat /opt/oss/manager/var/agent/managerip.conf | grep "^managerip" | awk -F= '{print $2}' | \
    sed 's/,/\n/g' | grep -v "${SETIP}" | sort -u)
    nodelist=($(echo "${omplist} ${nodelist}" | sed 's/ /\n/g' | grep -v "${localip}" | sort -u))
    rm -rf "${tmpPath}"
}

# 获取OMP_01的ip地址
function get_omp01_ip()
{
    local manager_ips=$(cat /opt/oss/manager/var/agent/managerip.conf | \
    grep -w 'managerip' | awk -F= '{print $2}' | sed 's/,/\n/g')
    local node_ips=$(cat /opt/oss/manager/var/tenants/manager/nodelist.json | python -c "import json; import sys; \
    obj=json.load(sys.stdin); print (obj['nodeList']['0']['IPAddresses'])" | sed 's/,/\n/g' | grep -w 'IP' | \
    awk -F"'" '{print $(NF-1)}')
    SETIP=""
    for tmp_node0 in ${node_ips}
    do
        for tmp_manager in ${manager_ips}
        do
            if [ "${tmp_manager}" == "${tmp_node0}" ]
            then
                SETIP="${tmp_manager}"
                return 0
            fi
        done
    done
}

# 判断节点是否需要执行
function check_node_message()
{
    # 调用用户禁用函数标志位，0表示不执行，1表示执行
    CONFIG_FLAG=0
    localip=$(su - ossadm -c '. /opt/oss/manager/bin/engr_profile.sh;python -c "from util import common;print(common.get_local_ip())"')
    get_omp01_ip
    if [ "${localip}" == "${SETIP}" ]
    then
        init_data
        if [ $? -ne 0 ]
        then
            return 1
        else
            CONFIG_FLAG=1
        fi
    fi
}

# 启动sshd加固
function config_sshd()
{
    LOG "INFO" "Start to enable user isolation."
    sshd_tool="/opt/oss/manager/agent/bin/sshd_ipsadm.sh"
    if [ -f "${sshd_tool}" ]
    then
        su - ossadm -c "bash ${sshd_tool} -cmd set >/dev/null 2>&1"
        if [ $? -ne 0 ]
        then
            su - admrunuser -c "bash ${sshd_tool} -cmd restore >/dev/null 2>&1"
            LOG "ERROR" "Execute '/opt/oss/manager/agent/bin/sshd_ipsadm.sh -cmd set' Failed."
            return 1
        else
            LOG "INFO" "Succeeded in enabling user isolation."
            return 0
        fi
    else
        LOG "ERROR" "/opt/oss/manager/agent/bin/sshd_ipsadm.sh not found."
        return 1
    fi
}

function delete_arr_ele()
{
    local delete_ip=$1
    for i in "${!nodelist[@]}"
    do
        if [ "${delete_ip}" == "${nodelist[i]}" ]
        then
            unset 'nodelist[i]'
        fi
    done
}

# 判断单节点sek_root脚本是否执行完成
function check_node()
{
    local node_ip=$1
    ssh_para="-o ConnectTimeout=10 -o stricthostkeychecking=no -o ConnectionAttempts=1 -o ServerAliveInterval=3"
    check_result=$(su - ossadm -c "ssh ${ssh_para} \"${node_ip}\" \"([ -n "${TMP_SUCCESS}" ] && [ -f "${TMP_SUCCESS}" ]) && (rm -f "${TMP_SUCCESS}" && echo "success") || echo "false"\"" 2>/dev/null)
    if [ "${check_result}" == "success" ]
    then
        LOG "INFO" "Checking ${node_ip} succeeded."
        delete_arr_ele "${node_ip}"
        echo "${node_ip}" >> "${SCRIPT_PATH}"/success_node.log
    fi
}

# 判断除OMP主节点的其他sek_root操作已完成
function check_finish()
{
    local begin_time=$((10#$(date "+%s")))
    [ ${#nodelist[@]} -eq 0 ] && return 0
    success_flag="${SCRIPT_PATH}/success_node.log"
    LOG "INFO" "Start to check nodes(${nodelist[*]})."
    if [ -f ${success_flag} ]
    then
        while read line
        do
            LOG "INFO" "Checking ${line} succeeded."
            delete_arr_ele "${line}"
        done <${success_flag}
    fi
    let end_time=${begin_time}+600+${#nodelist[@]}*60
    let time_out=${end_time}-${begin_time}
    if [ ${time_out} -ge 1800 ]
    then
        let end_time=${begin_time}+1800
    fi
    while true
    do
        [ ${#nodelist[@]} -eq 0 ] && break
        for str_node in "${nodelist[@]}"
        do
            check_node "${str_node}"
        done
        now_time=$((10#$(date "+%s")))
        if [ ${now_time} -ge ${end_time} ]
        then
            LOG "INFO" "Failed to check the node.(${nodelist[*]})"
            return 1
        fi
        sleep 3
    done
    LOG "INFO" "All nodes are checked successfully."
}

# 执行用户禁用，仅OMP01节点执行
function deny_user_ssh()
{
    if [ ${CONFIG_FLAG} -eq 1 ]
    then
        rm -f "${TMP_SUCCESS}"
        check_finish
        if [ $? -eq 0 ]
        then
            config_sshd
            if [ $? -ne 0 ]
            then
                LOG "ERROR" "Failed to enable user isolation."
                return 1
            fi
        else
            LOG "ERROR" "Failed to check other nodes."
            return 1
        fi
    else
        LOG "INFO" "This node(${localip}) not need to enable user isolation, SETIP is ${SETIP}."
        return 0
    fi
}

# 创建sek_root标志文件
function create_sek_tmp()
{
    [ -f "${TMP_SUCCESS}" ] && rm -f "${TMP_SUCCESS}"
    mkdir -p ${TMP_DIR} 2>/dev/null
    chown ossadm:ossgroup ${TMP_DIR} 2>/dev/null
    su - ossadm -c "touch ${TMP_SUCCESS}" &>>${LOG_FILE}
    if [ $? -eq 0 -a -f "${TMP_SUCCESS}" ]
    then
        LOG "INFO" "The flag file is created successfully."
    else
        LOG "ERROR" "Failed to create the flag file."
        return 1
    fi
}


function post_install()
{
    LOG "INFO" "Execute function: post_install"
    #后置处理
    if [ -f ${SCRIPT_PATH}/post_install.sh ];then
        bash ${SCRIPT_PATH}/post_install.sh 2>$TEMP_LOG_FILE
        if [ $? -eq 0 ];then
            LOG "INFO" "The ${SCRIPT_PATH}/post_install.sh script is executed successfully, and the backup customization file is updated successfully."
            rm -rf $TEMP_LOG_FILE
        else
            LOG "ERROR" "Failed to execute the ${SCRIPT_PATH}/post_install.sh script."
            ERR = 'cat $TEMP_LOG_FILE'
            LOG "ERROR" $ERR
            rm -rf $TEMP_LOG_FILE
            return 1
        fi
    fi
}

# 重置管理面升级标识文件
function remove_upgrade_flag()
{
    upgrade_flag=$(ls /opt/oss/log/ICMR/ 2>/dev/null | grep -i "upgrade.*flag")
    [ -n "${upgrade_flag}" ] && rm -f /opt/oss/log/ICMR/upgrade*.flag
}

# 判断是否为0号节点
# NODE_ID_FILE --平台存放节点id文件
function is_node_0()
{
    LOG "INFO" "Start execute function: is_node_0"
    if [ ! -f "${NODE_ID_FILE}" ]
    then
        LOG "INFO" "NODE_ID_FILE is not exist."
        return 1
    fi
    cat ${NODE_ID_FILE} | grep -qw "localnodeid=0"
    if [ $? -eq 0 ]
    then
        # 执行条件只在0号节点执行
        LOG "INFO" "Finish execute function: is_node_0"
        return 0
    fi
    LOG "INFO" "The script can be executed only on node 0 of NCE."
    return 1
}

# 创建/opt/upgrade目录并调整权限
function mkdir_upgrade()
{
    LOG "INFO" "Start execute function: mkdir_upgrade"
    # 执行条件只在0号节点执行,非0号节点退出
    is_node_0 || return 0
    # /opt/upgrade目录存在删除
    rm -rf /opt/upgrade
    # 创建目录
    mkdir -p /opt/upgrade
    # 刷新目录权限
    chmod 750 /opt/upgrade
    # 刷新目录属组（保证sopuser可写）
    chown sopuser:sopgroup /opt/upgrade
    LOG "INFO" "Finish execute function: mkdir_upgrade"
}

# 清除环境残留文件
function clean_env()
{
    tmp_dir=$(ls /tmp | grep "^tmplanfile")
    if [ -n "${tmp_dir}" ]
    then
        rm -rf /tmp/tmplanfile
    fi
    cbb_dir=$(ls /tmp | grep "^CBB")
    if [ -n "${cbb_dir}" ]
    then
        rm -rf /tmp/CBB_*
    fi
}

# 清理OMP节点的标志文件
function clean_tmp_file()
{
    if [ ${CONFIG_FLAG} -eq 1 ]
    then
        [ -f "${TMP_SUCCESS}" ] && rm -f "${TMP_SUCCESS}"
        if [[ -f "${TMP_SUCCESS}" ]]
        then
            LOG "ERROR" "Failed to delete the flag file."
        else
            LOG "INFO" "Succeeded in deleting the flag file."
        fi
    fi
}

# 开启root登录权限
function open_root_ssh()
{
    if [ -f "/opt/SEK/cmd/SetEnv.sh" ]
    then
        bash /opt/SEK/cmd/SetEnv.sh
    else
        LOG "ERROR" "/opt/SEK/cmd/SetEnv.sh does not exist. Please manually enable the SSH permission for the root user."
    fi
}

# 用于恢复由于Easysuite修改的配置项
function recover_sshd_for_tunnel()
{
    sed -i "/.*#Easysuite$/d" /etc/ssh/sshd_config
}

function open_port_listen() {
    if [ -f "/opt/oss/manager/adapter/common/resmgr/init_config.sh" ]
    then
        LOG "INFO" "Start to open port listening."
        su - ossadm -c "cd /opt/oss/manager/adapter/common/resmgr && bash init_config.sh &>$TEMP_LOG_FILE"
        if [ $? -ne 0 ]
            then
                LOG "ERROR" "Failed to open port listening."
                rm -rf $TEMP_LOG_FILE
                return 1
        fi
        LOG "INFO" "Port listening enabled successfully."
        rm -rf $TEMP_LOG_FILE
    fi
}

function sek_fi(){
    # 执行FI加固
    if [ -f ${SCRIPT_PATH}/sek_fi.sh ];then
        bash ${SCRIPT_PATH}/sek_fi.sh 2>$TEMP_LOG_FILE
        if [ $? -eq 0 ];then
            LOG "INFO" "Succeeded in executing the ${SCRIPT_PATH}/sek_fi.sh script."
            rm -rf $TEMP_LOG_FILE
        else
            LOG "ERROR" "Failed to execute the ${SCRIPT_PATH}/sek_fi.sh script."
            ERR=`cat $TEMP_LOG_FILE`
            LOG "ERROR" $ERR
            rm -rf $TEMP_LOG_FILE
            return 2
        fi
    fi
}
# 返回值0: 运行正常
#       1: 和root禁用无关的步骤失败
#       2：和root禁用相关的步骤失败，退出脚本前开启当前节点root登录权限
function main()
{
    LOG "INFO" "Start to execute the sek_root.sh script."

    # 生成加固日志文件
    createCheckFile

    # 获取os信息
    checkOS

    # 设置特定用户密码不过期
    set_expiration

    # 检查是否需要执行deny_user_ssh,获取节点信息失败时退出后续步骤
    check_node_message || return 1



    # 执行后置刷权限
    postdo >/dev/null 2>&1

    # 后置处理
    post_install || return 1

    # 安装时重置管理面升级标识文件
    remove_upgrade_flag

    # 创建/opt/upgrade目录并调整权限
    mkdir_upgrade

    # 清理环境
    clean_env

    # 南向ip禁用脚本 && 22端口监听
    open_port_listen || return 1

    # 防止sshd服务重启次数过多
    sleep 5

    # 执行sshd用户禁用，仅允许sopuser登录
    deny_user_ssh || return 2

    sek_fi || return 2

    recover_sshd_for_tunnel

    # 禁用root用户ssh登录权限
    if [ "${OSType}" == "suse" -o "${OSType}" == "EulerOS" ]
    then
        if [ "$1" == "id_sek_select_yes" ]
        then
            cd /opt/SEK/;bash RunSEK.sh
            local lockResult=$?
            if [[ ${lockResult} != 0 ]]
            then
                # 加固不成功,解加固后再退出
                cd /opt/SEK/cmd;bash SetEnv.sh >/dev/null 2>&1
                return 1
            fi
        fi
    fi

    # 创建sek_root标志文件
    create_sek_tmp || return 2
}


#加锁
function lock() {
    [[ -f /usr/lib/systemd/system/lock.service ]] || return 0
    systemctl enable lock &> /dev/null
    local lockResult=$?
    if [[ ${lockResult} != 0 ]]
    then
        #记录下失败日志
        LOG "ERROR" "Enable lock failed with ${lockResult}"
        return ${lockResult}
    fi
    systemctl start lock &> /dev/null
    lockResult=$?
    if [[ ${lockResult} != 0 ]]
    then
        #记录下失败日志
        LOG "ERROR" "Start lock failed with ${lockResult}"
    fi
    return ${lockResult}
}

main $*
result=$?
if [ ${result} -eq 0 ]
then
    LOG "INFO" "Succeeded in executing the sek_root.sh script."
    clean_tmp_file
elif [ ${result} -eq 1 ]
then
    LOG "ERROR" "Failed to execute the sek_root.sh script."
elif [ ${result} -eq 2 ]
then
    LOG "ERROR" "Failed to execute the sek_root.sh script."
    open_root_ssh
else
    LOG "ERROR" "Failed to execute the sek_root.sh script."
    open_root_ssh
fi

if [ ${result} -ne 0 ]
then
  getcheckResult ${result}
  exit 1
fi

lock
result=$?
getcheckResult ${result}
exit ${result}