#!/usr/bin/env bash

###############################################################################################
function fn_get_log_file_path()
{
    echo "${c_root_path}/cmd.check_process.log"
    return 0
}

###################################################
#
# 参数个数校验
#
###################################################
function fn_check_param_count()
{
    [[ $# != 0 ]] && fn_sys_log_info "Parameter incorrect." && return 1
    return 0
}

###################################################
#
# 判断节点密码是否正确
#
###################################################
function fn_check_ip_passwd()
{
    host_ip="$1"
    fn_sys_log_info "Password verification for [${host_ip}]."
    user="$2"
    fn_run_detect_passwd "${host_ip}" "${user}" "${remote_password}" "ls / >/dev/null 2>&1;" "5" 0
    result=$?
    [[ ${result} -eq 127 ]] && fn_sys_log_error "Timeout to wait for interactivity, Try again." && echo "false" && return 1
    [[ ${result} -eq 128 ]] && fn_sys_log_error "Failed to verify the password." && echo "false" && return 1
    fn_sys_log_info "Password verification succeeded." && echo "true" && return 0
}

###################################################
#
# 执行远端命令
#
###################################################
function fn_run_command_remote()
{
    fn_sys_log_info "Start executing remote cmds."
    host_ip="$1"
    user="$2"
    passwd="$3"
    command="$4"
    timeout="$5"
    is_log="$6"
    exec_cmd_remote ${host_ip} ${user} "${timeout}" "${command}" ${is_log}<<EOF
${passwd}
EOF
    result=$?
    [[ ${result} -ne 0 ]] && return ${result}
    fn_sys_log_info "Success to execute the remote command." && return 0

}

###################################################
#
# 执行远端命令，测试链接是否可以创建成功
#
###################################################
function fn_run_detect_passwd()
{
    fn_sys_log_info "Start executing remote cmds."
    host_ip="$1"
    user="$2"
    passwd="$3"
    command="$4"
    timeout="$5"
    is_log="$6"
    exec_cmd_remote ${host_ip} ${user} "${timeout}" "${command}" ${is_log}<<EOF
${passwd}
EOF
    result=$?
    [[ ${result} -ne 0 ]] && return ${result}
    fn_sys_log_info "Success to execute the remote command." && return 0

}

###################################################
#
# 清理远端工作空间
#
###################################################
function fn_clean_workspace_flag()
{
    fn_sys_log_info "Start to clean workspace."
    for each_ip in $(cat "${c_config_path}" | grep "${c_host_flag}" | awk -F'=' '{print $NF}' | awk -F"," '{ for(i=1; i<=NF; i++) print $i }'); do
        fn_sys_log_info "Clearing workspace on node [${each_ip}]"
        fn_run_command_remote "${each_ip}" "root" "${remote_password}" "rm -rf ${c_path_workspace} >/dev/null 2>&1; rm -rf ${flag_path} >/dev/null 2>&1;" "5" 1
        result=$?
        [[ ${result} -ne 0 ]] && fn_sys_log_error "Failed to clear workspace on node [${each_ip}]." && return 1
        fn_sys_log_info "Success to clear workspace on node [${each_ip}]."
    done
    return 0
}

###################################################
#
# 拉起远端检查
#
###################################################
function fn_invoke_local_check_remote()
{
    fn_sys_log_info "Start to invork local check on remote."
    for each_ip in $(cat "${c_config_path}" | grep "${c_host_flag}" | awk -F'=' '{print $NF}' | awk -F"," '{ for(i=1; i<=NF; i++) print $i }'); do
        fn_sys_log_info "Invoking on node [${each_ip}]"
        fn_run_command_remote "${each_ip}" "root" "${remote_password}" "nohup sh ${c_path_workspace}/shscript/module/check_local_launch.sh ${each_ip} > ${c_path_workspace}/${each_ip}_cmd.process_launch.log 2>&1 &" "5"  1
        result=$?
        [[ ${result} -ne 0 ]] && fn_sys_log_error "Failed to clear workspace on node [${each_ip}]." && return 1
        fn_sys_log_info "Success to clear workspace on node [${each_ip}]."
    done
    return 0
}

###################################################
#
# 远端检查是否均已结束
#
###################################################
function fn_check_result_remote()
{
    fn_sys_log_info "Start to invork local check on remote."
    local l_running_times=0
    local l_max_run_times=20
    while [ "${l_running_times}" -le "${l_max_run_times}" ]; do
        sleep 5

        fn_sys_log_info "Waiting for task complete: Elapsed time $((l_running_times*5)) Seconds. "
        l_pending_count=0
        for each_ip in $(cat "${c_config_path}" | grep "${c_host_flag}" | awk -F'=' '{print $NF}' | awk -F"," '{ for(i=1; i<=NF; i++) print $i }'); do
            fn_sys_log_info "Confirm the check result on the remote node [${each_ip}]."
            fn_run_command_remote "${each_ip}" "${g_remote_user}" "${remote_password}" "ls ${flag_path};" "5" 0
            result=$?
            [[ ${result} -ne 0 ]] && l_pending_count=$((l_pending_count+1)) && fn_sys_log_info "Remote node is executing: [${each_ip}]."
        done
        fn_sys_log_warn "[${l_pending_count}] process(es) are still running."
        [ "${l_pending_count}" -eq 0 ] && break
        l_running_times=$((l_running_times+1))
    done
    [[ "${l_running_times}" -le "${l_max_run_times}" ]] && fn_sys_log_info "The check on all nodes is complete." && return 0
    fn_sys_log_error "Check timeout failed and exited."
    return 1
}

###################################################
#
# 获取所有远程节点上的日志文件，并处理。
#
###################################################
function fn_process_result()
{
    fn_sys_log_info "Obtaining logs from remote nodes."
    for each_ip in $(cat "${c_config_path}" | grep "${c_host_flag}" | awk -F'=' '{print $NF}' | awk -F"," '{ for(i=1; i<=NF; i++) print $i }'); do
        file_name_ip=${each_ip}
        [[ ${each_ip} =~ ':' ]] && file_name_ip=$(echo ${each_ip} | sed 's/:/\\:/g')
        fn_cp_file_from_remote ${each_ip} "root" "240" "${c_path_workspace}/${file_name_ip}_cmd.process_launch.log" "${c_root_path}/log" ${remote_password}
        [[ $? != 0 ]] && fn_sys_log_error "Failed to obtain logs from the node [${each_ip}]." && return 1
    done

    return 0
}

###################################################
#
# 检查框架拷贝到其他节点
#
###################################################
function fn_cp_frame_remote()
{
    fn_sys_log_info "Copy the framework to all remote nodes."
    for each_ip in $(cat "${c_config_path}" | grep "${c_host_flag}" | awk -F'=' '{print $NF}' | awk -F"," '{ for(i=1; i<=NF; i++) print $i }'); do
        [[ "$(fn_cp_file_remote ${each_ip} "root" "240" ${c_root_path} ${c_path_workspace} ${remote_password})" == "false" ]] && fn_sys_log_error "Failed to copy the framework to the remote node [${each_ip}]." && return 1
    done
    return 0
}

###################################################
#
# 将检查框架拷贝到其他节点
#
###################################################
function fn_cp_file_remote()
{
    fn_sys_log_info "Copy the detection framework to the remote node."
    host_ip="$1"
    user="$2"
    timeout="$3"
    c_src_path="$4"
    c_path_des="$5"
    passwd="$6"
    cp_file_to_remote "${host_ip}" "${user}" "${timeout}" "${c_src_path}" "${c_path_des}" "0" <<EOF
${passwd}
EOF
    result=$?
    [[ ${result} -ne 0 ]] && fn_sys_log_error "Failed to copy files to the remote end." && echo "false" && return 1
    fn_sys_log_info "Success to copy files to the remote end." && echo "true" && return 0
}

###################################################
#
# 汇总日志文件
#
###################################################
function fn_cp_file_from_remote()
{
    fn_sys_log_info "Copy the log files of other nodes to the current node."
    host_ip="$1"
    user="$2"
    timeout="$3"
    c_src_path="$4"
    c_des_path="$5"
    passwd="$6"
    cp_file_from_remote "${host_ip}" "${user}" "${timeout}" "${c_src_path}" "${c_des_path}" "0" <<EOF
${passwd}
EOF
    result=$?
    [[ ${result} -ne 0 ]] && fn_sys_log_error "Failed to copy files from the remote end." && return 1
    fn_sys_log_info "Success to copy files from the remote end." && return 0
}


###################################################
#
# 解析日志，分析节点间rpm版本差异
#
###################################################
function fn_find_diff_rpm_ver()
{
    for each_ip in $(cat "${c_config_path}" | grep "${c_host_flag}" | awk -F'=' '{print $NF}' | awk -F"," '{ for(i=1; i<=NF; i++) print $i }'); do
        fn_sys_log_info "Start to parse node logs: [${each_ip}]"
        [[ ${each_ip} == ${omp_ip} ]] && continue
        fn_parses_rpm_log ${each_ip}
    done
}


###################################################
#
# 解析日志，判断返回正确与否
#
###################################################
function fn_adjust_result()
{
    result=0
    grep -ir "\[ERROR\]" "${c_log_file}" >/dev/null 2>&1;
    [[ $? -eq 0 ]] && result=$((result+1))

    find "${c_root_path}/log/" -name "*cmd.process_launch.log" | xargs -i cat {} | grep "\[ERROR\]" >> "${c_log_file}"
    [[ $? -eq 0 ]] && result=$((result+1))
    return ${result}
}

###################################################
#
# 解析其他节点日志文件，跟omp节点日志对比
#
###################################################
function fn_parses_rpm_log()
{
    local other_ip=$1
    omp_file_name_ip=${omp_ip}
    other_file_name_ip=${other_ip}

    for each_rpm in $(cat "${c_root_path}/log/${omp_file_name_ip}_cmd.process_launch.log" | grep "\[RPM_VER\]" | grep -v "grep" | awk -F '#' '{print $3}');do
        other_rpm=$(cat "${c_root_path}/log/${other_file_name_ip}_cmd.process_launch.log" | grep "#${each_rpm}#" | grep -v "grep" | awk -F '#' '{print $4}')
        [[ -z "${other_rpm}" ]] && continue
        omp_rpm=$(cat "${c_root_path}/log/${omp_file_name_ip}_cmd.process_launch.log" | grep "#${each_rpm}#" | grep -v "grep" | awk -F '#' '{print $4}')
        [[ ${other_rpm} != ${omp_rpm} ]] && fn_sys_log_error "[${other_ip}]: The RPM package version of the [${other_rpm}] is inconsistent with that of the OMP node."
    done
    return 0
}

###################################################
#
# 解析配置参数，获取remote节点.校验节点是否是合法ip
# 判断节点是否可连通，判断节点密码是否正确。
#
###################################################
function fn_check_remote_host()
{
    fn_sys_log_info "Start to get check node."
    for each_ip in $(cat "${c_config_path}" | grep "${c_host_flag}" | awk -F'=' '{print $NF}' | awk -F"," '{ for(i=1; i<=NF; i++) print $i }'); do
        fn_sys_log_info "check_ip is [${each_ip}]"
        [[ "$(fn_sys_is_valid_ip ${each_ip})" == "false" ]] && fn_sys_log_error "Token [${each_ip}] is not a valid ip" && return 1
        fn_sys_log_info "Check ip success."
        [[ "$(fn_ping_ip_ok ${each_ip})" == "false" ]] && fn_sys_log_error "Failed to connect to the remote node ${each_ip}." && return 1
        fn_sys_log_info "Check remote node ping success."
        fn_check_ip_passwd ${each_ip} ${g_remote_user}
        [[ $? -ne 0 ]] && fn_sys_log_error "Failed to log in to the remote node whose IP address is ${each_ip}." && return 1
        fn_sys_log_info "Check remote node password success."
    done
    return 0
}

###############################################################################################
function fn_init()
{
    readonly c_root_path=$(dirname $(readlink -f $0 ))
    source ${c_root_path}/shscript/utils/load_all_util.sh
    fn_sys_set_logging_verbose "$SYS_LOGGING_INFO"
    readonly c_log_file="$(fn_get_log_file_path)"
    rm -f "${c_log_file}"
    fn_sys_create_log_file "$c_log_file"
    fn_sys_set_log_file "$c_log_file"

    source "${c_root_path}/shscript/profile/check_profile.sh"

    return 0
}

###############################################################################################
function fn_precheck()
{
    fn_sys_log_info "Precheck before pre-installation check."
    fn_sys_assert_current_user_matched "root" || return 1
    fn_check_param_count "$@" || return 1
    fn_sys_assert_not_null "c_config_path" || return 1
    fn_sys_assert_is_file "c_config_path" || return 1
    fn_sys_log_info "Start to parse log from omp node: [${omp_ip}]."
    [[ "$(fn_sys_is_valid_ip "${omp_ip}")" == "false" ]] && fn_sys_log_error "The OMP IP address in the configuration file is invalid." && return 1

    fn_check_remote_host || return 1
    return 0
}

###############################################################################################
# 1 清理 所有远端空间/opt/HDSetupTool_tmp里面的检查脚本
# 2 创建 ${root_path}/log 用于存放取回来的日志。
# 3 从当前节点查看是否有已经有参数定制文件-将之分发到所有FI节点。
###############################################################################################
function fn_prepare()
{
    fn_sys_log_info "Prepare for pre-installation check."
    fn_clean_workspace_flag || return 1
    mkdir -p "${c_root_path}/log"
    [[ -f ${c_old_ficluster} ]] && rm -f ${c_ficluster_conf_path}; cp ${c_old_ficluster} ${c_ficluster_conf_path}
    return 0
}

###############################################################################################
function fn_do()
{
    fn_sys_log_info "Start the pre-installation check..."
    fn_cp_frame_remote || return 1
    fn_invoke_local_check_remote || return 1
    fn_check_result_remote || return 1
    fn_process_result || return 1
    fn_find_diff_rpm_ver || return 1

    fn_sys_log_info "Pre-installation check done."
    return 0
}

###############################################################################################
function fn_main()
{

    readonly c_start_time="$(date +%s)"
    fn_init || return 1
    fn_sys_log_enter
    fn_precheck "$@" || return 1
    fn_prepare || return 1
    fn_do || return 1
    fn_adjust_result

    local l_result="$?"
    local l_elapsed_seconds="$(fn_sys_get_elapsed_time "$c_start_time")"
    fn_sys_log_leave "Result [${l_result}]. It takes [${l_elapsed_seconds}] seconds"
    return "$l_result"
}

###############################################################################################
echo -n "Enter the password of the root user: "
IFS=$'\n'
read -sr remote_password
echo -e "\n"

fn_main "$@"
result="$?"
fn_sys_audit_log "${result}" "Init Check" "Execute remote_launch "
exit "${result}"
