#!/bin/bash -

###############################################################################################
# readonly SYS_CPU_ARM=AARCH64
# readonly SYS_CPU_X86_64=X86_64

###############################################################################################
function fn_sys_get_suse_version()
{
    local readonly c_version_file="/etc/SuSE-release"

    [ ! -f "$c_version_file" ] && echo "" && return 0

    local l_os_version="$(cat $c_version_file | grep '^VERSION[[:blank:]]*' | awk -F'=' '{ print $2 }' | sed 's#[[:blank:]]##g')"
    echo "$l_os_version"
    return 0
}

###############################################################################################
function fn_sys_is_suse_12()
{
    local l_suse_version="$(fn_sys_get_suse_version)"
    [ "$l_suse_version" == "12" ] && echo "true" && return 0
    echo "false"
    return 0
}

###############################################################################################
function fn_sys_is_rh2288x()
{
    lscpu |grep 'Gold 62' >/dev/null 2>&1
    [ "$?" -ne "0" ] && echo "false" && return 0

    echo "true"
    return 0
}

###############################################################################################
function fn_sys_get_os_is_euler()
{
    local readonly c_version_file="/etc/os-release"

    [ ! -f "$c_version_file" ] && echo "false" && return 0
    grep "EulerOS\|Kylin" "$c_version_file" &> /dev/null
    [ "$?" -eq 0 ] && echo "true" && return 0

    echo "false"
    return 0
}

###############################################################################################
function fn_sys_is_mount_point_existing()
{
    local l_mount_point="$1"

    [ -z "$l_mount_point" ] && echo "false" && return 1

    cat "/proc/mounts" | awk '{ print $2 }' | grep -Fwq "$l_mount_point" &> /dev/null
    [ "$?" -ne "0" ] && echo "false" && return 0

    echo "true"
    return 0
}

###############################################################################################
function fn_sys_is_physical_directory()
{
    local l_directory="$1"

    [ -z "$l_directory" ] && echo "false" && return 0
    [ -L "$l_directory" ] && echo "false" && return 0
    [ ! -d "$l_directory" ] && echo "false" && return 0

    echo "true"
    return 0
}

###############################################################################################
function fn_sys_is_cgroup_mount_point_existing()
{
    local l_mount_point="$1"

    [ "$(fn_sys_is_mount_point_existing "$l_mount_point")" == "false" ] && echo "false" && return 0

    cat /proc/mounts | awk '{ print $2";"$3 }' | grep "^${l_mount_point};cgroup$" &> /dev/null
    [ "$?" -eq "0" ] && echo "true" && return 0

    echo "false"
    return 0
}

###############################################################################################
function fn_sys_is_symbol_link()
{
    local l_file="$1"

    [ ! -e "$l_file" ] && echo "false" && return 0
    [ -L "$l_file" ] && echo "true" && return 0
    echo "false"
    return 0
}

###############################################################################################
function fn_sys_contains_symbol_link()
{
    local l_file="$1"

    local l_real_path="$(readlink -m "$l_file")"
    local l_formatted_file_path="$(echo "$l_file" | sed 's#/\+#/#g;s#/$##g')"
    [ -z "$l_formatted_file_path" ] && l_formatted_file_path="/"

    [ "$l_real_path" == "$l_formatted_file_path" ] && echo "false" && return 0
    echo "true"
    return 0
}

###############################################################################################
function fn_sys_is_hard_link()
{
    local l_file="$1"

    # hard link could only be applied to file
    [ ! -f "$l_file" ] && echo "false" && return 0

    local l_file_number=$(stat -c %h "$l_file")
    if [ "$l_file_number" -gt "1" ]; then
        echo "true"
    else
        echo "false"
    fi
    return 0
}

###############################################################################################
function fn_sys_is_link_file()
{
    local l_file="$1"

    [ "$(fn_sys_is_symbol_link "$l_file")" == "true" ] && echo "true" && return 0
    [ "$(fn_sys_is_hard_link "$l_file")" == "true" ] && echo "true" && return 0

    echo "false"
    return 0
}

###############################################################################################
function fn_sys_cancel_hard_link()
{
    local l_file="$1"

    if [ "$(fn_sys_is_hard_link "$l_file")" == "false" ]; then
        fn_sys_log_warn "File [${l_file}] is not hard link file"
        return 0
    fi

    cp -pf "$l_file" "${l_file}_bak.$$" && rm -rf "$l_file" && mv "${l_file}_bak.$$" "$l_file"
    [ "$?" -ne "0" ] && return 1

    return 0
}

###############################################################################################
function fn_sys_chmod_not_recursively()
{
    local l_file="$1"
    local l_permission="$2"

    if [ -L "$l_file" ]; then
        # for soft link, do not chmod
        fn_sys_log_warn "File [${l_file}] is soft link. Skip"
        return 0
    elif [ -d "$l_file" ]; then
        # for directory, chmod it self
        fn_sys_log_info "Set permission for directory [${l_file}] itself to [${l_permission}]"
        chmod "$l_permission" "$l_file"
        return 0
    elif [ -f "$l_file" ]; then
        if [ `fn_sys_is_hard_link "$l_file"` ]; then
            fn_sys_cancel_hard_link "$l_file"
            [ "$?" -ne "0" ] && fn_sys_log_error "Failed to cancel hard link for file [${l_file}]" && return 1
        fi
        fn_sys_log_info "Set permission for file [${l_file}] itself to [${l_permission}]"
        chmod "$l_permission" "$l_file"
        return 0
    else
        fn_sys_log_error "Invaild file [${l_file}]"
        return 1
    fi

    return 0
}

###############################################################################################
function fn_sys_chmod_recursively()
{
    local l_file="$1"
    local l_permission="$2"

    fn_sys_chmod_not_recursively "$l_file" "$l_permission"
    [ "$?" -ne "0" ] && fn_sys_log_error "Failed to set permission for file [${l_file}] to [${l_permission}]" && return 1

    [ ! -d "$l_file" ] && fn_sys_log_warn "File [${l_file}] is not type of directory. Do not set permission recursively" && return 0

    # for file, set permission except for hard link
    find "$l_file" -type f -links 1 -exec chmod "$l_permission" '{}' +
    # for directory, set permission except for soft link
    find "$l_file" -type d ! -type l -exec chmod "$l_permission" '{}' +

    return 0
}

###############################################################################################
function fn_sys_chown_not_recursively()
{
    local l_file="$1"
    local l_owner="$2"

    if [ -L "$l_file" ]; then
        # for soft link, do not chown
        fn_sys_log_warn "File [${l_file}] is soft link. Skip"
        return 0
    elif [ -d "$l_file" ]; then
        # for directory, chown itself
        fn_sys_log_info "Set owner for directory [${l_file}] itself to [${l_owner}]"
        chown "$l_owner" "$l_file"
        return 0
    elif [ -f "$l_file" ]; then
        if [ `fn_sys_is_hard_link "$l_file"` ]; then
            fn_sys_cancel_hard_link "$l_file"
            [ "$?" -ne "0" ] && fn_sys_log_error "Failed to cancel hard link for file [${l_file}]" && return 1
        fi
        fn_sys_log_info "Set permission for file [${l_file}] itself to [${l_owner}]"
        chown "$l_owner" "$l_file"
        return 0
    else
        fn_sys_log_error "Invaild file [${l_file}]"
        return 1
    fi

    return 0
}

###############################################################################################
function fn_sys_chown_recursively()
{
    local l_file="$1"
    local l_owner="$2"

    fn_sys_chown_not_recursively "$l_file" "$l_owner"
    [ "$?" -ne "0" ] && fn_sys_log_error "Failed to set owner for file [${l_file}] to [${l_owner}]" && return 1

    [ ! -d "$l_file" ] && fn_sys_log_warn "File [${l_file}] is not type of directory. Do not set owner recursively" && return 0

    # for file, set owner except for hard link
    find "$l_file" -type f -links 1 -exec chown "$l_owner" '{}' +
    # for directory, set owner except for soft link
    find "$l_file" -type d ! -type l -exec chown "$l_owner" '{}' +

    return 0
}

###############################################################################################
function fn_sys_get_memory_in_kb()
{
    [ ! -f "/proc/meminfo" ] && echo 0 && return 0
    cat "/proc/meminfo" | grep 'MemTotal' | awk '{print $2}'

    return 0
}

###############################################################################################
function fn_sys_get_number_of_cpus()
{
    lscpu | grep '^CPU(s)' | awk '{print $2}'

    return 0
}

###############################################################################################
function fn_sys_get_processor_architect()
{
    uname -p | tr [a-z] [A-Z]

    return 0
}

###############################################################################################
function fn_sys_is_local_ip()
{
    local l_ip="$1"

    # for euler
    ifconfig | grep "^[[:blank:]]*inet[[:blank:]]\+${l_ip}[[:blank:]]" &> /dev/null
    [ "$?" -eq "0" ] && echo "true" && return 0

    # for suse
    ifconfig | grep "^[[:blank:]]*inet[[:blank:]]\+addr:[[:blank:]]*${l_ip}[[:blank:]]" &> /dev/null
    [ "$?" -eq "0" ] && echo "true" && return 0

    echo "false"
    return 0
}

###############################################################################################
function fn_sys_forbid_ftp_shell()
{
    local l_user_name="$1"

    fn_sys_log_info "Forbid FTP shell for user [${l_user_name}] "
    local readonly c_ftp_config="/usr/lib64/ssh/sftp-server.sh /usr/lib64/ssh/sftp-server /usr/lib/ssh/sftp-server.sh /usr/lib/ssh/sftp-server"
    for each_file in ${c_ftp_config}; do
        if [ -f "${each_file}" ]; then
            fn_sys_execute_cmd usermod -s ${each_file} "$l_user_name"
        fi
    done

    return 0
}

###############################################################################################
function fn_sys_set_ftp_chroot()
{
    local l_user_name="$1"

    fn_sys_log_info "Add user [${l_user_name}] to FTP chroot file"

    local readonly c_ftp_chroot_file="/etc/vsftpd.chroot_list"
    [ ! -f "$c_ftp_chroot_file" ] && fn_sys_log_warn "FTP chroot file [${c_ftp_chroot_file}] is not existing. Skip" && return 0

    grep "^[[:blank:]]*${l_user_name}[[:blank:]]*$" "$c_ftp_chroot_file" &> /dev/null
    if [ "$?" -ne "0" ]; then
        echo "$l_user_name" >> "$c_ftp_chroot_file"
    fi

    return 0
}

###############################################################################################
function fn_sys_reset_ftp_chroot()
{
    local l_user_name="$1"

    fn_sys_log_info "Remove user [${l_user_name}] from FTP chroot file"

    local readonly c_ftp_chroot_file="/etc/vsftpd.chroot_list"
    [ ! -f "$c_ftp_chroot_file" ] && fn_sys_log_warn "FTP chroot file [${c_ftp_chroot_file}] is not existing. Skip" && return 0

    grep "^[[:blank:]]*${l_user_name}[[:blank:]]*$" "$c_ftp_chroot_file" &> /dev/null
    if [ "$?" -eq "0" ]; then
        sed -i "/^${l_user_name}$/d" "$c_ftp_chroot_file"
    fi

    return 0
}

###############################################################################################
function fn_sys_remove_allow_user()
{
    local l_user_name="$1"

    local readonly c_sshd_config="/etc/ssh/sshd_config"

    fn_sys_log_info "Remove allow user [${l_user_name}] from [${c_sshd_config}]"

    sed -i "$c_sshd_config" \
        -e "/^[[:blank:]]*AllowUsers[[:blank:]]\+${l_user_name}@.*$/d" \
        -e "/^[[:blank:]]*AllowUsers[[:blank:]]\+${l_user_name}[[:blank:]]*$/d" \

    return 0
}

###############################################################################################
function fn_sys_set_allow_user()
{
    local l_user_name="$1"

    local readonly c_sshd_config="/etc/ssh/sshd_config"

    fn_sys_log_info "Add allow user [${l_user_name}] to [${c_sshd_config}]"

    sed -i "$c_sshd_config" \
        -e "/^[[:blank:]]*AllowUsers[[:blank:]]\+${l_user_name}@.*$/d" \
        -e "/^[[:blank:]]*AllowUsers[[:blank:]]\+${l_user_name}[[:blank:]]*$/d" \
        -e "/^[[:blank:]]*Match[[:blank:]]\+All/a AllowUsers ${l_user_name}"

    return 0
}

###############################################################################################
function fn_sys_set_sftp_chroot()
{
    local l_user_name="$1"

    fn_sys_log_info "Add user [${l_user_name}] to sFTP chroot"

    local readonly c_sshd_config="/etc/ssh/sshd_config"
    fn_sys_assert_is_file "c_sshd_config" || return 1

    grep "^Match[[:blank:]]\+User[[:blank:]]\+${l_user_name}[[:blank:]]*$" "$c_sshd_config" &> /dev/null
    [ "$?" -eq "0" ] && fn_sys_log_warn "User [${l_user_name}] is already in sFTP chroot. Skip" && return 0

    echo "Match User ${l_user_name}" >> "$c_sshd_config"
    echo "    ChrootDirectory %h" >> "$c_sshd_config"
    echo "    ForceCommand internal-sftp" >> "$c_sshd_config"

    local l_suse_version=$(fn_sys_get_suse_version)
    if [ "$(fn_sys_get_os_is_euler)" == "false" ] && [ "$l_suse_version" -le 11 ]; then
        echo "Match" >> "$c_sshd_config"
    fi

    return 0
}

###############################################################################################
function fn_sys_reset_sftp_chroot()
{
    local l_user_name="$1"

    fn_sys_log_info "Remove user [${l_user_name}] from sFTP chroot"

    local readonly c_sshd_config="/etc/ssh/sshd_config"
    fn_sys_assert_is_file "c_sshd_config" || return 1

    local l_start_line="$(cat -n "$c_sshd_config" | grep "^[[:blank:]]*[0-9]\+[[:blank:]]*Match[[:blank:]]\+User[[:blank:]]\+${l_user_name}[[:blank:]]*$" | awk '{ print $1 }')"
    [ -z "$l_start_line" ] && fn_sys_log_warn "User [${l_user_name}] is not existing in sFTP chroot. Skip" && return 0

    local l_end_line=$((l_start_line+2))
    local l_suse_version=$(fn_sys_get_suse_version)
    if [ "$(fn_sys_get_os_is_euler)" == "false" ] && [ "$l_suse_version" -le 11 ]; then
        l_end_line=$((l_end_line+1))
    fi
    fn_sys_log_var "l_start_line"
    fn_sys_log_var "l_end_line"

    sed -i "${l_start_line},${l_end_line}d" "$c_sshd_config" || return 1

    return 0
}

###############################################################################################
function fn_sys_restart_sshd()
{
    local l_ssh_port=$(cat /etc/ssh/sshd_config | grep '^[[:blank:]]*Port[[:blank:]]' | sed 's#^[[:blank:]]##g' | awk '{ print $2 }' | sed 's#[[:blank:]]##g')
    [ -z "$l_ssh_port" ] && l_ssh_port=22

    fn_sys_execute_cmd service sshd restart
    sleep 1

    local l_max_retry_round=10
    for((l_round=1; l_round<=l_max_retry_round; l_round++)); do
        lsof -i:${l_ssh_port} | grep "^sshd[[:blank:]].*(LISTEN)[[:blank:]]*$" &> /dev/null
        if [ "$?" -eq "0" ]; then
            return 0
        else
            fn_sys_log_info "Waiting for SSHD service up"
            sleep 2
        fi
    done

    fn_sys_log_error "SSHD service is not UP"
    return 1
}

###############################################################################################
function fn_sys_restart_nscd()
{
    local l_retry_count="$1"
    local l_interval="$2"

    [ -z "$l_retry_count" ] && l_retry_count=3
    [ -z "$l_interval" ] && l_interval=2

    for round in $(seq 1 "$l_retry_count"); do
        fn_sys_execute_cmd service nscd restart
        if [ "$?" -eq "0" ]; then
            fn_sys_log_info "Round [${round}]: nscd restarted"
            return 0
        else
            fn_sys_log_warn "Round [${round}]: nscd not restarted"
            sleep "$l_interval"
        fi
    done

    return 1
}

###############################################################################################
function fn_sys_check_connectivity_to()
{
    local l_target_ip="$1"
    local l_retry_count="$2"
    local l_interval="$3"

    [ -z "$l_retry_count" ] && l_retry_count=10
    [ -z "$l_interval" ] && l_interval=2

    fn_sys_log_info "Check connectivity for target host [${l_target_ip}]"

    for round in $(seq 1 "$l_retry_count"); do
        ping -c 2 "$l_target_ip" &> /dev/null
        if [ "$?" -eq "0" ]; then
            fn_sys_log_info "Round [${round}]: Target host [${l_target_ip}] is reachable"
            return 0
        else
            fn_sys_log_warn "Round [${round}]: Target host [${l_target_ip}] is not reachable"
            sleep "$l_interval"
        fi
    done

    fn_sys_log_error "Check connectivity result: Target host [${l_target_ip}] is not reachable"

    return 1
}

###############################################################################################
function fn_sys_is_valid_ipv4_ip()
{
    local l_ip="$1"

    echo "$l_ip" | grep -E "^([0-9]{1,3}\.){3}[0-9]{1,3}$" &> /dev/null
    [ "$?" -ne "0" ] && echo "false" && return 0

    for token in $(echo "$l_ip" | awk -F '.' '{ for(i=1; i<=NF; i++) print $i }'); do
        [ "$token" -gt "255" ] && echo "false" && return 0
    done

    echo "true"
    return 0
}

###############################################################################################
function fn_sys_is_valid_ipv6_ip()
{
    local l_ip="$1"

    echo "$l_ip" | grep '\.' &> /dev/null
    if [ "$?" -eq "0" ]; then
        local l_ip_last_segment="$(echo "$l_ip" | awk -F':' '{ print $NF }')"
        [ "$(fn_sys_is_valid_ip "$l_ip_last_segment")" == "false" ] && echo "false" && return 0

        echo "$l_ip" | egrep -i "^(([A-F0-9]{1,4}:){5}[A-F0-9]{1,4}\
|([A-F0-9]{1,4}:){4}:[A-F0-9]{1,4}\
|([A-F0-9]{1,4}:){3}(:[A-F0-9]{1,4}){1,2}\
|([A-F0-9]{1,4}:){2}(:[A-F0-9]{1,4}){1,3}\
|[A-F0-9]{1,4}:(:[A-F0-9]{1,4}){1,4}\
|([A-F0-9]{1,4}:){1,5}\
|:(:[A-F0-9]{1,4}){1,5}\
|:):([0-9]{1,3}\.){3}[0-9]{1,3}$" &> /dev/null
        [ "$?" -ne "0" ] && echo "false" && return 0
    else
        echo "$l_ip" | egrep -i "^(([A-F0-9]{1,4}:){7}[A-F0-9]{1,4}\
|([A-F0-9]{1,4}:){6}:[A-F0-9]{1,4}\
|([A-F0-9]{1,4}:){5}(:[A-F0-9]{1,4}){1,2}\
|([A-F0-9]{1,4}:){4}(:[A-F0-9]{1,4}){1,3}\
|([A-F0-9]{1,4}:){3}(:[A-F0-9]{1,4}){1,4}\
|([A-F0-9]{1,4}:){2}(:[A-F0-9]{1,4}){1,5}\
|[A-F0-9]{1,4}:(:[A-F0-9]{1,4}){1,6}\
|([A-F0-9]{1,4}:){1,7}:\
|:(:[A-F0-9]{1,4}){1,7}\
|::)$" &> /dev/null
        [ "$?" -ne "0" ] && echo "false" && return 0
    fi

    echo "true"
    return 0
}

###############################################################################################
function fn_sys_is_valid_ip()
{
    local l_ip="$1"

    [ -z "$l_ip" ] && echo "false" && return 0

    echo "$l_ip" | grep ":" &> /dev/null
    if [ "$?" -ne "0" ]; then
        fn_sys_is_valid_ipv4_ip "$l_ip"
    else
        fn_sys_is_valid_ipv6_ip "$l_ip"
    fi

    return 0
}

###############################################################################################
function fn_sys_get_file_permission_in_octal()
{
    local l_file_path="$1"

    [ ! -e "$l_file_path" ] && echo "" && return 1
    stat -c "%a" "$l_file_path" 2> /dev/null

    return 0
}

###############################################################################################
function fn_sys_get_file_owner_and_group_name()
{
    local l_file_path="$1"

    [ ! -e "$l_file_path" ] && echo "" && return 1
    stat -c "%U:%G" "$l_file_path" 2> /dev/null

    return 0
}

###############################################################################################
function fn_sys_get_real_path()
{
    local l_file_path="$1"

    [ ! -e "$l_file_path" ] && echo "" && return 1

    if [ -L "$l_file_path" ]; then
        readlink "$l_file_path"
    else
        echo "$l_file_path"
    fi

    return 0
}

###############################################################################################
function fn_sys_umount_directory()
{
    local l_mount_point="$1"

    fn_sys_assert_current_user_matched "root" || return 1

    if [ "$(fn_sys_is_mount_point_existing "$l_mount_point")" == "false" ]; then
        fn_sys_log_warn "Mount point path [${l_mount_point}] is not existing"
        return 0
    fi

    fn_sys_execute_cmd fuser -kmv "$l_mount_point"
    fn_sys_execute_cmd umount "$l_mount_point" || return 1

    return 0
}

###############################################################################################
function fn_sys_umount_directory_with_retry()
{
    local l_mount_point="$1"
    local l_max_retry_round="$2"
    local l_retry_interval="$3"

    [ -z "$l_max_retry_round" ] && l_max_retry_round="5"
    [ -z "$l_retry_interval" ] && l_retry_interval="5"

    fn_sys_assert_current_user_matched "root" || return 1
    fn_sys_assert_is_directory "l_mount_point" || return 1
    fn_sys_assert_is_number "l_max_retry_round" || return 1
    fn_sys_assert_is_number "l_retry_interval" || return 1

    local l_failed="true"
    for((i=1; i<=l_max_retry_round; i++)); do
        fn_sys_log_info "Round [${i}]: [${l_mount_point}] ... Trying"
        fn_sys_umount_directory "$l_mount_point"
        if [ "$?" -ne "0" ]; then
            fn_sys_log_warn "Round [${i}]: Umount [${l_mount_point}] ... Failed"
            sleep "$l_retry_interval"
        else
            l_failed="false"
            fn_sys_log_info "Round [${i}]: Umount [${l_mount_point}] ... Done"
            break
        fi
    done
    [ "$l_failed" == "true" ] && fn_sys_log_error "Failed to umount [${l_mount_point}]" && return 1

    return 0
}

###############################################################################################
function fn_sys_is_valid_host_name()
{
    local l_host_name="$1"

    [ -z "$l_host_name" ] && echo "false" && return 0
    echo "$l_host_name" | grep -E '^[a-zA-Z][a-zA-Z0-9\-]{1,63}$' &> /dev/null
    [ "$?" -eq "0" ] && echo "true" && return 0

    echo "false"
    return 0
}

###############################################################################################
function fn_sys_is_valid_nic_name()
{
    local l_eth_name="$1"

    [ -z "$l_eth_name" ] && echo "false" && return 1

    echo "$l_eth_name" | grep -E '^[A-Za-z]+[0-9]+$' &> /dev/null
    [ "$?" -eq "0" ] && echo "true" && return 0

    echo "false"
    return 1
}

###############################################################################################
function fn_sys_is_real_os_nic()
{
    local l_eth_name="$1"

    [ "$(fn_sys_is_valid_nic_name "$l_eth_name")" != "true" ] && echo "false" && return 1

    if [ "$(fn_sys_get_os_is_euler)" == "true" ] ; then
        ifconfig | grep "^${l_eth_name}:" &> /dev/null
        [ "$?" -eq "0" ] && echo "true" && return 0
    else
		ifconfig | grep "^${l_eth_name}[[:blank:]]" &> /dev/null
        [ "$?" -eq "0" ] && echo "true" && return 0
    fi

    echo "false"
    return 1
}

###############################################################################################
function fn_sys_get_os_uid()
{
    local l_user_name="$1"

    [ -z "$l_user_name" ] && return 1

    id -u "$l_user_name" 2> /dev/null

    return 0
}

###############################################################################################
function fn_sys_get_os_gid()
{
    local l_group_name="$1"

    [ -z "$l_group_name" ] && return 1

    cat /etc/group | grep "^${l_group_name}:" | awk -F':' '{ print $3 }' 2> /dev/null

    return 0
}


function fn_ping_ip_ok()
{
    local ipaddr="$1"
    [ -z "$ipaddr" ] && echo "false" && return 0
    echo "$ipaddr" | grep ":" &> /dev/null

    if [ "$?" -ne "0" ]; then
        ping -c 1 "$ipaddr" -i 0.2 -W 1 > /dev/null 2>&1
        [[ $? -ne 0 ]] && echo "false" && return 0
    else
        ping6 -c 1 "$ipaddr" -i 0.2 -W 1 > /dev/null 2>&1
        [[ $? -ne 0 ]] && echo "false" && return 0
    fi
    echo "true" && return 0
}