#!/bin/bash

###################################################
# 功能: 在远端服务器运行命令 或 拷贝文件
###################################################

###################################################
#
# 在远端服务器运行命令
#
###################################################
function exec_cmd_remote()
{
    local host="$1"        #远端主机名
    local g_user_name="$2" #登录用户
    local timeout="$3"     #命令超时时间
    local command="$4"     #运行的命令
    local log="$5"         #是否记录日志(可以为空,默认记录日志)
    local ret=0            #返回值

    IFS=$'\n'
    read -sr password
    [[ -z "$log" ]] && log=1
    [[ "$log" -ne 0 ]] && log=1

    #入参判断
    if [ -z "$host" ]; then
        fn_sys_log_error "[${FUNCNAME}:${LINENO}] host is null"
        return 1
    fi

    if [ -z "$timeout" ]; then
        fn_sys_log_error "[${FUNCNAME}:${LINENO}] timeout is null"
        return 1
    fi

    if [ -z "$command" ]; then
        fn_sys_log_error "[${FUNCNAME}:${LINENO}] command is null"
        return 1
    fi

    echo "$host" | grep ":"
    [[ "$?" -eq "0" ]] && g_ip_model="IPv6"
    [[ "$?" -ne "0" ]] && g_ip_model="IPv4"
    #当log入参为0时，不输出任何信息；
    #当log入参为1时，把输出信息定向到日志文件；
    #当log入参为其他时，回显输出信息。
    fn_sys_log_info "ssh -o StrictHostKeyChecking=no "${g_user_name}"@"${host}" \""${command}"\""
    l_process_timeout=$((timeout+5))
    if [[ $log -eq 0 ]]; then
        timeout ${l_process_timeout}s expect >/dev/null 2>&1 <<END
            set timeout ${timeout}
            spawn ssh -o StrictHostKeyChecking=no ${g_user_name}@${host} "${command}"
            expect {
                "*assword*" {
                    send "${password}\n"
                    expect {
                      "*assword*" {exit 128}
                    }
                }
                timeout {exit 127}
                expect eof
            }
            catch wait result
            exit [lindex \$result 3]
END
        ret=$?
        fn_sys_log_info "Timeout: [${timeout}]; return code: [${ret}]"
        [[ $ret -ne 0 ]] && return $ret
    elif [[ $log -eq 1 ]]; then
        timeout ${l_process_timeout}s expect >> ${c_log_file} <<END
            set timeout ${timeout}
            spawn ssh -o StrictHostKeyChecking=no ${g_user_name}@${host} "${command}"
            expect {
                "*assword*" {
                    send "${password}\n"
                    expect {
                      "*assword*" {exit 128}
                    }
                }
                timeout {exit 127}
                expect eof
            }
            catch wait result
            exit [lindex \$result 3]
END
        ret=$?
        fn_sys_log_info "Timeout: [${timeout}]; return code: [${ret}]"
        [[ $ret -ne 0 ]] && fn_sys_log_error "Failed to run the [${command}] on the remote node [${host}], ret code: [${ret}]." && return $ret
    else
        timeout ${l_process_timeout}s expect <<END
            set timeout ${timeout}
            spawn ssh -o StrictHostKeyChecking=no ${g_user_name}@${host} "${command}"
            expect {
                "*assword*" {
                    send "${password}\n"
                    expect {
                      "*assword*" {exit 128}
                    }
                }
                timeout {exit 127}
                expect eof
            }
            catch wait result
            exit [lindex \$result 3]
END
        ret=$?
        fn_sys_log_info "Timeout: [${timeout}]; return code: [${ret}]"
        [[ $ret -ne 0 ]] && fn_sys_log_error "Failed to run the [${command}] on the remote node [${host}], ret code: [${ret}]." && return $ret
    fi
    fn_sys_log_info "Success to run the [${command}] on the remote node [${host}]."
    return $ret
}

###################################################
#
# 把文件拷贝至远端服务器
#
###################################################
function cp_file_to_remote()
{
    local host="$1"       #远端主机名
    local g_user_name="$2"    #用户名
    local timeout="$3"    #命令超时时间
    local src="$4"        #源文件
    local dst="$5"        #目标文件
    local log="$6"        #是否记录日志(可以为空,默认记录日志)
    local ret=0           #返回值

    IFS=$'\n'
    read -sr password

    [[ -z "$log" ]] && log=1
    [[ "$log" -ne 0 ]] && log=1

    #入参判断
    if [ -z "$host" ]; then
        fn_sys_log_error "[${FUNCNAME}:${LINENO}] host is null"
        return 1
    fi

    [[ "$(fn_sys_is_valid_ipv6_ip "${host}")" == "true" ]] && host="\[${host}\]"

    if [ -z "$timeout" ]; then
        fn_sys_log_error "[${FUNCNAME}:${LINENO}] timeout is null"
        return 1
    fi

    if [ -z "$src" ]; then
        fn_sys_log_error "[${FUNCNAME}:${LINENO}] src is null"
        return 1
    fi

    if [ -z "$dst" ]; then
        fn_sys_log_error "[${FUNCNAME}:${LINENO}] dst is null"
        return 1
    fi

    #log为0时,不记录日志
    if [ -z "$log" ]; then
        log=1
    fi

    if [ $log -ne 0 ]; then 
        expect -c"
            set timeout ${timeout}
            spawn scp -r -o StrictHostKeyChecking=no "${src}" "${g_user_name}"@"${host}":"${dst}"
            expect {
              \"*assword*\" {
                send \"${password}\n\"
                expect {
                  \"*assword*\" {exit 2}
                  eof
                }
              }
              eof
            }
            catch wait result;
            exit [lindex \$result 3]" >> "$g_log_file"
        ret=$?
    else
        expect -c"
            set timeout ${timeout}
            spawn scp -r -o StrictHostKeyChecking=no "${src}" "${g_user_name}"@"${host}":"${dst}"
            expect {
              \"*assword*\" {
                send \"${password}\n\"
                expect {
                  \"*assword*\" {exit 2}
                  eof
                }
              }
              eof
            }
            catch wait result;
            exit [lindex \$result 3]" > /dev/null 2>&1
        ret=$?
    fi

    [[ ${ret} -ne 0 ]] && fn_sys_log_error "Failed to copy [${src}] to the remote server [${host}], ret code: [${ret}]" && return ${ret}
    fn_sys_log_info "Succeeded in copying [${src}] to the remote server [${host}]."
    return ${ret}
}

###################################################
#
# 把远端服务器文件拷贝至本地
#
###################################################
function cp_file_from_remote()
{
    local host="$1"       #远端主机名
    local g_user_name="$2"    #用户名
    local timeout="$3"    #命令超时时间
    local src="$4"        #源文件
    local dst="$5"        #目标文件
    local log="$6"        #是否记录日志(可以为空,默认记录日志)
    local ret=0           #返回值

    IFS=$'\n'
    read -sr password

    [[ -z "$log" ]] && log=1
    [[ "$log" -ne 0 ]] && log=1

    #入参判断
    if [ -z "$host" ]; then
        fn_sys_log_error "[${FUNCNAME}:${LINENO}] host is null"
        return 1
    fi

    [[ "$(fn_sys_is_valid_ipv6_ip "${host}")" == "true" ]] && host="\[${host}\]"

    if [ -z "$timeout" ]; then
        fn_sys_log_error "[${FUNCNAME}:${LINENO}] timeout is null"
        return 1
    fi

    if [ -z "$src" ]; then
        fn_sys_log_error "[${FUNCNAME}:${LINENO}] src is null"
        return 1
    fi

    if [ -z "$dst" ]; then
        fn_sys_log_error "[${FUNCNAME}:${LINENO}] dst is null"
        return 1
    fi

    #log为0时,不记录日志
    if [ -z "$log" ]; then
        log=1
    fi

    if [ $log -ne 0 ]; then
        expect -c"
            set timeout ${timeout}
            spawn scp -r -o StrictHostKeyChecking=no "${g_user_name}"@"${host}":"${src}" "${dst}"
            expect {
              \"*assword*\" {
                send \"${password}\n\"
                expect {
                  \"*assword*\" {exit 2}
                  eof
                }
              }
              eof
            }
            catch wait result;
            exit [lindex \$result 3]" >> "$g_log_file"
        ret=$?
    else
        expect -c"
            set timeout ${timeout}
            spawn scp -r -o StrictHostKeyChecking=no "${g_user_name}"@"${host}":"${src}" "${dst}"
            expect {
              \"*assword*\" {
                send \"${password}\n\"
                expect {
                  \"*assword*\" {exit 2}
                  eof
                }
              }
              eof
            }
            catch wait result;
            exit [lindex \$result 3]" > /dev/null 2>&1
        ret=$?
    fi

    [[ ${ret} -ne 0 ]] && fn_sys_log_error "Failed to copy [${src}] from the remote server [${host}], ret code: [${ret}]" && return ${ret}
    fn_sys_log_info "Succeeded in copying [${src}] from the remote server [${host}]."
    return ${ret}
}
