#!/usr/bin/python
# -*- coding: utf-8 -*-
import re
import time
import traceback

from getDBConnection import get_zenith_session
from common_tasks.base_task import BaseTask

ONE_PACKAGE = 500


class UpgradeCommonDevInfo(BaseTask):
    """这个升级依赖于DevTypeRegInfo的升级，必须在DevTypeRegInfo升级之后再执行"""

    def __init__(self, product_name="NCE"):
        super(UpgradeCommonDevInfo, self).__init__()
        self.set_product_name(product_name)
        self.info("UpgradeCommonDevInfo init product_name is %s" % product_name)
        self.src_db_session = get_zenith_session('eamdb', 'eamdb', product_name)
        if self.src_db_session is None:
            self.error("eamdb is None")
            return
        self.src_db_session.autocommit(True)
        self.src_db_cursor = self.src_db_session.cursor()

        self.dst_db_session = get_zenith_session('neresdb', 'neresdb', product_name)
        if self.dst_db_session is None:
            self.error("neresdb_sess is None")
            return
        self.dst_db_session.autocommit(True)
        self.dst_db_cursor = self.dst_db_session.cursor()

        self.src_table = "tbl_ne"
        self.src_table_cols = ("resId", "dn", "label", "productName", "typeId", "ipaddr", "memo", "parentResId")
        self.src_table_cols_index = {y: x for x, y in enumerate(self.src_table_cols)}
        self.dst_table = "CommonDevInfo"
        self.dst_table_cols = ("neResId", "tenantId", "neId", "name", "typeName", "typeId", "uniqueAddress",
                               "remark", "alias", "extendInfo", "macAddress", "createAt", "updateAt")
        self.idmapping_dic = {}
        self.neid_uniqueid_dic = {}
        self.neid_mac_dic = {}
        self.create_time_dic = {}

    def get_resid_from_idmapping(self, paras):
        topodb_sess = get_zenith_session('topodb', 'topodb', self.product_name)
        if topodb_sess is None:
            self.error("get topodb session fail")
            return
        topodb_sess.autocommit(True)
        topodb_sess_cursor = topodb_sess.cursor()
        # 只获取物理视图（topoid=101）中别名alias字段非空的数据，这样可以大大减少无效数据
        query_alias_stmt = "select resId, alias from tbl_node where topoid=101 and alias is not null"
        self.debug("exec sql: %s" % query_alias_stmt)
        topodb_sess_cursor.execute(query_alias_stmt)
        self.idmapping_dic = dict(topodb_sess_cursor.fetchall())
        self.debug("idmapping_dic: %s" % self.idmapping_dic)

        topodb_sess_cursor.close()
        topodb_sess.close()

    def get_id_uniqueid_mapping(self):
        neresdb_sess = get_zenith_session('neresdb', 'neresdb', self.product_name)
        if neresdb_sess is None:
            self.error("get neresdb session fail")
            return
        neresdb_sess.autocommit(True)
        neresdb_cursor = neresdb_sess.cursor()
        select_stmt = "select `neId`, `uniqueId` from `NeIdAlloc`"
        self.debug("exec sql: %s" % select_stmt)
        neresdb_cursor.execute(select_stmt)
        result = list(neresdb_cursor.fetchall())
        # 这里强制转换成字符串，避免数据库字段类型的影响。
        self.neid_uniqueid_dic = dict([(str(x), str(y)) for x, y in result])
        self.debug("neid_uniqueid_dic: %s" % self.neid_uniqueid_dic)
        neresdb_cursor.close()
        neresdb_sess.close()

    def get_neid_mac_mapping(self):
        eamdb_sess = get_zenith_session('eamdb', 'eamdb', self.product_name)
        if eamdb_sess is None:
            self.error("get eamdb session fail")
            return
        eamdb_sess.autocommit(True)
        eamdb_cursor = eamdb_sess.cursor()
        select_stmt = "select DEVID, VALUE from tbl_ResAttr where MAINTYPE=1 and ATTRID=20"
        self.debug("exec sql: %s" % select_stmt)
        eamdb_cursor.execute(select_stmt)
        result = list(eamdb_cursor.fetchall())
        # 这里强制转换成字符串，避免数据库字段类型的影响。
        self.neid_mac_dic = dict([(str(x), str(y)) for x, y in result])
        self.debug("neid_mac_dic: %s" % self.neid_mac_dic)
        eamdb_cursor.close()
        eamdb_sess.close()

    def get_create_time_mapping(self):
        db_sess = get_zenith_session('cmdbcoresvrdb', 'cmdbcoresvrdb', self.product_name)
        if db_sess is None:
            self.error("get cmdbcoresvrdb session fail")
            return
        db_sess.autocommit(True)
        db_cursor = db_sess.cursor()
        select_stmt = "select `nativeId`, `createTime` from i_fixednetworkelement"
        # SQL> select `nativeId`, `createTime` from i_fixednetworkelement;
        #
        # nativeId                                                         createTime
        # ---------------------------------------------------------------- --------------------
        # NE=167772161                                                     1644801354000
        #
        # 1 rows fetched.
        self.debug("exec sql: %s" % select_stmt)
        db_cursor.execute(select_stmt)
        result = list(db_cursor.fetchall())
        # 这里强制转换成字符串，避免数据库字段类型的影响。
        self.create_time_dic = dict([(str(x), str(y)) for x, y in result])
        self.debug("create_time_dic: %s" % self.create_time_dic)
        db_cursor.close()
        db_sess.close()

    def convert_data(self, paras):
        def covert_ip_addr(c_uniqueId):
            if re.compile(r"([\d]{1,3}\.[\d]{1,3}\.[\d]{1,3}\.[\d]{1,3})").match(c_uniqueId):
                # 能匹配上，就是IPv4
                return ("ipv4://" + c_uniqueId).replace("_", "/")
            else:
                # 不是IPv4就是IPv6,因为IPv6的格式比较灵活，所以只要不是IPv4的就当做IPv6处理
                return ("ipv6://" + c_uniqueId).replace("_", "/")

        resId = paras[self.src_table_cols_index.get("resId")]
        tenantId = "default-organization-id"
        neId = paras[self.src_table_cols_index.get("dn")].lstrip("NE=")
        name = paras[self.src_table_cols_index.get("label")]
        typeName = paras[self.src_table_cols_index.get("productName")]
        typeId = paras[self.src_table_cols_index.get("typeId")]
        uniqueAddress = self.neid_uniqueid_dic.get(str(neId))
        self.debug("uniqueAddress: %s" % uniqueAddress)
        if not uniqueAddress:
            ipaddr = paras[self.src_table_cols_index.get("ipaddr")]
            if ipaddr is None:
                # 特殊情况下，ipaddr可能是None值
                uniqueAddress = None
            else:
                # 这里确保ipaddr一定是字符串类型
                ipaddr = str(ipaddr)
                if re.compile(r"([\d]+\-[\d]+)").match(ipaddr):
                    # 如果是xx-yy的格式，则是网元ID
                    # 物理ID：9-9528
                    # 转化成：phyid://xx-yy
                    # Done: 方案刷新，直接拼接xx-yy格式的网元ID
                    uniqueAddress = "phyid://%s" % ipaddr
                else:
                    # IP的地址格式跟FAN的处理过程一样
                    uniqueAddress = covert_ip_addr(ipaddr)
                    # 对于dummy设备，在uniqueAddress后加_dummy_resId
                    if typeId == "51707905":
                        uniqueAddress = "%s_dummy_%s" % (uniqueAddress, resId)

        remark = paras[self.src_table_cols_index.get("memo")]
        alias = self.idmapping_dic.get(str(resId))
        extendInfo = None
        macAddress = self.neid_mac_dic.get(neId)
        createTime = str(self.create_time_dic.get(paras[self.src_table_cols_index.get("dn")], 0))
        if createTime.isdigit():
            # 这里还涉及到createTime与createAt之间的单位转换，createAt的单位是ms，createTime的单位待确定
            createAt = createTime
        else:
            createAt = str(0)
        updateAt = str(int(time.time() * 1000))

        return tuple(None if x is None else str(x) for x in (
            resId, tenantId, neId, name, typeName, typeId, uniqueAddress, remark, alias, extendInfo, macAddress,
            createAt, updateAt))

    def to_UpgradePara(self, datas):
        self.get_resid_from_idmapping(datas)
        self.get_id_uniqueid_mapping()
        self.get_neid_mac_mapping()
        try:
            self.get_create_time_mapping()
        except TypeError as te:
            self.warning("get_create_time_mapping exception: %s" % te)
        except BaseException as be:
            self.warning("get_create_time_mapping exception: %s" % be)

        col_names = "`" + ("`, `".join(self.dst_table_cols)) + "`"
        val_ids = ":" + (",:".join((str(x + 1) for x in range(len(self.dst_table_cols)))))
        insert_stmt = "INSERT INTO `%s` (%s) VALUES(%s)" % (self.dst_table, col_names, val_ids)
        self.debug("insert sql stmt: %s" % insert_stmt)
        list_datas = []
        for data in datas:
            self.debug("original data is: %s, length is:%s" % (data, len(data)))
            data = self.convert_data(data)
            self.debug("coverted date is: %s, length is:%s" % (data, len(data)))
            if len(data) == len(self.dst_table_cols):
                tuple_data = tuple(data)
                list_datas.append(tuple_data)
            if len(list_datas) == ONE_PACKAGE:
                self.exec_sql(insert_stmt, list_datas)
                list_datas = []

        if len(list_datas) != 0:
            self.debug("data is:%s" % list_datas)
            self.exec_sql(insert_stmt, list_datas)

    def close_session(self):
        self.dst_db_cursor.close()
        self.dst_db_session.close()
        self.src_db_cursor.close()
        self.src_db_session.close()

    def get_type_id_from_resource_futrue(self):
        """
        支持传送 三方网元的升级
        1、先从平台 RESOURCETYPE_FEATURE 表中查询 DaemonProcID=48 的设备类型ID集合
        2、根据设备类型集合匹配原则，从TBL_NE表升级轻量级网元管理器的网元表；
        :return:
        """
        featuredb_sess = get_zenith_session('invmetadatadb', 'invmetadatadb', self.product_name)
        if featuredb_sess is None:
            self.error("UpgradeCommonDevInfo get invmetadatadb session fail")
            return None
        featuredb_sess.autocommit(True)
        featuredb_sess_cursor = featuredb_sess.cursor()
        # 1、先从平台RESOURCETYPE_FEATURE 表中查询 DaemonProcID=48 的设备类型ID集合
        # 2、根据设备类型集合匹配原则，从TBL_NE表升级轻量级网元管理器的网元表；
        query_type_id_stmt = "SELECT TYPEID FROM RESOURCETYPE_FEATURE " \
                             "WHERE FEATURENAME='DaemonProcID' AND FEATUREVALUE='48'"
        self.debug("exec sql: %s" % query_type_id_stmt)
        featuredb_sess_cursor.execute(query_type_id_stmt)
        ret = {x[0] for x in featuredb_sess_cursor.fetchall()}
        self.debug("typeids: %s" % ret)

        featuredb_sess_cursor.close()
        featuredb_sess.close()
        return ret

    def do(self):
        try:
            self.info('UpgradeCommonDevInfo::do start!')
            # 查询neresdb数据库DevTypeRegInfo表，匹配typeId字段，能够匹配成功的执行升级；
            # 匹配不上的代表非纳管范围内设备类型，不参与升级
            # 优先升级非Dummy类型的设备，再升级Dummy设备
            dummy_type_id = "51707905"

            # 支持NCE-T的三方网元升级
            try:
                third_party_neids = self.get_type_id_from_resource_futrue()
            except Exception as e:
                self.error('UpgradeCommonDevInfo get third_ne type id error')
                third_party_neids = {"16777218", "16908289", "16842753"}
            third_party_neids = {"16777218", "16908289", "16842753"} if not third_party_neids else third_party_neids
            special_type_id = "16777217"

            select_stmt_type_ids = "select `typeId` from `DevTypeRegInfo` where `domain`='NCE-IP'"
            self.dst_db_cursor.execute(select_stmt_type_ids)
            self.debug("execute sql: %s" % select_stmt_type_ids)
            typeIds = list(self.dst_db_cursor.fetchall())
            self.debug("typeIds: %s" % typeIds)
            # 如果IP注册过网元类型，说明有IP网元管理器，则不处理此类型的设备，如果没有IP网元管理器，则升级
            if not typeIds:
                self.debug("typeIds is None, need to upgrade typeId 16777217 in tbl_ne")
                third_party_neids.add(special_type_id)
            else:
                third_party_neids.discard(special_type_id)

            # 先查询没有dummy网元的数据
            select_stmt = "select %s from %s where typeId in (%s)" \
                          % (",".join(self.src_table_cols), self.src_table, ",".join(third_party_neids))
            datas = self.exec_query_sql(self.src_db_cursor, select_stmt)

            # 再查询dummy网元的数据
            select_stmt = "select %s from %s where typeId = '%s'" \
                          % (",".join(self.src_table_cols), self.src_table, dummy_type_id)
            # 两次查询的数据拼接到一起，dummy网元的数据排在后面
            datas.extend(self.exec_query_sql(self.src_db_cursor, select_stmt))

            self.info('get para data success count: %d' % len(datas))

            self.to_UpgradePara(datas)
            self.info('to_UpgradePara done')
            self.close_session()
        except Exception as e:
            self.close_session()
            self.error('UpgradeCommonDevInfo got exception')
            self.error(traceback.format_exc())
            return -1
        self.info('UpgradeCommonDevInfo::do done')
        return 0


if __name__ == '__main__':
    tool = UpgradeCommonDevInfo()
    print('[INFO] UpgradeCommonDevInfo start>>>>>>>>>>>>>>>>>>>>>>>>>>>>>>>>>>>>>>>>>>>>>>>>>>')
    tool.do()
    print('[INFO] UpgradeCommonDevInfo finished<<<<<<<<<<<<<<<<<<<<<<<<<<<<<<<<<<<<<<<<<<<<<<<')
