#!/usr/bin/python
# -*- coding: utf-8 -*-
import time
import traceback
import uuid

from getDBConnection import get_zenith_session
import common_tasks.const_sql as const_sql
from common_tasks.base_task import BaseTask

ONE_PACKAGE = 500


class UpgradeQxDcnForRouter(BaseTask):
    def __init__(self, product_name="NCE"):
        super(UpgradeQxDcnForRouter, self).__init__()
        self.set_product_name(product_name)
        self.info("UpgradeQxDcnForRouter init product_name is %s" % product_name)
        self.src_db_session = get_zenith_session('IPBaseDB', 'IPBaseDB', product_name)
        if self.src_db_session is None:
            self.error("IPBaseDB is None")
            return
        self.src_db_session.autocommit(True)
        self.src_db_cursor = self.src_db_session.cursor()

        self.dst_db_session = get_zenith_session('neresdb', 'neresdb', product_name)
        if self.dst_db_session is None:
            self.error("neresdb_sess is None")
            return
        self.dst_db_session.autocommit(True)
        self.dst_db_cursor = self.dst_db_session.cursor()

        self.src_table = "tbl_DCNGNeInfo"
        # 在查询源数据的方法中，确保返回的字段与这里定义的一致。
        self.src_table_cols = ("iDevID", "iNeID", "iGNeType", "iNeIPType", "strNeIP", "strIPMask", "iPort",
                               "iConnectMode", "iMainGNeID", "iBackUpOneGneId", "iBackUpTwoGneId",
                               "iBackUpThreeGneId", "iNotifyMode", "strAclName", "strAclNum", "strSSLDir")
        self.src_table_cols_index = {y:x for x, y in enumerate(self.src_table_cols)}
        self.dst_table = "QxDcn"
        self.dst_table_cols = ("tenantId", "neResId", "neId", "gatewayType", "gatewayProtocol", "masterGateway",
                               "ipAddr", "nsapAddr", "connectionType", "port", "sslCertificateId", "phyNeId",
                               "commuProType", "backup1Gateway", "backup2Gateway", "backup3Gateway",
                               "createAt", "updateAt")
        self.idmapping_dic = {}
        self.logicid_to_addr = {}

    def get_resid_from_idmapping(self, paras):
        self.debug("get_resid_from_idmapping::start")
        idmappingdb_sess = get_zenith_session('idmappingdb', 'idmappingdb', self.product_name)
        if idmappingdb_sess is None:
            self.error("get idmappingdb_sess session fail")
            return
        idmappingdb_sess.autocommit(True)
        id_mapping_cursor = idmappingdb_sess.cursor()

        tmp_table_name = "tmp_neid_%s" % self.dst_table
        id_mapping_cursor.execute(const_sql.DROP_TEMP_TABLE % tmp_table_name)
        id_mapping_cursor.execute(const_sql.CREATE_TEMP_TABLE % tmp_table_name)
        insert_stmt = "insert into tmp_%s values(:1)" % tmp_table_name

        nativeIds = []
        for para in paras:
            data = []
            nativeId = "NE=%s" % para[self.src_table_cols_index.get("iDevID")]
            data.append(nativeId)
            tuple_data = tuple(data)
            nativeIds.append(tuple_data)
            if len(nativeIds) == ONE_PACKAGE:
                id_mapping_cursor.executemany(insert_stmt, nativeIds)
                self.debug("one package:%s" % nativeIds)
                nativeIds = []

        if len(nativeIds) != 0:
            id_mapping_cursor.executemany(insert_stmt, nativeIds)
            self.debug("last package datas: %s" % nativeIds)

        query_stmt = const_sql.INNER_JOIN_TEMP_TABLE % tmp_table_name
        self.debug("query sql: %s" % query_stmt)
        id_mapping_cursor.execute(query_stmt)
        result = id_mapping_cursor.fetchall()

        for r in result:
            l = list(r)
            self.idmapping_dic[l[1].lstrip("NE=")] = l[0]
            self.debug("result: %s: %s" % (l[1], l[0]))

        id_mapping_cursor.execute(const_sql.DROP_TEMP_TABLE % tmp_table_name)
        id_mapping_cursor.close()
        idmappingdb_sess.close()

        # 根据网关的逻辑ID获取uniqueAddress
        # 每ONE_PACKAGE个逻辑ID执行一次，避免一次执行太多的查询数据
        select_stmt_logic_to_addr = "select `neId`, `uniqueAddress` from `NeBasicInfo` where `neId` in (%s)"
        for x in range(0, len(paras), ONE_PACKAGE):
            temp_paras = paras[x: x + ONE_PACKAGE]
            sql = select_stmt_logic_to_addr % ",".join([str(x[self.src_table_cols_index["iDevID"]]) for x in temp_paras])
            self.debug("execute sql: %s" % sql)
            self.dst_db_cursor.execute(sql)
            self.logicid_to_addr.update(dict([(str(neId), str(uniqueAddress))
                                              for neId, uniqueAddress in self.dst_db_cursor.fetchall()]))

    def convert_data(self, paras):
        tenantId = "default-organization-id"
        neResId = self.idmapping_dic.get(str(paras[self.src_table_cols_index.get("iDevID")]))
        neId = paras[self.src_table_cols_index.get("iDevID")]
        # JAVA实例中都是网关
        gatewayType = "Gateway"
        gatewayProtocol = "IP"

        masterGateway = paras[self.src_table_cols_index.get("iMainGNeID")]
        # Done： 拿设备逻辑ID到NeBasicInfo查询到UniqueAddress，转换成IP地址
        ipAddr = None
        uniqueAddress = self.logicid_to_addr.get(str(paras[self.src_table_cols_index.get("iDevID")]))
        if uniqueAddress is None:
            ipAddr = None
        elif uniqueAddress.startswith("ipv"):
            ipAddr = uniqueAddress.split("/")[-1]

        nsapAddr = None

        connectionTypeMap = {
            "1": "Common",
            "2": "SSL"
        }
        # 非网关赋值None；如果是网关，如果不是5432，就给默认值Common
        connectionType = connectionTypeMap.get(str(paras[self.src_table_cols_index.get("iConnectMode")]), "Common")
        port = paras[self.src_table_cols_index.get("iPort")]
        sslCertificateId = paras[self.src_table_cols_index.get("strSSLDir")]
        phyNeId = paras[self.src_table_cols_index.get("iNeID")]
        commuProType = -1
        # Done: qxDcn_C++的backup1Gateway是数字格式的物理ID，QxDcn的cBackup1GneId是设备逻辑ID，
        # Done: 需要根据物理ID查询出逻辑ID，然后赋值
        backup1Gateway = paras[self.src_table_cols_index.get("iBackUpOneGneId")]
        backup2Gateway = paras[self.src_table_cols_index.get("iBackUpTwoGneId")]
        backup3Gateway = paras[self.src_table_cols_index.get("iBackUpThreeGneId")]

        createAt = int(time.time() * 1000)
        updateAt = int(time.time() * 1000)

        return tuple(None if x is None else str(x) for x in (
            tenantId, neResId, neId, gatewayType, gatewayProtocol, masterGateway, ipAddr, nsapAddr, connectionType,
            port, sslCertificateId, phyNeId,commuProType, backup1Gateway, backup2Gateway, backup3Gateway,
            createAt, updateAt))

    def to_UpgradePara(self, datas):
        self.get_resid_from_idmapping(datas)
        col_names = "`" + ("`, `".join(self.dst_table_cols)) + "`"
        val_ids = ":" + (",:".join((str(x+1) for x in range(len(self.dst_table_cols)))))
        insert_stmt = "INSERT INTO `%s` (%s) VALUES(%s)" % (self.dst_table, col_names, val_ids)
        self.debug("insert sql stmt: %s" % insert_stmt)
        list_datas = []
        for data in datas:
            self.debug("original data is: %s, length is:%s" % (data, len(data)))
            data = self.convert_data(data)
            self.debug("coverted date is: %s, length is:%s" % (data, len(data)))
            if len(data) == len(self.dst_table_cols):
                list_datas.append(data)
            if len(list_datas) == ONE_PACKAGE:
                self.exec_sql(insert_stmt, list_datas)
                list_datas = []

        if len(list_datas) != 0:
            self.debug("data is:%s" % list_datas)
            self.exec_sql(insert_stmt, list_datas)

    def close_session(self):
        self.dst_db_cursor.close()
        self.dst_db_session.close()
        self.src_db_cursor.close()
        self.src_db_session.close()

    def do(self):
        try:
            self.info ('UpgradeQxDcnForRouter::do start!')
            select_stmt = 'select %s from %s where iGNeType=1' % (",".join(self.src_table_cols), self.src_table)
            self.debug("execute sql: %s" % select_stmt)
            try:
                self.src_db_cursor.execute(select_stmt)
            except BaseException as be:
                self.warning("execute sql failed, err is: %s" % str(be))
                return 0
            templates = self.src_db_cursor.fetchall()
            if templates:
                templates = list(templates)
            self.info('get stelnet para data success count:%d'%len(templates))

            self.to_UpgradePara(templates)
            self.info('to_UpgradeQxDcnForRouter done')
            self.close_session()
        except Exception as e:
            self.close_session()
            self.error('UpgradeQxDcnForRouter got exception')
            self.error(traceback.format_exc())
            return -1
        self.info('UpgradeQxDcnForRouter::do done')
        return 0
        
if __name__ == '__main__':
    tool = UpgradeQxDcnForRouter()
    print('[INFO] UpgradeQxDcnForRouter start>>>>>>>>>>>>>>>>>>>>>>>>>>>>>>>>>>>>>>>>>>>>>>>>>>')
    tool.do()
    print('[INFO] UpgradeQxDcnForRouter finished<<<<<<<<<<<<<<<<<<<<<<<<<<<<<<<<<<<<<<<<<<<<<<<')
