# -*- coding:utf-8 -*-
import json
import sys

from util import httpclient
from util import ossext

ENCODING_CODE = "utf-8"
DEFAULT_BACKUP_MODE = "SFTP"
DEFAULT_FTPUSER = "ftpuser"
DEFAULT_BACKUP_PATH = "backup"

CODE_200 = 200
CODE_201 = 201
CODE_202 = 202
SUCCESS = 0
ERROR = 1
FTPUSER_PWD = sys.stdin.readline().strip()
EXPAND_CURRENT_NODE_IP_JSON = sys.stdin.readline().strip()
EXPAND_JSON_FILE = sys.stdin.readline().strip()
SFTP_SERVER_IP = sys.stdin.readline().strip()


def get_all_ip_json():
    sys.stdout.write("Start to get all ip information\n")
    # get all ip of NCE
    url = "/rest/productconfiguration/v1/ips?action=query"
    para = {"query": []}
    client = httpclient.IRHttpClient()
    task_response = client.post(url, para)
    if len(task_response) < 2:
        message = "query all ip in NCE failed, query url {0} failed\n".format(url)
        sys.stdout.write(message)
        return False
    status = task_response[0]
    response = task_response[1]
    if status not in [CODE_200, CODE_201, CODE_202]:
        message = "query all ip in NCE failed, the url is {0}, " \
                  "the response is {1}\n".format(url, response)
        sys.stdout.write(message)
        return False

    json_result = json.loads(response)
    sys.stdout.write("Get all ip information successful\n")
    return json_result


def get_current_sftp_ip():
    # Obtaining the IP Address of the Newly Configured Backup Server
    sys.stdout.write("Start to get current ip information\n")
    sftp_ip = None
    with open(EXPAND_CURRENT_NODE_IP_JSON, mode="r", encoding=ENCODING_CODE) as current_node_file:
        current_node_ip_json = json.load(current_node_file)
    current_ipv4_list = list()
    current_ipv6_list = list()
    for eth in current_node_ip_json.get("iplist"):
        if eth.get("status").lower() == "down":
            continue
        for ipv4_list in eth.get("iplist"):
            current_ipv4_list.append(ipv4_list.get("ip"))

        for ipv6_list in eth.get("ipv6list"):
            current_ipv6_list.append(ipv6_list.get("ip"))

    sys.stdout.write("The local ipv4 list is {}, ipv6 list is {}\n".format(current_ipv4_list, current_ipv6_list))
    if not current_ipv4_list and not current_ipv6_list:
        return sftp_ip

    all_ip_json = get_all_ip_json()
    if not all_ip_json:
        return sftp_ip

    for entity in all_ip_json.get("entity"):
        node_ip = entity.get("nodeIP")
        if node_ip in current_ipv4_list or node_ip in current_ipv6_list:
            sftp_ip = node_ip
            sys.stdout.write(
                "Get the sftp server ip is {}\n".format(sftp_ip))
            with open(SFTP_SERVER_IP, mode="w", encoding=ENCODING_CODE) as sftp_ip_output:
                sftp_ip_output.write("ip={}=".format(sftp_ip))
            return sftp_ip
    return sftp_ip


def query_backup_server_info():
    # get backup server information
    url = "/rest/plat/brmgr/v1/main/baseinfo/bssinfo"
    client = httpclient.IRHttpClient()

    task_response = client.post(url)

    if len(task_response) < 2:
        message = "query backup server information failed, the url is {0}\n".format(url)
        sys.stdout.write(message)
        return False, None

    status = task_response[0]
    response = task_response[1]
    if status not in [CODE_200, CODE_201, CODE_202]:
        # backup server not exist
        message = "query backup server information failed, the url is {0}, " \
                  "the response is {1}\n".format(url, response)
        sys.stdout.write(message)
        return False, None

    return True, response


def generate_single_server_info(single_server_info_param, is_exist):
    sftp_server_ip_address = single_server_info_param.get("ftpServerIP")
    password = single_server_info_param.get("ftpPvalue")
    if is_exist:
        username = single_server_info_param.get("ftpUsername")
        backup_path = single_server_info_param.get("backupPath").strip("/")
    else:
        username = DEFAULT_FTPUSER
        backup_path = DEFAULT_BACKUP_PATH

    result = {
        "SFTPServerIPAddress": sftp_server_ip_address,
        "username": username,
        "password": password,
        "backuppath": backup_path
    }

    return result


def update_global_para_info(global_para_info_param, server_info_param, is_exist):
    single_server_info = generate_single_server_info(server_info_param, is_exist)
    sftp_list = global_para_info_param.get("SFTPList")
    sftp_list.append(single_server_info)
    global_para_info_param.update({"SFTPList": sftp_list})


def generate_global_para():
    global_para_info = {
        "backupmode": "SFTP",
        "NFS": {
            "nfsbackuppath": "/export/home/backup"
        },
        "SFTPList": []
    }

    # Add the original backup server information to the JSON file.
    status, response = query_backup_server_info()
    if status and response:
        exist_server_info_list = json.loads(response)
        for exist_server_info in exist_server_info_list:
            update_global_para_info(global_para_info_param=global_para_info, server_info_param=exist_server_info,
                                    is_exist=True)

    # Add the new backup server information to the JSON file.
    new_server_ip = get_current_sftp_ip()
    sys.stdout.write("Get the new sftp server ip : {}\n".format(new_server_ip))
    if not new_server_ip:
        sys.stdout.write("Get the new sftp server ip failed: {}\n".format(new_server_ip))
        return {}
    new_server_pwd = ossext.Cipher.encrypt(FTPUSER_PWD)
    new_server_info = {
        "ftpServerIP": new_server_ip,
        "ftpPvalue": new_server_pwd
    }
    update_global_para_info(global_para_info_param=global_para_info, server_info_param=new_server_info, is_exist=False)
    return global_para_info


if __name__ == '__main__':
    server_info = generate_global_para()
    if server_info:
        # Save the parameter information to a file.
        with open(EXPAND_JSON_FILE, mode="w", encoding=ENCODING_CODE) as f:
            json.dump(server_info, f)
        sys.exit(SUCCESS)
    sys.exit(ERROR)
