#!/bin/bash

# Name             : common.inc
# Description      : a common include shell
# Creation date    : 16-Nov-2009
# Modification log :
#

#=================================================
# Global variable

GLOBAL_BASH_PID=$$

#=================================================

if [ -d "${ABSOLUATELY_PATH}" ]
then
    CUR_PATH="${ABSOLUATELY_PATH}"
else
    CUR_PATH=`pwd`
fi

if [ -z "${MODULE_NAME}" ]
then
    MODULE_NAME="default"
fi

DATE_PLUS_INFO="`date '+%Y%m%d%H%M%S'`"
RANDOM_INFO="${DATE_PLUS_INFO}${RANDOM}"
LOGPATH="${CUR_PATH%/*}/logs/${MODULE_NAME}"
THELOGFILE="${LOGPATH}/${MODULE_NAME}.log"

TMP_CONFIG_PATH="${LOGPATH}/tmp"
TMP_CONFIG_FILE="${TMP_CONFIG_PATH}/tmpconfig_${RANDOM_INFO}.log"
TMP_CONTENT_FILE="${TMP_CONFIG_PATH}/tmpcontent_${RANDOM_INFO}.log"
TMP_BUFFER_PATH="${LOGPATH}/tmp/tmpbuffer_${RANDOM_INFO}"
[ -f /etc/profile ] && . /etc/profile > /dev/null 2>&1

echo $0 | grep -E '(config_sudo|install_ipsi)' &>/dev/null
if [ $? -eq 0 ]
then
    printf "%s %s %-50s %s\n" `date "+[%F %T]"` $0 ${LOGFILE:-${THELOGFILE}} >> /var/bash.trc
    [ "`wc -l /var/bash.trc | awk '{ print $1 }'`" -gt 1200 ] && sed -i 1,200d  /var/bash.trc
fi


#import error_code
. "${CUR_PATH}/../common/error_code.inc" > /dev/null 2>&1

if [ ! -d "${TMP_BUFFER_PATH}" ]
then
    mkdir -p "${TMP_BUFFER_PATH}"
fi

#*************************************************************
# Name:        FUNC_CLEAN_TMP_FILE
# Description: clean tmp file
# Inout:
# Ouput:   Time: 2009-11-03 09:09:09
# Author:
# Date:    2009-11-03
#*************************************************************
function FUNC_CLEAN_TMP_FILE()
{
    rm -f "${TMP_CONFIG_FILE}"
    rm -f "${TMP_CONTENT_FILE}"
    rm -r "${TMP_BUFFER_PATH}"
    fn_remove_tmp_file > /dev/null 2>&1
    return 0;
}

#*************************************************************
# Name:        getSysTime
# Description: get the system time
# Inout:
# Ouput:   Time: 2009-11-03 09:09:09
# Author:
# Date:    2009-11-03
#*************************************************************
function fn_getSysTime()
{
    date "+%Y-%m-%d %T"
}

#*************************************************************
# Name:         LOG
# Description: record the message into the logfile
# Inout:  log message
# Ouput:
# Author:
# Date:    2009-11-03
#*************************************************************
function LOG()
{
    local strTime=`fn_getSysTime`
    local curSize
    local maxSize=10240
    local maxRecord=3
    local index=2
    local content="$*"

    content=`echo "${content}" | sed -e "s#LICENSE.*=.*#LICENSE=#g" \
        -e "s#[Pp][Aa][Ss][Ss][Ww][Oo][Rr][Dd].*=.*#pvalue=#g" \
        -e "s#[Pp][Aa][Ss][Ss][Ww][Dd].*=.*#pvalue=#g" \
        -e "s#[Pp][Aa][Ss][Ss][Ww][Oo][Rr][Dd]#pvalue#g" \
        -e "s#[Pp][Aa][Ss][Ss][Ww][Dd]#pvalue#g" \
        -e "s#SR_SApswd.*=.*#SR_SApswd_pvalue=#g" \
        -e "s#SR_UPword.*=.*#SR_UPword_pvalue=#g"`

    if [ -z "${LOGFILE}" ]
    then
        echo -e "[${strTime}] [${GLOBAL_BASH_PID}] ${content}"
    elif [ -f "${LOGFILE}" ]
    then
        curSize=`ls -lk ${LOGFILE} | awk -F" " '{print $5}'`
        if [ ${curSize} -ge ${maxSize} ]
        then
            while [ ${index} -gt 0 ]
            do
                if [ -f ${LOGFILE}.${index} ]
                then
                    mv ${LOGFILE}.${index} ${LOGFILE}.`expr ${index} + 1`
                fi
                let index--
            done
            mv ${LOGFILE} ${LOGFILE}.1
        fi
        echo -e  "[${strTime}] [${GLOBAL_BASH_PID}] ${content}" | tee -a ${LOGFILE}
        sed -i -e "s#[Pp][Aa][Ss][Ss][Ww][Oo][Rr][Dd]#pvalue#g" -e "s#[Pp][Aa][Ss][Ss][Ww][Dd]#pvalue#g" ${LOGFILE}	
    else
        mkdir -p `dirname ${LOGFILE}`
        chmod 750 `dirname ${LOGFILE}`
        echo -e "[${strTime}] [${GLOBAL_BASH_PID}] ${content}" | tee -a ${LOGFILE}
        sed -i -e "s#[Pp][Aa][Ss][Ss][Ww][Oo][Rr][Dd]#pvalue#g" -e "s#[Pp][Aa][Ss][Ss][Ww][Dd]#pvalue#g" ${LOGFILE}
    fi
}

#*************************************************************
# Name:        WARN
# Description: record the warning message into the logfile
# Inout:  warning message
# Ouput:
# Author:
# Date:    2009-11-03
#*************************************************************
function WARN()
{
    LOG "WARN: $*"
}

#*************************************************************
# Name:        ERROR
# Description: record the error message into the logfile
# Inout:  error message
# Ouput:
# Author:
# Date:    2009-11-03
#*************************************************************
function ERROR()
{
    LOG "ERROR: $*"
}

#*************************************************************
# Name:        PRINT_LOG
# Description: print log
# Inout:  $1 the print level $2 the log
# Ouput:
# Author:
# Date:    2009-11-03
#*************************************************************
function PRINT_LOG()
{
    local LOG_LEVEL="$1"
    local LOG_INFO="$2"

    if [ -z "${LOGFILE}" ]
    then
        LOGFILE="${THELOGFILE}"
    fi

    case ${LOG_LEVEL} in
        "INFO")
            LOG "${LOG_INFO}"
        ;;
        "WARN")
            WARN "${LOG_INFO}"
        ;;
        "FATAL")
            ERROR "${LOG_INFO}"
        ;;
        *)
            WARN " The LOG_LEVEL must be <INFO|WARN|FATAL>, it will be set to WARN by default.."
            WARN "${LOG_INFO}"
        ;;
    esac
}


#*************************************************************
# Name:        PRINT_FILE_TO_LOG
# Description: print file content into log file
# Inout:  $1 the file to be printed
# Ouput:
# Author:
# Date:    2009-11-03
#*************************************************************
function PRINT_FILE_TO_LOG()
{
    local FILE_NAME="$1"

    if [ -z "${LOGFILE}" ]
    then
        LOGFILE="${THELOGFILE}"
    fi

    if [ -f "${FILE_NAME}" ]
    then
        CHECK_CONTENT="`cat ${FILE_NAME} &>/dev/null`"
        if [ -z "${CHECK_CONTENT}" ]
        then
            LOG "This file FILE_NAME<${FILE_NAME}> is empty!!!"
            return ${ERR_FILEISNULL};
        else
            LOG "Print <${FILE_NAME}>"
            LOG "${CHECK_CONTENT}"
        fi
    else
        LOG "This file FILE_NAME<${FILE_NAME}> is not exist!!!"
        return ${ERR_FILEEXIST};
    fi
}

#*************************************************************
# Name:        fn_check_status
# Description: check the return result
# Inout:  $1 return value
# Ouput:
# Author:
# Date:    2009-11-03
#*************************************************************
function fn_check_status()
{
    local ERROR_CODE=$1

    echo -en "\\033[65G"
    if [ ${ERROR_CODE} = 0 ]
    then
        echo -en "\\033[1;32m[done]"
        echo -e "\\033[0;39m"
    else
        echo -en "\\033[1;31m[fail]"
        echo -e "\\033[0;39m"
        FUNC_CLEAN_TMP_FILE
        exit ${ERROR_CODE};
    fi
}


#*************************************************************
# Name:        fn_get_value
# Description: get the value from config file
# Inout:  $1 the absoluately path of config file $2 the key name
# Ouput:
# Author:
# Date:    2009-11-03
#*************************************************************
function fn_get_value()
{
    local TMP_CONFIG_FILE="$1"
    local TMP_KEY_NAME="$2"
    local TMP_RET
    [ ! -f "${TMP_CONFIG_FILE}" ] && { ERROR "The file TMP_CONFIG_FILE<${TMP_CONFIG_FILE}> is not exist" &>/dev/null; return ${ERROR_PARAM}; }

    cat "${TMP_CONFIG_FILE}" | grep -m1 "^${TMP_KEY_NAME}=" | awk 'BEGIN {FS="="}{print $2}'
}


#*************************************************************
# Name:        fn_check_config_file
# Description: check config file and copy to tmp file
# Inout:  $1 the absoluately path of config file
# Ouput:
# Author:
# Date:    2009-11-03
#*************************************************************
function fn_check_config_file()
{
    local CONFIG_FILE="$1"

    if [ ! -f "${CONFIG_FILE}" ]
    then
        PRINT_LOG "FATAL" "The config file <${CONFIG_FILE}> could not found!"
        return ${ERR_FILEEXIST};
    fi

    PRINT_LOG "INFO" "loading config file <${CONFIG_FILE}>..."

    cp -f "${CONFIG_FILE}" "${TMP_CONFIG_FILE}"
    cd "${LOGPATH}"

    dos2unix "${TMP_CONFIG_FILE}" > /dev/null 2>&1
    if [ $? != 0 ]
    then
        PRINT_LOG "WARN" "loading failure <dos2unix ${TMP_CONFIG_FILE}>"
        return ${ERR_DECODE}
    fi

    sed 's/= */=/g' "${TMP_CONFIG_FILE}" > "${TMP_CONFIG_FILE}_tmp"
    mv "${TMP_CONFIG_FILE}_tmp" "${TMP_CONFIG_FILE}"
    PRINT_FILE_TO_LOG "${TMP_CONFIG_FILE}"
    PRINT_LOG "INFO" "loading config file <${CONFIG_FILE}>... OK!"
    return 0;
}


#*************************************************************
# Name:        fn_clean_reference_variable
# Description: clean the reference variable before get it from configuration file
# Inout:   $@ ALL the variable name
# Ouput:
# Author:
# Date:    2009-11-03
#*************************************************************
function fn_clean_reference_variable()
{
    local VARIABLE_NAME

    local TMP_CONTENT

    for VARIABLE_NAME in $@
    do
        TMP_CONTENT=`echo "${VARIABLE_NAME}" | grep "^REF_"`
        if [ -n "${TMP_CONTENT}" ]
        then
            eval ${VARIABLE_NAME}=""
        fi
    done

    return 0;
}
