#!/usr/bin/env bash

current_dir=$(dirname $0)
log=${current_dir}/rollback.log
version=$1
upgrade_path=$2

. "${current_dir}/get_os_info.sh"
. "${current_dir}/upgrade_rollback_func.sh"

############################################################
# Name:        rollback_sudo
# Description: 回滚sudoconfig
#
############################################################
function rollback_sudo() {
    print_log "INFO" "start to rollback sudoconfig."
    # 判断是否需要回滚sudoconfig
    if [ ! -f "${current_dir}/backup_sudoconfig.flag" ];then
        print_log "WARN" "no need to rollback sudoconfig"
        return
    fi
    bash ${CBBPath}/rebuild/sudoconfig_rollback.sh -oldsudobin "${sudobin}" -newsudobin "${sudobin}" -version "${version}"
    [ $? -ne 0 ] && print_log "ERROR" "rollback sudoconfig failed, please read log in ${current_dir}/../sudoconfig/rebuild/logs/" && return 1
    rm -f "${current_dir}/backup_sudoconfig.flag"
    print_log "INFO" "sucess to rollback sudoconfig."
    return 0
}


############################################################
# Name:        rollback_osconfig
# Description: 回滚osconfig
#
############################################################
function rollback_osconfig() {
    print_log "INFO" "start to rollback osconfig."
    # 判断是否需要回滚osconfig
    if [ ! -f "${current_dir}/backup_osconfig.flag" ];then
        print_log "WARN" "no need to rollback osconfig."
        return
    fi
    bash "${OSConfigPath}"/osconfig_rollbackup.sh -backuppath ${upgrade_path}/../bak_${version} -oldinstallpath /usr/local/osconfig -newinstallpath /usr/local/osconfig -oldsudobin "${sudobin}" -version ${version}
    [ $? -ne 0 ] && print_log "ERROR" "rollback osconfig fail, please read log in ${current_dir}/../osconfig/logs/" && return 1
    rm -f "${current_dir}/backup_osconfig.flag"
    print_log "INFO" "success to rollback osconfig."
    return 0
}


############################################################
# Name:        rollback_common_osconfig
# Description: 回滚common_osconfig
#
############################################################
function rollback_common_osconfig() {
    print_log "INFO" "start to rollback common osconfig."
    # 判断是否需要回滚操作系统配置
    if [ ! -f "${current_dir}/nce_config.flag" ];then
        print_log "WARN" "no need to rollback common osconfig."
        return
    fi
    bash /usr/local/osconfig/os/bin/commonosconfig.sh rollback "${version}"
    [ $? -ne 0 ] && print_log "ERROR" "upgrade common osconfig fail." && return 1
    rm -f "${current_dir}/nce_config.flag"
    print_log "INFO" "success to rollback common osconfig."
    return 0
}

############################################################
# Name:        pre_rollback
# Description: 回滚前置操作
#
############################################################
function pre_rollback() {
    print_log "INFO" "start to execute operation before rollback."
    # 恢复升级删除的配置
    sudo "${sudobin}"/execute.sh "${current_dir}/sudobin-upgrade_pkg.tar" "/opt/rollback_config_$(date '+%Y%m%d%H%M%S')" pre_rollback.sh
    print_log "INFO" "finish to execute operation before rollback."
    return 0
}

############################################################
# Name:        post_rollback
# Description: 回滚后置操作
#
############################################################
function post_rollback() {
    print_log "INFO" "start to execute operation after rollback."
    # 因平台回滚导致的签名工具acl权限丢失，要在回滚后重新刷新，避免回滚后验签失败
    sudo "${sudobin}"/execute.sh "${current_dir}/sudobin-upgrade_pkg.tar" "/opt/rollback_config_$(date '+%Y%m%d%H%M%S')" post_rollback.sh
    print_log "INFO" "finish to execute operation after rollback."
    return 0
}

############################################################
# Name:        main
# Description: 主函数，入口
#
############################################################
function main() {
    # 修改文件权限
    chmod 640 "${current_dir}"/*
    chmod 550 "${current_dir}"/*.sh

    # 解压osconfig和sudoconfig包到/home目录
    unzip_os_mediation

    # 回滚OS配置
    pre_rollback || return 1
    rollback_common_osconfig || return 1

    # 回滚osconfig
    rollback_osconfig || return 1

    # 回滚sudoconfig
    rollback_sudo || return 1

    # 回滚后置操作
    post_rollback || return 1

    print_log "INFO" "success to rollback."
    return 0
}

print_log "INFO" "start to rollback, version: ${version}" > ${log} 2>&1

main >> ${log} 2>&1
result=$?
if [ "${result}" -eq 0 ];then
    touch "${current_dir}"/rollback_succeed_flag
else
    touch "${current_dir}"/rollback_fail_flag
fi
exit "${result}"