#!/usr/bin/env bash

current_dir=$(dirname $0)
log=${current_dir}/upgrade.log
version=$1
upgrade_path=$2

. "${current_dir}/get_os_info.sh"
. "${current_dir}/upgrade_rollback_func.sh"


############################################################
# Name:        backup_sudo
# Description: 备份sudoconfig，升级前需备份
#
############################################################
function backup_sudo() {
    print_log "INFO" "start to backup sudoconfig."
    bash ${CBBPath}/rebuild/sudoconfig_backup.sh -oldsudobin "${sudobin}" -newsudobin "${sudobin}" -version ${version}
    [ $? -ne 0 ] && print_log "ERROR" "backup sudoconfig failed, please read log in ${current_dir}/../sudoconfig/rebuild/logs/" && return 1
    # 是否可以回滚sudoconfig的标志
    touch "${current_dir}"/backup_sudoconfig.flag
    print_log "INFO" "sucess to backup sudoconfig."
    return 0
}


############################################################
# Name:        upgrade_sudo
# Description: 升级sudoconfig
#
############################################################
function upgrade_sudo() {
    print_log "INFO" "start to upgrade sudoconfig."
    bash ${CBBPath}/rebuild/sudoconfig_upgrade.sh -oldsudobin "${sudobin}" -newsudobin "${sudobin}" -version ${version}
    [ $? -ne 0 ] && print_log "ERROR" "upgrade sudoconfig failed, please read log in ${current_dir}/../sudoconfig/rebuild/logs/" && return 1
    print_log "INFO" "sucess to upgrade sudoconfig."
    return 0
}


############################################################
# Name:        backup_osconfig
# Description: 备份osconfig，升级前备份
#
############################################################
function backup_osconfig() {
    print_log "INFO" "start to backup osconfig."
    mkdir -p "${upgrade_path}/../bak_${version}"
    chown ossadm:ossgroup "${upgrade_path}/../bak_${version}"
    [ $? -ne 0 ] && print_log "ERROR" "create osconfig backup path fail." && return 1
    bash ${OSConfigPath}/osconfig_backup.sh -backuppath ${upgrade_path}/../bak_${version} -oldinstallpath /usr/local/osconfig -newinstallpath /usr/local/osconfig -version ${version}
    [ $? -ne 0 ] && print_log "ERROR" "backup osconfig fail, please read log in ${current_dir}/../osconfig/logs/" && return 1
    # 是否可以回滚osconfig的标志
    touch "${current_dir}"/backup_osconfig.flag
    print_log "INFO" "success to backup osconfig."
    return 0
}


############################################################
# Name:        upgrade_osconfig
# Description: 升级osconfig
#
############################################################
function upgrade_osconfig() {
    print_log "INFO" "start to upgrade osconfig."
    bash ${OSConfigPath}/osconfig_upgrade.sh -oldinstallpath /usr/local/osconfig -newinstallpath /usr/local/osconfig -newsudobin "${sudobin}"
    [ $? -ne 0 ] && print_log "ERROR" "upgrade osconfig fail, please read log in ${current_dir}/../osconfig/logs/" && return 1
    print_log "INFO" "success to upgrade osconfig"
    return 0
}


############################################################
# Name:        upgrade_common_osconfig
# Description: 升级common_osconfig
#
############################################################
function upgrade_common_osconfig() {
    print_log "INFO" "start to upgrade common osconfig."
    # 是否可以执行系统配置回滚的标志
    touch "${current_dir}"/nce_config.flag
    bash /usr/local/osconfig/os/bin/commonosconfig.sh upgrade "${version}"
    [ $? -ne 0 ] && print_log "ERROR" "upgrade common osconfig fail." && return 1
    print_log "INFO" "success to upgrade common osconfig."
    return 0
}

############################################################
# Name:        pre_upgrade
# Description: 升级前置操作
#
############################################################
function pre_upgrade() {
    print_log "INFO" "start to execute operation before upgrade."
    # 解锁华为suse lock服务锁住的目录，其他os不涉及
    sudo "${sudobin}"/execute.sh "${current_dir}/sudobin-upgrade_pkg.tar" "/opt/upgrade_config_$(date '+%Y%m%d%H%M%S')" pre_upgrade.sh
    print_log "INFO" "finish to execute operation before upgrade."
    return 0
}


############################################################
# Name:        main
# Description: 主函数，入口
#
############################################################
function main() {
    # 修改文件权限
    chmod 640 "${current_dir}"/*
    chmod 550 "${current_dir}"/*.sh

    # 升级前置操作
    pre_upgrade || return 1

    # 解压osconfig和sudoconfig包到升级目录
    unzip_os_mediation

    # 备份sudoconfig
    backup_sudo || return 1

    # 升级sudoconfig
    upgrade_sudo || return 1

    # 备份osconfig
    backup_osconfig || return 1

    # 升级osconfig
    upgrade_osconfig || return 1

    # 升级OS配置
    upgrade_common_osconfig || return 1

    print_log "INFO" "success to upgrade."
    return 0
}

print_log "INFO" "start to upgrade, version: ${version}" > ${log} 2>&1

main >> ${log} 2>&1
result=$?
if [ "${result}" -eq 0 ];then
    touch "${current_dir}"/upgrade_succeed_flag
else
    touch "${current_dir}"/upgrade_fail_flag
fi
exit "${result}"






